/*
 * Decompiled with CFR 0.152.
 */
package io.journalkeeper.utils.threads;

import io.journalkeeper.utils.threads.VirtualThread;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VirtualThreadExecutor {
    private static final Logger logger = LoggerFactory.getLogger(VirtualThreadExecutor.class);
    private final long keepAliveTimeMs;
    private final long maxIntervalMs;
    private final int steps;
    private final long maxUseTime;
    private final DelayQueue<DelayCommand> commandQueue = new DelayQueue();
    private final List<Thread> workThreads;
    private final Set<VirtualThread> toBeRemoved = ConcurrentHashMap.newKeySet();
    private final Set<VirtualThread> virtualThreads = ConcurrentHashMap.newKeySet();

    public VirtualThreadExecutor(long keepAliveTimeMs, long maxIntervalMs, int steps, long maxUseTime, int threadCount) {
        this.keepAliveTimeMs = keepAliveTimeMs;
        this.maxIntervalMs = maxIntervalMs;
        this.steps = steps;
        this.maxUseTime = maxUseTime;
        this.workThreads = IntStream.range(0, threadCount).mapToObj(index -> {
            Thread thread = new Thread(new WorkThread());
            thread.setName("VirtualThreadExecutor-" + index);
            thread.start();
            return thread;
        }).collect(Collectors.toList());
    }

    public void start(VirtualThread vt, String name) {
        this.virtualThreads.add(vt);
        this.commandQueue.add(new DelayCommand(vt, name));
    }

    public void start(VirtualThread vt, long minDelayMs, String name) {
        this.virtualThreads.add(vt);
        this.commandQueue.add(new DelayCommand(vt, minDelayMs, name));
    }

    public void stop(VirtualThread vt) throws InterruptedException {
        if (this.virtualThreads.remove(vt)) {
            this.toBeRemoved.add(vt);
            if (this.commandQueue.removeIf(cmd -> ((DelayCommand)cmd).virtualThread == vt)) {
                this.toBeRemoved.remove(vt);
            } else {
                while (this.toBeRemoved.contains(vt)) {
                    Thread.sleep(50L);
                }
            }
        }
    }

    private void wait(Thread thread) {
        logger.info("Stopping thread {}...", (Object)thread.getName());
        long t0 = System.currentTimeMillis();
        long timeout = 1000L;
        while (System.currentTimeMillis() - t0 < timeout && thread.isAlive()) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    public void stop() {
        this.workThreads.forEach(Thread::interrupt);
        this.workThreads.forEach(this::wait);
    }

    private class WorkThread
    implements Runnable {
        private WorkThread() {
        }

        @Override
        public void run() {
            DelayCommand cmd = null;
            boolean dryRun = true;
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    cmd = (DelayCommand)VirtualThreadExecutor.this.commandQueue.take();
                    if (VirtualThreadExecutor.this.toBeRemoved.remove(cmd.virtualThread)) continue;
                    long start = System.currentTimeMillis();
                    dryRun = true;
                    while (VirtualThreadExecutor.this.maxUseTime + start > System.currentTimeMillis() && cmd.virtualThread.run()) {
                        if (dryRun) {
                            dryRun = false;
                        }
                        Thread.yield();
                    }
                }
                catch (InterruptedException e) {
                    logger.warn("Virtual thread interrupted!");
                    break;
                }
                catch (Throwable e) {
                    logger.warn("Exception on {} :", (Object)(cmd == null ? "" : cmd.name), (Object)e);
                }
                if (null == cmd) continue;
                long now = System.currentTimeMillis();
                if (dryRun) {
                    if (VirtualThreadExecutor.this.keepAliveTimeMs + cmd.lastRunTime <= now && cmd.delay < VirtualThreadExecutor.this.maxIntervalMs) {
                        DelayCommand delayCommand = cmd;
                        delayCommand.delay = delayCommand.delay + VirtualThreadExecutor.this.maxIntervalMs / (long)VirtualThreadExecutor.this.steps;
                    }
                } else {
                    cmd.delay = cmd.minDelayMs;
                    cmd.lastRunTime = now;
                }
                cmd.startTime = now + cmd.delay;
                VirtualThreadExecutor.this.commandQueue.put(cmd);
            }
        }
    }

    private static class DelayCommand
    implements Delayed {
        private final VirtualThread virtualThread;
        private final long minDelayMs;
        private final String name;
        private volatile long startTime = System.currentTimeMillis();
        private volatile long lastRunTime = System.currentTimeMillis();
        private volatile long delay = 0L;

        private DelayCommand(VirtualThread virtualThread, String name) {
            this(virtualThread, 0L, name);
        }

        private DelayCommand(VirtualThread virtualThread, long minDelayMs, String name) {
            this.virtualThread = virtualThread;
            this.minDelayMs = minDelayMs;
            this.name = name;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long diff = this.startTime - System.currentTimeMillis();
            return unit.convert(diff, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed another) {
            if (this.startTime < ((DelayCommand)another).startTime) {
                return -1;
            }
            if (this.startTime > ((DelayCommand)another).startTime) {
                return 1;
            }
            return 0;
        }
    }
}

