/*
 * Decompiled with CFR 0.152.
 */
package io.journalkeeper.persistence.local.cache;

import io.journalkeeper.persistence.local.cache.BufferHolder;
import io.journalkeeper.persistence.local.cache.MemoryCacheManager;
import io.journalkeeper.persistence.local.cache.PreloadCacheMetric;
import io.journalkeeper.utils.format.Format;
import io.journalkeeper.utils.spi.Singleton;
import io.journalkeeper.utils.threads.AsyncLoopThread;
import io.journalkeeper.utils.threads.ThreadBuilder;
import io.journalkeeper.utils.threads.Threads;
import io.journalkeeper.utils.threads.ThreadsFactory;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.Cleaner;
import sun.misc.VM;
import sun.nio.ch.DirectBuffer;

@Singleton
public class PreloadBufferPool
implements MemoryCacheManager {
    private static final Logger logger = LoggerFactory.getLogger(PreloadBufferPool.class);
    private static final String PRELOAD_THREAD = "PreloadBuffer-PreloadThread";
    private static final String EVICT_THREAD = "PreloadBuffer-EvictThread";
    private static final double DEFAULT_CACHE_RATIO = 0.9;
    private static final float DEFAULT_EVICT_RATIO = 0.9f;
    private static final float DEFAULT_CORE_RATIO = 0.8f;
    private static final long INTERVAL_MS = 50L;
    private static final String MAX_MEMORY_KEY = "memory_cache.max_memory";
    private static final String EVICT_RATIO_KEY = "memory_cache.evict_ratio";
    private static final String CORE_RATIO_KEY = "memory_cache.core_ratio";
    private static PreloadBufferPool instance = null;
    private final Threads threads = ThreadsFactory.create();
    private final long maxMemorySize;
    private final long coreMemorySize;
    private final long evictMemorySize;
    private final AtomicLong usedSize = new AtomicLong(0L);
    private final Set<BufferHolder> directBufferHolders = ConcurrentHashMap.newKeySet();
    private final Set<BufferHolder> mMapBufferHolders = ConcurrentHashMap.newKeySet();
    private Map<Integer, PreLoadCache> bufferCache = new ConcurrentHashMap<Integer, PreLoadCache>();

    public PreloadBufferPool() {
        this.maxMemorySize = Format.parseSize((String)System.getProperty(MAX_MEMORY_KEY), (long)Math.round((double)VM.maxDirectMemory() * 0.9));
        float evictRatio = PreloadBufferPool.getFloatProperty(EVICT_RATIO_KEY, 0.9f);
        this.evictMemorySize = Math.round((float)this.maxMemorySize * evictRatio);
        float coreRatio = PreloadBufferPool.getFloatProperty(CORE_RATIO_KEY, 0.8f);
        this.coreMemorySize = Math.round((float)this.maxMemorySize * coreRatio);
        this.threads.createThread(this.buildPreloadThread());
        this.threads.createThread(this.buildEvictThread());
        this.threads.start();
        logger.info("Max direct memory: {}, core direct memory: {}, evict direct memory: {}.", new Object[]{Format.formatSize((long)this.maxMemorySize), Format.formatSize((long)this.coreMemorySize), Format.formatSize((long)this.evictMemorySize)});
    }

    private static float getFloatProperty(String key, float defaultValue) {
        try {
            return Float.parseFloat(System.getProperty(key, String.valueOf(defaultValue)));
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    private AsyncLoopThread buildPreloadThread() {
        return ThreadBuilder.builder().name(PRELOAD_THREAD).sleepTime(50L, 50L).doWork(this::preLoadBuffer).onException(e -> logger.warn("{} exception:", (Object)PRELOAD_THREAD, (Object)e)).daemon(true).build();
    }

    private AsyncLoopThread buildEvictThread() {
        return ThreadBuilder.builder().name(EVICT_THREAD).sleepTime(50L, 50L).condition(() -> this.usedSize.get() > this.evictMemorySize).doWork(this::evict).onException(e -> logger.warn("{} exception:", (Object)EVICT_THREAD, (Object)e)).daemon(true).build();
    }

    private synchronized void evict() {
        for (PreLoadCache preLoadCache : this.bufferCache.values()) {
            if (!this.needEviction()) break;
            while (preLoadCache.cache.size() > preLoadCache.maxCount && !this.needEviction()) {
                try {
                    this.destroyOne((ByteBuffer)preLoadCache.cache.remove());
                }
                catch (NoSuchElementException noSuchElementException) {}
            }
        }
        if (this.needEviction()) {
            List sorted = Stream.concat(this.directBufferHolders.stream(), this.mMapBufferHolders.stream()).filter(BufferHolder::isFree).map(bufferHolder -> new LruWrapper<BufferHolder>((BufferHolder)bufferHolder, bufferHolder.lastAccessTime())).sorted(Comparator.comparing(rec$ -> ((LruWrapper)rec$).getLastAccessTime())).collect(Collectors.toList());
            while (this.needEviction() && !sorted.isEmpty()) {
                LruWrapper wrapper = (LruWrapper)sorted.remove(0);
                BufferHolder holder = (BufferHolder)wrapper.get();
                if (holder.lastAccessTime() != wrapper.getLastAccessTime()) continue;
                holder.evict();
            }
        }
    }

    @Override
    public void printMetric() {
        long totalUsed = this.usedSize.get();
        long plUsed = this.bufferCache.values().stream().mapToLong(preLoadCache -> {
            long cached = ((PreLoadCache)preLoadCache).cache.size();
            long usedPreLoad = ((PreLoadCache)preLoadCache).onFlyCounter.get();
            long totalSize = (long)((PreLoadCache)preLoadCache).bufferSize * (cached + usedPreLoad);
            logger.info("PreloadCache usage: cached: {} * {} = {}, used: {} * {} = {}, total: {}", new Object[]{Format.formatSize((long)((PreLoadCache)preLoadCache).bufferSize), cached, Format.formatSize((long)((long)((PreLoadCache)preLoadCache).bufferSize * cached)), Format.formatSize((long)((PreLoadCache)preLoadCache).bufferSize), usedPreLoad, Format.formatSize((long)((long)((PreLoadCache)preLoadCache).bufferSize * usedPreLoad)), Format.formatSize((long)totalSize)});
            return totalSize;
        }).sum();
        long mmpUsed = this.mMapBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        long directUsed = this.directBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        logger.info("Direct memory usage: preload/direct/mmp/used/max: {}/{}/{}/{}/{}.", new Object[]{Format.formatSize((long)plUsed), Format.formatSize((long)directUsed), Format.formatSize((long)mmpUsed), Format.formatSize((long)totalUsed), Format.formatSize((long)this.maxMemorySize)});
    }

    private boolean needEviction() {
        return this.usedSize.get() > this.evictMemorySize;
    }

    private boolean isOutOfMemory() {
        return this.usedSize.get() > this.maxMemorySize;
    }

    private boolean isHungry() {
        return this.usedSize.get() < this.coreMemorySize;
    }

    @Override
    public synchronized void addPreLoad(int bufferSize, int coreCount, int maxCount) {
        PreLoadCache preLoadCache = this.bufferCache.putIfAbsent(bufferSize, new PreLoadCache(bufferSize, coreCount, maxCount));
        if (null != preLoadCache) {
            preLoadCache.referenceCount.incrementAndGet();
        }
    }

    @Override
    public synchronized void removePreLoad(int bufferSize) {
        PreLoadCache preLoadCache = this.bufferCache.get(bufferSize);
        if (null != preLoadCache && preLoadCache.referenceCount.decrementAndGet() <= 0) {
            this.bufferCache.remove(bufferSize);
            preLoadCache.cache.forEach(this::destroyOne);
        }
    }

    private void destroyOne(ByteBuffer byteBuffer) {
        this.usedSize.getAndAdd(-1 * byteBuffer.capacity());
        this.releaseIfDirect(byteBuffer);
    }

    private void preLoadBuffer() {
        for (PreLoadCache preLoadCache : this.bufferCache.values()) {
            if (preLoadCache.cache.size() >= preLoadCache.coreCount) continue;
            if (this.isHungry()) {
                try {
                    while (preLoadCache.cache.size() < preLoadCache.coreCount && this.usedSize.get() + (long)preLoadCache.bufferSize < this.maxMemorySize) {
                        preLoadCache.cache.add(this.createOne(preLoadCache.bufferSize));
                    }
                    continue;
                }
                catch (OutOfMemoryError ignored) {
                    return;
                }
            }
            List outdated = this.directBufferHolders.stream().filter(b -> b.size() == preLoadCache.bufferSize).filter(BufferHolder::isFree).map(bufferHolder -> new LruWrapper<BufferHolder>((BufferHolder)bufferHolder, bufferHolder.lastAccessTime())).sorted(Comparator.comparing(rec$ -> ((LruWrapper)rec$).getLastAccessTime())).collect(Collectors.toList());
            while (preLoadCache.cache.size() < preLoadCache.coreCount && !outdated.isEmpty()) {
                LruWrapper wrapper = (LruWrapper)outdated.remove(0);
                BufferHolder holder = (BufferHolder)wrapper.get();
                if (holder.lastAccessTime() != wrapper.getLastAccessTime()) continue;
                holder.evict();
            }
        }
    }

    private ByteBuffer createOne(int size) {
        this.reserveMemory(size);
        return ByteBuffer.allocateDirect(size);
    }

    private void reserveMemory(int size) {
        this.usedSize.addAndGet(size);
        try {
            PreLoadCache preLoadCache;
            while (this.isOutOfMemory() && null != (preLoadCache = (PreLoadCache)this.bufferCache.values().stream().filter(p -> ((PreLoadCache)p).cache.size() > 0).findAny().orElse(null))) {
                this.destroyOne((ByteBuffer)preLoadCache.cache.remove());
            }
            if (this.isOutOfMemory()) {
                this.threads.wakeupThread(EVICT_THREAD);
                for (int i = 0; i < 5 && this.isOutOfMemory(); ++i) {
                    try {
                        Thread.sleep(10L);
                        continue;
                    }
                    catch (InterruptedException e) {
                        logger.warn("Interrupted: ", (Throwable)e);
                    }
                }
                if (this.isOutOfMemory()) {
                    throw new OutOfMemoryError();
                }
            }
        }
        catch (Throwable t) {
            this.usedSize.getAndAdd(-1 * size);
        }
    }

    private void releaseIfDirect(ByteBuffer byteBuffer) {
        if (byteBuffer instanceof DirectBuffer) {
            try {
                Method getCleanerMethod = byteBuffer.getClass().getMethod("cleaner", new Class[0]);
                getCleanerMethod.setAccessible(true);
                Cleaner cleaner = (Cleaner)getCleanerMethod.invoke((Object)byteBuffer, new Object[0]);
                cleaner.clean();
            }
            catch (Exception e) {
                logger.warn("Exception: ", (Throwable)e);
            }
        }
    }

    @Override
    public void allocateMMap(BufferHolder bufferHolder) {
        this.reserveMemory(bufferHolder.size());
        this.mMapBufferHolders.add(bufferHolder);
    }

    @Override
    public ByteBuffer allocateDirect(int bufferSize, BufferHolder bufferHolder) {
        ByteBuffer buffer = this.allocateDirect(bufferSize);
        this.directBufferHolders.add(bufferHolder);
        return buffer;
    }

    private ByteBuffer allocateDirect(int bufferSize) {
        try {
            PreLoadCache preLoadCache = this.bufferCache.get(bufferSize);
            if (null != preLoadCache) {
                try {
                    ByteBuffer byteBuffer = (ByteBuffer)preLoadCache.cache.remove();
                    preLoadCache.onFlyCounter.getAndIncrement();
                    return byteBuffer;
                }
                catch (NoSuchElementException e) {
                    logger.warn("Pool is empty, create ByteBuffer: {}", (Object)Format.formatSize((long)bufferSize));
                    ByteBuffer byteBuffer = this.createOne(bufferSize);
                    preLoadCache.onFlyCounter.getAndIncrement();
                    this.threads.wakeupThread(PRELOAD_THREAD);
                    return byteBuffer;
                }
            }
            logger.warn("No cached buffer in pool, create ByteBuffer: {}", (Object)Format.formatSize((long)bufferSize));
            return this.createOne(bufferSize);
        }
        catch (OutOfMemoryError outOfMemoryError) {
            logger.debug("OOM: {}/{}.", (Object)Format.formatSize((long)this.usedSize.get()), (Object)Format.formatSize((long)this.maxMemorySize));
            throw outOfMemoryError;
        }
    }

    @Override
    public void releaseDirect(ByteBuffer byteBuffer, BufferHolder bufferHolder) {
        this.directBufferHolders.remove(bufferHolder);
        int size = byteBuffer.capacity();
        PreLoadCache preLoadCache = this.bufferCache.get(size);
        if (null != preLoadCache) {
            if (this.needEviction() && preLoadCache.cache.size() >= preLoadCache.maxCount) {
                this.destroyOne(byteBuffer);
            } else {
                byteBuffer.clear();
                preLoadCache.cache.add(byteBuffer);
            }
            preLoadCache.onFlyCounter.getAndDecrement();
        } else {
            this.destroyOne(byteBuffer);
        }
    }

    @Override
    public void releaseMMap(BufferHolder bufferHolder) {
        this.mMapBufferHolders.remove(bufferHolder);
        this.usedSize.getAndAdd(-1 * bufferHolder.size());
    }

    @Override
    public Collection<PreloadCacheMetric> getCaches() {
        return new ArrayList<PreloadCacheMetric>(this.bufferCache.values());
    }

    @Override
    public long getMaxMemorySize() {
        return this.maxMemorySize;
    }

    @Override
    public long getTotalUsedMemorySize() {
        return this.usedSize.get();
    }

    @Override
    public long getDirectUsedMemorySize() {
        return this.directBufferHolders.stream().mapToLong(BufferHolder::size).sum() + this.bufferCache.values().stream().mapToLong(c -> c.getBufferSize() * c.getCachedCount()).sum();
    }

    @Override
    public void close() {
        if (null != instance) {
            PreloadBufferPool.instance.threads.stop();
            PreloadBufferPool.instance.bufferCache.values().forEach(p -> {
                while (!((PreLoadCache)p).cache.isEmpty()) {
                    instance.destroyOne((ByteBuffer)((PreLoadCache)p).cache.remove());
                }
            });
            PreloadBufferPool.instance.directBufferHolders.parallelStream().forEach(BufferHolder::evict);
            PreloadBufferPool.instance.mMapBufferHolders.parallelStream().forEach(BufferHolder::evict);
            PreloadBufferPool.instance.bufferCache.values().forEach(p -> {
                while (!((PreLoadCache)p).cache.isEmpty()) {
                    instance.destroyOne((ByteBuffer)((PreLoadCache)p).cache.remove());
                }
            });
        }
        logger.info("Preload buffer pool closed.");
    }

    @Override
    public long getMapUsedMemorySize() {
        return this.mMapBufferHolders.stream().mapToLong(BufferHolder::size).sum();
    }

    private static class LruWrapper<V> {
        private final long lastAccessTime;
        private final V t;

        LruWrapper(V t, long lastAccessTime) {
            this.lastAccessTime = lastAccessTime;
            this.t = t;
        }

        private long getLastAccessTime() {
            return this.lastAccessTime;
        }

        private V get() {
            return this.t;
        }
    }

    public static class PreLoadCache
    implements PreloadCacheMetric {
        private final int bufferSize;
        private final int coreCount;
        private final int maxCount;
        private final Queue<ByteBuffer> cache = new ConcurrentLinkedQueue<ByteBuffer>();
        private final AtomicInteger onFlyCounter = new AtomicInteger(0);
        private final AtomicInteger referenceCount;

        PreLoadCache(int bufferSize, int coreCount, int maxCount) {
            this.bufferSize = bufferSize;
            this.coreCount = coreCount;
            this.maxCount = maxCount;
            this.referenceCount = new AtomicInteger(1);
        }

        @Override
        public int getBufferSize() {
            return this.bufferSize;
        }

        @Override
        public int getCoreCount() {
            return this.coreCount;
        }

        @Override
        public int getMaxCount() {
            return this.maxCount;
        }

        @Override
        public int getUsedCount() {
            return this.onFlyCounter.get();
        }

        @Override
        public int getCachedCount() {
            return this.cache.size();
        }
    }
}

