/*
 * Decompiled with CFR 0.152.
 */
package io.journalkeeper.core.journal;

import io.journalkeeper.core.api.JournalEntry;
import io.journalkeeper.core.api.JournalEntryParser;
import io.journalkeeper.core.api.RaftJournal;
import io.journalkeeper.core.journal.JournalSnapshot;
import io.journalkeeper.core.journal.NosuchPartitionException;
import io.journalkeeper.exceptions.IndexOverflowException;
import io.journalkeeper.exceptions.IndexUnderflowException;
import io.journalkeeper.exceptions.JournalException;
import io.journalkeeper.persistence.BufferPool;
import io.journalkeeper.persistence.JournalPersistence;
import io.journalkeeper.persistence.PersistenceFactory;
import io.journalkeeper.persistence.TooManyBytesException;
import io.journalkeeper.utils.ThreadSafeFormat;
import java.io.Closeable;
import java.io.Flushable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Journal
implements RaftJournal,
Flushable,
Closeable {
    public static final int INDEX_STORAGE_SIZE = 8;
    private static final Logger logger = LoggerFactory.getLogger(Journal.class);
    private static final String PARTITION_PATH = "index";
    private static final String INDEX_PATH = "index/all";
    private static final String JOURNAL_PROPERTIES_PATTERN = "^persistence\\.journal\\.(.*)$";
    private static final String INDEX_PROPERTIES_PATTERN = "^persistence\\.index\\.(.*)$";
    private static final Properties DEFAULT_JOURNAL_PROPERTIES = new Properties();
    private static final Properties DEFAULT_INDEX_PROPERTIES = new Properties();
    private final AtomicLong commitIndex = new AtomicLong(0L);
    private final byte[] indexBytes = new byte[8];
    private final ByteBuffer indexBuffer = ByteBuffer.wrap(this.indexBytes);
    private final byte[] commitIndexBytes = new byte[8];
    private final ByteBuffer commitIndexBuffer = ByteBuffer.wrap(this.commitIndexBytes);
    private final JournalPersistence indexPersistence;
    private final JournalPersistence journalPersistence;
    private final Map<Integer, JournalPersistence> partitionMap;
    private final PersistenceFactory persistenceFactory;
    private final BufferPool bufferPool;
    private final JournalEntryParser journalEntryParser;
    private Path basePath = null;
    private Properties indexProperties;
    private ReadWriteLock readWriteLock = new ReentrantReadWriteLock();

    public Journal(PersistenceFactory persistenceFactory, BufferPool bufferPool, JournalEntryParser journalEntryParser) {
        this.indexPersistence = persistenceFactory.createJournalPersistenceInstance();
        this.journalPersistence = persistenceFactory.createJournalPersistenceInstance();
        this.persistenceFactory = persistenceFactory;
        this.journalEntryParser = journalEntryParser;
        this.partitionMap = new ConcurrentHashMap<Integer, JournalPersistence>();
        this.bufferPool = bufferPool;
    }

    public long minIndex() {
        return this.indexPersistence.min() / 8L;
    }

    public long maxIndex() {
        return this.indexPersistence.max() / 8L;
    }

    public long minIndex(int partition) {
        return this.getPartitionPersistence(partition).min() / 8L;
    }

    public long maxIndex(int partition) {
        return this.getPartitionPersistence(partition).max() / 8L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void compact(JournalSnapshot journalSnapshot) throws IOException {
        if (journalSnapshot.minIndex() <= this.minIndex()) {
            return;
        }
        if (journalSnapshot.minIndex() > this.commitIndex()) {
            throw new IllegalArgumentException(String.format("Given snapshot min index %s should less than commit index %s!", ThreadSafeFormat.formatWithComma((long)journalSnapshot.minIndex()), ThreadSafeFormat.formatWithComma((long)this.commitIndex())));
        }
        Map<Integer, Long> partitionMinIndices = journalSnapshot.partitionMinIndices();
        Map<Integer, JournalPersistence> map = this.partitionMap;
        synchronized (map) {
            Iterator<Map.Entry<Integer, JournalPersistence>> iterator = this.partitionMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<Integer, JournalPersistence> entry = iterator.next();
                int partition = entry.getKey();
                JournalPersistence partitionPersistence = entry.getValue();
                if (partitionMinIndices.containsKey(partition)) {
                    long minPartitionIndices = partitionMinIndices.get(entry.getKey());
                    partitionPersistence.compact(minPartitionIndices * 8L);
                    continue;
                }
                this.journalPersistence.close();
                this.journalPersistence.delete();
                iterator.remove();
            }
            for (Map.Entry<Integer, Long> entry : partitionMinIndices.entrySet()) {
                int partition = entry.getKey();
                if (this.partitionMap.containsKey(partition)) continue;
                JournalPersistence partitionPersistence = this.persistenceFactory.createJournalPersistenceInstance();
                partitionPersistence.recover(this.basePath.resolve(PARTITION_PATH).resolve(String.valueOf(partition)), partitionMinIndices.get(partition) * 8L, this.indexProperties);
                this.partitionMap.put(partition, partitionPersistence);
            }
        }
        this.indexPersistence.compact(journalSnapshot.minIndex() * 8L);
        this.journalPersistence.compact(journalSnapshot.minOffset());
    }

    public long append(JournalEntry entry) {
        long offset = this.journalPersistence.max();
        byte[] serializedEntry = entry.getSerializedBytes();
        try {
            this.journalPersistence.append(serializedEntry);
            this.indexBuffer.clear();
            this.indexBuffer.putLong(offset);
            this.indexPersistence.append(this.indexBytes);
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
        return this.maxIndex();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear(JournalSnapshot snapshot) throws IOException {
        this.commitIndex.set(snapshot.minIndex());
        Map<Integer, Long> partitionMinIndices = snapshot.partitionMinIndices();
        Map<Integer, JournalPersistence> map = this.partitionMap;
        synchronized (map) {
            Iterator<Map.Entry<Integer, JournalPersistence>> iterator = this.partitionMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<Integer, JournalPersistence> entry = iterator.next();
                int partition = entry.getKey();
                JournalPersistence partitionPersistence = entry.getValue();
                this.journalPersistence.close();
                this.journalPersistence.delete();
                iterator.remove();
            }
            for (Map.Entry<Integer, Long> entry : partitionMinIndices.entrySet()) {
                int partition = entry.getKey();
                JournalPersistence partitionPersistence = this.persistenceFactory.createJournalPersistenceInstance();
                partitionPersistence.recover(this.basePath.resolve(PARTITION_PATH).resolve(String.valueOf(partition)), partitionMinIndices.get(partition) * 8L, this.indexProperties);
                this.partitionMap.put(partition, partitionPersistence);
            }
        }
    }

    public List<Long> append(List<JournalEntry> entries) {
        long offset = this.journalPersistence.max();
        long index = this.maxIndex();
        ByteBuffer indicesBuffer = ByteBuffer.wrap(new byte[entries.size() * 8]);
        ArrayList<byte[]> entryBuffers = new ArrayList<byte[]>(entries.size());
        ArrayList<Long> indices = new ArrayList<Long>(entries.size());
        for (JournalEntry entry : entries) {
            entryBuffers.add(entry.getSerializedBytes());
            indicesBuffer.putLong(offset);
            offset += (long)entry.getLength();
            indices.add(++index);
        }
        try {
            this.journalPersistence.append(entryBuffers);
            this.indexPersistence.append(indicesBuffer.array());
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
        return indices;
    }

    public void commit(long index) throws IOException {
        long finalCommitIndex;
        while ((finalCommitIndex = this.commitIndex.get()) < index && finalCommitIndex < this.maxIndex() && this.commitIndex.compareAndSet(finalCommitIndex, finalCommitIndex + 1L)) {
            long offset = this.readOffset(finalCommitIndex);
            JournalEntry header = this.readEntryHeaderByOffset(offset);
            this.commitIndexBuffer.clear();
            this.commitIndexBuffer.putLong(offset);
            this.appendPartitionIndex(this.commitIndexBytes, header.getPartition(), header.getBatchSize());
        }
    }

    public JournalEntry readEntryHeaderByOffset(long offset) {
        try {
            byte[] headerBytes = this.journalPersistence.read(offset, this.journalEntryParser.headerLength());
            return this.journalEntryParser.parseHeader(headerBytes);
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
    }

    public long commitIndex() {
        return this.commitIndex.get();
    }

    private void appendPartitionIndex(byte[] offset, int partition, int batchSize) throws IOException {
        if (!this.partitionMap.containsKey(partition)) {
            this.addPartition(partition, 0L);
        }
        JournalPersistence partitionPersistence = this.getPartitionPersistence(partition);
        if (batchSize > 1) {
            byte[] bytes = new byte[batchSize * 8];
            ByteBuffer pb = ByteBuffer.wrap(bytes);
            for (int j = 0; j < batchSize; ++j) {
                if (j == 0) {
                    pb.put(offset);
                    continue;
                }
                pb.putLong(-1 * j);
            }
            partitionPersistence.append(bytes);
        } else {
            partitionPersistence.append(offset);
        }
    }

    private JournalPersistence getPartitionPersistence(int partition) {
        JournalPersistence partitionPersistence = this.partitionMap.get(partition);
        if (null == partitionPersistence) {
            throw new NosuchPartitionException(partition);
        }
        return partitionPersistence;
    }

    public long appendBatchRaw(List<byte[]> storageEntries) {
        long[] offsets = new long[storageEntries.size()];
        long offset = this.journalPersistence.max();
        for (int i = 0; i < offsets.length; ++i) {
            offsets[i] = offset;
            offset += (long)storageEntries.get(i).length;
        }
        try {
            for (byte[] storageEntry : storageEntries) {
                this.journalPersistence.append(storageEntry);
            }
            int indexBufferLength = 8 * storageEntries.size();
            ByteBuffer indexBuffer = ByteBuffer.allocate(indexBufferLength);
            for (long index : offsets) {
                indexBuffer.putLong(index);
            }
            indexBuffer.flip();
            try {
                this.indexPersistence.append(indexBuffer.array());
            }
            catch (TooManyBytesException e) {
                ByteBuffer buffer = ByteBuffer.allocate(8);
                for (long index : offsets) {
                    buffer.putLong(0, index);
                    this.indexPersistence.append(buffer.array());
                }
            }
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
        return this.maxIndex();
    }

    public JournalEntry readByPartition(int partition, long index) {
        int relIndex;
        long journalOffset;
        JournalPersistence pp = this.getPartitionPersistence(partition);
        long offset = this.readOffset(pp, index);
        if (offset < 0L) {
            journalOffset = this.readOffset(pp, index + offset);
            relIndex = (int)(-1L * offset);
        } else {
            journalOffset = offset;
            relIndex = 0;
        }
        JournalEntry journal = this.readByOffset(journalOffset);
        journal.setOffset(relIndex);
        return journal;
    }

    public List<JournalEntry> batchReadByPartition(int partition, long startPartitionIndex, int maxSize) {
        LinkedList<JournalEntry> list = new LinkedList<JournalEntry>();
        int size = 0;
        long index = startPartitionIndex;
        while (size < maxSize) {
            JournalEntry batchEntry = this.readByPartition(partition, index);
            int count = batchEntry.getBatchSize() - batchEntry.getOffset();
            size += count;
            index += (long)count;
            list.add(batchEntry);
        }
        return list;
    }

    public JournalEntry read(long index) {
        return this.journalEntryParser.parse(this.readRaw(index));
    }

    public JournalEntry readByOffset(long offset) {
        return this.journalEntryParser.parse(this.readRawByOffset(offset));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] readRaw(long index) {
        this.readWriteLock.readLock().lock();
        this.checkIndex(index);
        try {
            long offset = this.readOffset(index);
            byte[] byArray = this.readRawByOffset(offset);
            return byArray;
        }
        finally {
            this.readWriteLock.readLock().unlock();
        }
    }

    private byte[] readRawByOffset(long offset) {
        this.readWriteLock.readLock().lock();
        try {
            int length = this.readEntryLengthByOffset(offset);
            byte[] byArray = this.journalPersistence.read(offset, length);
            return byArray;
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
        finally {
            this.readWriteLock.readLock().unlock();
        }
    }

    private int readEntryLengthByOffset(long offset) {
        return this.readEntryHeaderByOffset(offset).getLength();
    }

    public long readOffset(long index) {
        return this.readOffset(this.indexPersistence, index);
    }

    private long readOffset(JournalPersistence indexPersistence, long index) {
        try {
            return indexPersistence.readLong(index * 8L);
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
    }

    public List<JournalEntry> batchRead(long index, int size) {
        this.checkIndex(index);
        ArrayList<JournalEntry> list = new ArrayList<JournalEntry>(size);
        long i = index;
        while (list.size() < size && i < this.maxIndex()) {
            list.add(this.read(i++));
        }
        return list;
    }

    public List<byte[]> readRaw(long index, int size) {
        this.checkIndex(index);
        ArrayList<byte[]> list = new ArrayList<byte[]>(size);
        long i = index;
        while (list.size() < size && i < this.maxIndex()) {
            list.add(this.readRaw(i++));
        }
        return list;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getTerm(long index) {
        this.readWriteLock.readLock().lock();
        try {
            if (index == -1L) {
                int n = -1;
                return n;
            }
            this.checkIndex(index);
            long offset = this.readOffset(index);
            int n = this.readEntryHeaderByOffset(offset).getTerm();
            return n;
        }
        finally {
            this.readWriteLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void compareOrAppendRaw(List<byte[]> rawEntries, long startIndex) {
        List entries = rawEntries.stream().map(arg_0 -> ((JournalEntryParser)this.journalEntryParser).parse(arg_0)).collect(Collectors.toList());
        try {
            long index = startIndex;
            int i = 0;
            while (i < entries.size()) {
                if (index < this.maxIndex() && this.getTerm(index) != ((JournalEntry)entries.get(i)).getTerm()) {
                    this.readWriteLock.writeLock().lock();
                    try {
                        this.truncate(index);
                    }
                    finally {
                        this.readWriteLock.writeLock().unlock();
                    }
                }
                if (index == this.maxIndex()) {
                    this.appendBatchRaw(rawEntries.subList(i, entries.size()));
                    break;
                }
                ++i;
                ++index;
            }
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
    }

    private void truncate(long index) throws IOException {
        long finalCommitIndex;
        long journalOffset = this.readOffset(index);
        this.truncatePartitions(journalOffset);
        this.indexPersistence.truncate(index * 8L);
        this.journalPersistence.truncate(journalOffset);
        while ((finalCommitIndex = this.commitIndex.get()) > this.maxIndex() && !this.commitIndex.compareAndSet(finalCommitIndex, this.maxIndex())) {
            Thread.yield();
        }
    }

    private void truncatePartitions(long journalOffset) throws IOException {
        for (JournalPersistence partitionPersistence : this.partitionMap.values()) {
            long offset;
            long position;
            for (position = partitionPersistence.max() - 8L; position > partitionPersistence.min() && (offset = this.readOffset(partitionPersistence, position / 8L)) >= journalOffset; position -= 8L) {
            }
            partitionPersistence.truncate(position <= partitionPersistence.min() ? 0L : position + 8L);
        }
    }

    public void recover(Path path, long commitIndex, JournalSnapshot journalSnapshot, Properties properties) throws IOException {
        this.basePath = path;
        Path indexPath = path.resolve(INDEX_PATH);
        Path partitionPath = path.resolve(PARTITION_PATH);
        Properties journalProperties = this.replacePropertiesNames(properties, JOURNAL_PROPERTIES_PATTERN, DEFAULT_JOURNAL_PROPERTIES);
        this.journalPersistence.recover(path, journalSnapshot.minOffset(), journalProperties);
        this.truncateJournalTailPartialEntry();
        this.indexProperties = this.replacePropertiesNames(properties, INDEX_PROPERTIES_PATTERN, DEFAULT_INDEX_PROPERTIES);
        this.indexPersistence.recover(indexPath, journalSnapshot.minIndex() * 8L, this.indexProperties);
        this.indexPersistence.truncate(this.indexPersistence.max() - this.indexPersistence.max() % 8L);
        this.truncateExtraIndices();
        this.buildMissingIndices();
        this.checkAndSetCommitIndex(commitIndex);
        this.recoverPartitions(partitionPath, journalSnapshot.partitionMinIndices(), this.indexProperties);
        this.flush();
        logger.debug("Journal recovered, minIndex: {}, maxIndex: {}, partitions: {}, path: {}.", new Object[]{this.minIndex(), this.maxIndex(), this.partitionMap.keySet(), path.toAbsolutePath().toString()});
    }

    private void checkAndSetCommitIndex(long commitIndex) {
        this.commitIndex.set(commitIndex);
        if (this.commitIndex.get() < this.minIndex()) {
            logger.info("Journal commitIndex {} should be not less than minIndex {}, set commitIndex to {}.", new Object[]{this.commitIndex.get(), this.minIndex(), this.minIndex()});
            this.commitIndex.set(this.minIndex());
        } else if (this.commitIndex.get() > this.maxIndex()) {
            logger.info("Journal commitIndex {} should be not greater than maxIndex {}, set commitIndex to {}.", new Object[]{this.commitIndex.get(), this.maxIndex(), this.maxIndex()});
            this.commitIndex.set(this.maxIndex());
        }
    }

    private void recoverPartitions(Path partitionPath, Map<Integer, Long> partitionIndices, Properties properties) throws IOException {
        int length;
        long commitOffset;
        HashMap<Integer, Long> lastIndexedOffsetMap = new HashMap<Integer, Long>(partitionIndices.size());
        for (Map.Entry<Integer, Long> entry : partitionIndices.entrySet()) {
            int partition = entry.getKey();
            long lastIncludedIndex = entry.getValue();
            JournalPersistence pp = this.persistenceFactory.createJournalPersistenceInstance();
            pp.recover(partitionPath.resolve(String.valueOf(partition)), lastIncludedIndex * 8L, properties);
            pp.truncate(pp.max() - pp.max() % 8L);
            this.truncateTailPartialBatchIndices(pp);
            this.partitionMap.put(partition, pp);
            lastIndexedOffsetMap.put(partition, this.getLastIndexedOffset(pp));
        }
        long l2 = commitOffset = this.commitIndex.get() == this.maxIndex() ? this.journalPersistence.max() : this.readOffset(this.commitIndex.get());
        for (long offset = lastIndexedOffsetMap.values().stream().mapToLong(l -> l).min().orElse(this.journalPersistence.min()); offset < commitOffset; offset += (long)length) {
            JournalEntry header = this.readEntryHeaderByOffset(offset);
            length = this.readEntryLengthByOffset(offset);
            if (offset <= (Long)lastIndexedOffsetMap.get(header.getPartition())) continue;
            this.indexBuffer.clear();
            this.indexBuffer.putLong(offset);
            this.appendPartitionIndex(this.indexBytes, header.getPartition(), header.getBatchSize());
        }
        for (Integer partition : partitionIndices.keySet()) {
            long journalOffset;
            JournalPersistence partitionPersistence = this.partitionMap.get(partition);
            long partitionIndex = partitionPersistence.max() / 8L - 1L;
            while (partitionIndex * 8L >= partitionPersistence.min() && (journalOffset = this.readOffset(partitionPersistence, partitionIndex)) >= commitOffset) {
                --partitionIndex;
            }
            partitionPersistence.truncate(++partitionIndex * 8L);
        }
    }

    private long getLastIndexedOffset(JournalPersistence pp) {
        long lastIndexedOffset = this.journalPersistence.min();
        if (pp.max() > 0L) {
            long lastIndex = pp.max() / 8L - 1L;
            long lastOffset = this.readOffset(pp, lastIndex);
            if (lastOffset < 0L) {
                long firstIndexOfBatchEntries = lastIndex + lastOffset;
                lastIndexedOffset = this.readOffset(pp, firstIndexOfBatchEntries);
            } else {
                lastIndexedOffset = lastOffset;
            }
        }
        return lastIndexedOffset;
    }

    private void truncateTailPartialBatchIndices(JournalPersistence pp) throws IOException {
        long firstIndexOfBatchEntries;
        long batchEntriesOffset;
        JournalEntry header;
        long lastIndex;
        long lastOffset;
        if (pp.max() > pp.min() && (lastOffset = this.readOffset(pp, lastIndex = pp.max() / 8L - 1L)) < 0L && (lastIndex - 1L) * -1L < (long)(header = this.readEntryHeaderByOffset(batchEntriesOffset = this.readOffset(pp, firstIndexOfBatchEntries = lastIndex + lastOffset))).getBatchSize()) {
            pp.truncate(firstIndexOfBatchEntries * 8L);
        }
    }

    private Properties replacePropertiesNames(Properties properties, String fromNameRegex, Properties defaultProperties) {
        Properties jp = new Properties(defaultProperties);
        properties.stringPropertyNames().forEach(k -> {
            String name = k.replaceAll(fromNameRegex, "$1");
            jp.setProperty(name, properties.getProperty((String)k));
        });
        return jp;
    }

    private void buildMissingIndices() throws IOException {
        long indexOffset;
        if (this.indexPersistence.max() - 8L >= this.indexPersistence.min()) {
            long offset = this.readOffset(this.indexPersistence.max() / 8L - 1L);
            int length = this.readEntryLengthByOffset(offset);
            indexOffset = offset + (long)length;
        } else {
            indexOffset = this.journalPersistence.min();
        }
        LinkedList<Long> indices = new LinkedList<Long>();
        while (indexOffset < this.journalPersistence.max()) {
            indices.add(indexOffset);
            indexOffset += (long)this.readEntryLengthByOffset(indexOffset);
        }
        if (!indices.isEmpty()) {
            ByteBuffer buffer = this.bufferPool.allocate(indices.size() * 8);
            indices.forEach(buffer::putLong);
            buffer.flip();
            this.indexPersistence.append(buffer.array());
            this.bufferPool.release(buffer);
        }
    }

    private void truncateExtraIndices() throws IOException {
        long position = this.indexPersistence.max();
        while ((position -= 8L) >= this.indexPersistence.min() && this.readOffset(position / 8L) >= this.journalPersistence.max()) {
        }
        this.indexPersistence.truncate(position + 8L);
    }

    private void truncateJournalTailPartialEntry() throws IOException {
        long lastEntryPosition = -1L;
        JournalEntry lastEntryHeader = null;
        for (long position = this.journalPersistence.max() - (long)this.journalEntryParser.headerLength(); position >= this.journalPersistence.min(); --position) {
            try {
                JournalEntry header = this.readByOffset(position);
                if (lastEntryPosition < 0L) {
                    lastEntryPosition = position;
                    lastEntryHeader = header;
                    if (lastEntryPosition != this.journalPersistence.min()) continue;
                    this.truncatePartialEntry(lastEntryPosition, lastEntryHeader);
                    return;
                }
                if (position + (long)header.getLength() == lastEntryPosition) {
                    this.truncatePartialEntry(lastEntryPosition, lastEntryHeader);
                    return;
                }
                lastEntryPosition = position;
                lastEntryHeader = header;
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        this.journalPersistence.truncate(this.journalPersistence.min());
    }

    private void truncatePartialEntry(long lastEntryPosition, JournalEntry lastEntryHeader) throws IOException {
        if (lastEntryPosition + (long)lastEntryHeader.getLength() <= this.journalPersistence.max()) {
            this.journalPersistence.truncate(lastEntryPosition + (long)lastEntryHeader.getLength());
        } else {
            this.journalPersistence.truncate(lastEntryPosition);
        }
    }

    @Override
    public void flush() {
        long flushed;
        while ((flushed = Stream.concat(Stream.of(this.journalPersistence, this.indexPersistence), this.partitionMap.values().stream()).filter(p -> p.flushed() < p.max()).peek(p -> {
            try {
                p.flush();
            }
            catch (IOException e) {
                logger.warn("Flush {} exception: ", (Object)p.getBasePath(), (Object)e);
                throw new JournalException((Throwable)e);
            }
        }).count()) > 0L) {
        }
    }

    public boolean isDirty() {
        return Stream.concat(Stream.of(this.journalPersistence, this.indexPersistence), this.partitionMap.values().stream()).anyMatch(p -> p.flushed() < p.max());
    }

    private void checkIndex(long index) {
        long position = index * 8L;
        if (position < this.indexPersistence.min()) {
            throw new IndexUnderflowException();
        }
        if (position >= this.indexPersistence.max()) {
            throw new IndexOverflowException();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void rePartition(Set<Integer> partitions) {
        try {
            Map<Integer, JournalPersistence> map = this.partitionMap;
            synchronized (map) {
                for (int partition : partitions) {
                    if (this.partitionMap.containsKey(partition)) continue;
                    this.addPartition(partition, 0L);
                }
                ArrayList<Integer> toBeRemoved = new ArrayList<Integer>();
                for (Map.Entry<Integer, JournalPersistence> entry : this.partitionMap.entrySet()) {
                    if (partitions.contains(entry.getKey())) continue;
                    toBeRemoved.add(entry.getKey());
                }
                for (Integer partition : toBeRemoved) {
                    this.removePartition(partition);
                }
                logger.debug("Journal repartitioned, partitions: {}, path: {}.", this.partitionMap.keySet(), (Object)this.basePath.toAbsolutePath().toString());
            }
        }
        catch (IOException e) {
            throw new JournalException((Throwable)e);
        }
    }

    public long queryIndexByTimestamp(int partition, long timestamp) {
        try {
            if (this.partitionMap.containsKey(partition)) {
                JournalPersistence indexStore = this.partitionMap.get(partition);
                long searchedIndex = this.binarySearchByTimestamp(timestamp, indexStore, indexStore.min() / 8L, indexStore.max() / 8L - 1L);
                while (searchedIndex - 1L >= indexStore.min() && timestamp <= this.getStorageTimestamp(indexStore, searchedIndex - 1L)) {
                    --searchedIndex;
                }
                return searchedIndex;
            }
        }
        catch (Throwable e) {
            logger.warn("Query index by timestamp exception: ", e);
        }
        return -1L;
    }

    private long binarySearchByTimestamp(long timestamp, JournalPersistence indexStore, long leftIndexInclude, long rightIndexInclude) {
        if (rightIndexInclude <= leftIndexInclude) {
            return -1L;
        }
        if (timestamp <= this.getStorageTimestamp(indexStore, leftIndexInclude)) {
            return leftIndexInclude;
        }
        if (timestamp >= this.getStorageTimestamp(indexStore, rightIndexInclude)) {
            return rightIndexInclude;
        }
        if (leftIndexInclude + 1L == rightIndexInclude) {
            return leftIndexInclude;
        }
        long mid = leftIndexInclude + (rightIndexInclude - leftIndexInclude) / 2L;
        long midTimestamp = this.getStorageTimestamp(indexStore, mid);
        if (timestamp < midTimestamp) {
            return this.binarySearchByTimestamp(timestamp, indexStore, leftIndexInclude, mid);
        }
        return this.binarySearchByTimestamp(timestamp, indexStore, mid, rightIndexInclude);
    }

    private long getStorageTimestamp(JournalPersistence indexStore, long index) {
        JournalEntry header = this.readEntryHeaderByOffset(this.readOffset(indexStore, index));
        return header.getTimestamp();
    }

    public Set<Integer> getPartitions() {
        return new HashSet<Integer>(this.partitionMap.keySet());
    }

    public void addPartition(int partition) throws IOException {
        this.addPartition(partition, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addPartition(int partition, long minIndex) throws IOException {
        Map<Integer, JournalPersistence> map = this.partitionMap;
        synchronized (map) {
            if (!this.partitionMap.containsKey(partition)) {
                JournalPersistence partitionPersistence = this.persistenceFactory.createJournalPersistenceInstance();
                partitionPersistence.recover(this.basePath.resolve(PARTITION_PATH).resolve(String.valueOf(partition)), minIndex * 8L, this.indexProperties);
                this.partitionMap.put(partition, partitionPersistence);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removePartition(int partition) throws IOException {
        Map<Integer, JournalPersistence> map = this.partitionMap;
        synchronized (map) {
            JournalPersistence removedPersistence = this.partitionMap.remove(partition);
            if (removedPersistence != null) {
                logger.info("Partition removed: {}, journal: {}.", (Object)partition, (Object)this.basePath.toAbsolutePath().toString());
                removedPersistence.close();
                removedPersistence.delete();
            }
        }
    }

    @Override
    public void close() throws IOException {
        for (JournalPersistence persistence : this.partitionMap.values()) {
            persistence.close();
        }
        this.indexPersistence.close();
        this.journalPersistence.close();
    }

    public long flushedIndex() {
        return this.indexPersistence.flushed() / 8L;
    }

    public JournalPersistence getJournalPersistence() {
        return this.journalPersistence;
    }

    public JournalPersistence getIndexPersistence() {
        return this.indexPersistence;
    }

    public Map<Integer, JournalPersistence> getPartitionMap() {
        return Collections.unmodifiableMap(this.partitionMap);
    }

    public long maxOffset() {
        return this.journalPersistence.max();
    }

    public Map<Integer, Long> calcPartitionIndices(long journalOffset) {
        HashMap<Integer, Long> partitionIndices = new HashMap<Integer, Long>(this.partitionMap.size());
        for (Map.Entry<Integer, JournalPersistence> entry : this.partitionMap.entrySet()) {
            long offset;
            int partition = entry.getKey();
            JournalPersistence partitionPersistence = entry.getValue();
            long index = this.maxIndex(partition);
            while (--index >= this.minIndex(partition) && (offset = this.readOffset(partitionPersistence, index)) >= journalOffset) {
            }
            partitionIndices.put(partition, index + 1L);
        }
        return partitionIndices;
    }

    public String toString() {
        return "Journal{commitIndex=" + this.commitIndex + ", indexPersistence=" + this.indexPersistence + ", journalPersistence=" + this.journalPersistence + ", basePath=" + this.basePath + '}';
    }

    static {
        DEFAULT_JOURNAL_PROPERTIES.put("file_data_size", String.valueOf(0x8000000));
        DEFAULT_JOURNAL_PROPERTIES.put("cached_file_core_count", String.valueOf(3));
        DEFAULT_JOURNAL_PROPERTIES.put("cached_file_max_count", String.valueOf(10));
        DEFAULT_JOURNAL_PROPERTIES.put("max_dirty_size", String.valueOf(0x8000000));
        DEFAULT_INDEX_PROPERTIES.put("file_data_size", String.valueOf(0x1000000));
        DEFAULT_INDEX_PROPERTIES.put("cached_file_core_count", String.valueOf(12));
        DEFAULT_INDEX_PROPERTIES.put("cached_file_max_count", String.valueOf(40));
    }
}

