/*
 * Decompiled with CFR 0.152.
 */
package io.journalkeeper.core.transaction;

import io.journalkeeper.core.api.JournalEntry;
import io.journalkeeper.core.api.JournalEntryParser;
import io.journalkeeper.core.api.UpdateRequest;
import io.journalkeeper.core.api.transaction.JournalKeeperTransactionContext;
import io.journalkeeper.core.api.transaction.UUIDTransactionId;
import io.journalkeeper.core.journal.Journal;
import io.journalkeeper.core.transaction.TransactionEntry;
import io.journalkeeper.core.transaction.TransactionEntrySerializer;
import io.journalkeeper.core.transaction.TransactionEntryType;
import io.journalkeeper.exceptions.TransactionException;
import io.journalkeeper.rpc.client.ClientServerRpc;
import io.journalkeeper.rpc.client.UpdateClusterStateRequest;
import io.journalkeeper.rpc.client.UpdateClusterStateResponse;
import io.journalkeeper.utils.state.ServerStateMachine;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class JournalTransactionState
extends ServerStateMachine {
    private static final Logger logger = LoggerFactory.getLogger(JournalTransactionState.class);
    private static final long RETRY_COMPLETE_TRANSACTION_INTERVAL_MS = 10000L;
    private final Journal journal;
    private final Map<Integer, TransactionEntryType> partitionStatusMap;
    private final Map<UUID, TransactionState> openingTransactionMap;
    private final ClientServerRpc server;
    private final TransactionEntrySerializer transactionEntrySerializer = new TransactionEntrySerializer();
    private final AtomicInteger nextFreePartition = new AtomicInteger(30000);
    private final DelayQueue<CompleteTransactionRetry> retryCompleteTransactions = new DelayQueue();
    private final ScheduledExecutorService scheduledExecutor;
    private final long transactionTimeoutMs;
    private ScheduledFuture retryCompleteTransactionScheduledFuture = null;
    private ScheduledFuture checkOutdatedTransactionsScheduledFuture = null;

    JournalTransactionState(Journal journal, long transactionTimeoutMs, ClientServerRpc server, ScheduledExecutorService scheduledExecutor) {
        super(false);
        this.journal = journal;
        this.transactionTimeoutMs = transactionTimeoutMs;
        this.server = server;
        this.scheduledExecutor = scheduledExecutor;
        this.partitionStatusMap = new HashMap<Integer, TransactionEntryType>(32);
        this.openingTransactionMap = new HashMap<UUID, TransactionState>(32);
    }

    protected void doStart() {
        super.doStart();
        this.recoverTransactionState();
        this.retryCompleteTransactionScheduledFuture = this.scheduledExecutor.scheduleWithFixedDelay(this::retryCompleteTransactions, 10000L, 10000L, TimeUnit.MILLISECONDS);
        this.checkOutdatedTransactionsScheduledFuture = this.scheduledExecutor.scheduleWithFixedDelay(this::abortOutdatedTransactions, this.transactionTimeoutMs, this.transactionTimeoutMs, TimeUnit.MILLISECONDS);
    }

    protected void doStop() {
        if (null != this.retryCompleteTransactionScheduledFuture) {
            this.retryCompleteTransactionScheduledFuture.cancel(false);
        }
        if (null != this.checkOutdatedTransactionsScheduledFuture) {
            this.checkOutdatedTransactionsScheduledFuture.cancel(false);
        }
        super.doStop();
    }

    private void abortOutdatedTransactions() {
        long currentTimestamp = System.currentTimeMillis();
        this.openingTransactionMap.forEach((transactionId, state) -> {
            JournalEntry journalEntry;
            TransactionEntry transactionEntry;
            int partition = state.getPartition();
            long i = this.journal.maxIndex(partition);
            while (--i >= this.journal.minIndex(partition) && transactionId.equals((transactionEntry = this.transactionEntrySerializer.parse((journalEntry = this.journal.readByPartition(partition, i)).getPayload().getBytes())).getTransactionId())) {
                if (transactionEntry.getType() != TransactionEntryType.TRANSACTION_START) continue;
                long transactionCreateTimestamp = transactionEntry.getTimestamp();
                if (transactionCreateTimestamp + this.transactionTimeoutMs >= currentTimestamp) break;
                logger.info("Abort outdated transaction: {}.", (Object)transactionId.toString());
                this.writeTransactionCompleteEntry((UUID)transactionId, false, partition);
                break;
            }
        });
    }

    private void retryCompleteTransactions() {
        CompleteTransactionRetry retry;
        while ((retry = (CompleteTransactionRetry)this.retryCompleteTransactions.poll()) != null) {
            this.completeTransaction(retry.getTransactionId(), true, retry.getPartition());
        }
    }

    private void recoverTransactionState() {
        block0: for (int i = 0; i < 32; ++i) {
            int partition = 30000 + i;
            if (this.journal.getPartitions().contains(partition) && this.journal.maxIndex(partition) > 0L) {
                logger.info("Recover transaction partition {}...", (Object)partition);
                long index = this.journal.maxIndex(partition);
                long minIndexOfPartition = this.journal.minIndex(partition);
                boolean lastEntryOfTheTransaction = true;
                while (index-- > minIndexOfPartition) {
                    JournalEntry journalEntry = this.journal.readByPartition(partition, index);
                    TransactionEntry transactionEntry = this.transactionEntrySerializer.parse(journalEntry.getPayload().getBytes());
                    if (lastEntryOfTheTransaction) {
                        this.partitionStatusMap.put(partition, transactionEntry.getType());
                        if (transactionEntry.getType() == TransactionEntryType.TRANSACTION_PRE_COMPLETE) {
                            this.retryCompleteTransactions.put(new CompleteTransactionRetry(transactionEntry.getTransactionId(), partition));
                        }
                        lastEntryOfTheTransaction = false;
                    }
                    if (transactionEntry.getType() != TransactionEntryType.TRANSACTION_START) continue;
                    this.openingTransactionMap.put(transactionEntry.getTransactionId(), new TransactionState(partition, new JournalKeeperTransactionContext(new UUIDTransactionId(transactionEntry.getTransactionId()), transactionEntry.getContext(), transactionEntry.getTimestamp())));
                    continue block0;
                }
                continue;
            }
            this.partitionStatusMap.put(partition, TransactionEntryType.TRANSACTION_COMPLETE);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int nextFreePartition() {
        int partition;
        int start = partition = this.nextPartition();
        Map<Integer, TransactionEntryType> map = this.partitionStatusMap;
        synchronized (map) {
            while (this.partitionStatusMap.getOrDefault(partition, TransactionEntryType.TRANSACTION_COMPLETE) != TransactionEntryType.TRANSACTION_COMPLETE) {
                partition = this.nextPartition();
                if (partition != start) continue;
                throw new TransactionException("No free transaction partition!");
            }
            return partition;
        }
    }

    private int nextPartition() {
        int partition = this.nextFreePartition.getAndIncrement();
        this.nextFreePartition.compareAndSet(30032, 30000);
        return partition;
    }

    void applyEntry(TransactionEntry entry, int partition, Map<UUID, CompletableFuture<Void>> pendingCompleteTransactionFutures) {
        if (partition < 30000 || partition >= 30032) {
            logger.warn("Ignore transaction entry, cause: partition {} is not a transaction partition.", (Object)partition);
            return;
        }
        this.partitionStatusMap.put(partition, entry.getType());
        switch (entry.getType()) {
            case TRANSACTION_START: {
                this.openingTransactionMap.put(entry.getTransactionId(), new TransactionState(partition, new JournalKeeperTransactionContext(new UUIDTransactionId(entry.getTransactionId()), entry.getContext(), entry.getTimestamp())));
                break;
            }
            case TRANSACTION_PRE_COMPLETE: {
                this.completeTransaction(entry.getTransactionId(), entry.isCommitOrAbort(), partition);
                break;
            }
            case TRANSACTION_COMPLETE: {
                this.openingTransactionMap.remove(entry.getTransactionId());
                this.partitionStatusMap.remove(partition);
                CompletableFuture<Void> future = pendingCompleteTransactionFutures.remove(entry.getTransactionId());
                if (null == future) break;
                future.complete(null);
            }
        }
    }

    private void completeTransaction(UUID transactionId, boolean commitOrAbort, int partition) {
        if (commitOrAbort) {
            JournalEntry journalEntry;
            TransactionEntry transactionEntry;
            int entryCount = 0;
            long i = this.journal.maxIndex(partition);
            HashMap<Integer, List> transactionEntriesByPartition = new HashMap<Integer, List>();
            long minIndex = this.journal.minIndex(partition);
            while (transactionId.equals((transactionEntry = this.transactionEntrySerializer.parse((journalEntry = this.journal.readByPartition(partition, --i)).getPayload().getBytes())).getTransactionId()) && transactionEntry.getType() != TransactionEntryType.TRANSACTION_START) {
                if (transactionEntry.getType() == TransactionEntryType.TRANSACTION_ENTRY) {
                    ++entryCount;
                    List list = transactionEntriesByPartition.computeIfAbsent(transactionEntry.getPartition(), p -> new LinkedList());
                    list.add(transactionEntry);
                }
                if (i > minIndex) continue;
            }
            AtomicInteger unFinishedRequests = new AtomicInteger(entryCount);
            ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>(entryCount);
            for (Map.Entry me : transactionEntriesByPartition.entrySet()) {
                int bizPartition = (Integer)me.getKey();
                List transactionEntries = (List)me.getValue();
                for (TransactionEntry te : transactionEntries) {
                    futures.add(((CompletableFuture)this.server.updateClusterState(new UpdateClusterStateRequest(new UpdateRequest(te.getEntry(), bizPartition, te.getBatchSize()))).exceptionally(UpdateClusterStateResponse::new)).thenAccept(response -> {
                        if (response.success()) {
                            unFinishedRequests.decrementAndGet();
                        } else {
                            logger.warn("Transaction commit {} failed! Cause: {}.", (Object)transactionId.toString(), (Object)response.errorString());
                        }
                    }));
                }
            }
            CompletableFuture.allOf(futures.toArray(new CompletableFuture[entryCount])).thenRun(() -> {
                if (unFinishedRequests.get() > 0) {
                    this.retryCompleteTransactions.add(new CompleteTransactionRetry(transactionId, partition));
                } else {
                    this.writeTransactionCompleteEntry(transactionId, true, partition);
                }
            });
        } else {
            this.writeTransactionCompleteEntry(transactionId, false, partition);
        }
    }

    JournalEntry wrapTransactionalEntry(JournalEntry entry, UUID transactionId, JournalEntryParser journalEntryParser) {
        int transactionPartition = this.getPartition(transactionId);
        if (transactionPartition > 0) {
            int bizPartition = entry.getPartition();
            int batchSize = entry.getBatchSize();
            int term = entry.getTerm();
            TransactionEntry transactionEntry = new TransactionEntry(transactionId, bizPartition, batchSize, entry.getPayload().getBytes());
            byte[] serializedTransactionEntry = this.transactionEntrySerializer.serialize(transactionEntry);
            JournalEntry wrappedEntry = journalEntryParser.createJournalEntry(serializedTransactionEntry);
            wrappedEntry.setPartition(transactionPartition);
            wrappedEntry.setTerm(term);
            return wrappedEntry;
        }
        throw new TransactionException(String.format("Transaction %s is not open!", transactionId.toString()));
    }

    private void writeTransactionCompleteEntry(UUID transactionId, boolean commitOrAbort, int partition) {
        TransactionEntry entry = new TransactionEntry(transactionId, TransactionEntryType.TRANSACTION_COMPLETE, commitOrAbort);
        byte[] serializedEntry = this.transactionEntrySerializer.serialize(entry);
        ((CompletableFuture)this.server.updateClusterState(new UpdateClusterStateRequest(new UpdateRequest(serializedEntry, partition, 1))).exceptionally(UpdateClusterStateResponse::new)).thenAccept(response -> {
            if (response.success()) {
                logger.info("Transaction {} {}.", (Object)transactionId.toString(), (Object)(commitOrAbort ? "committed" : "aborted"));
            } else {
                logger.warn("Transaction {} {} failed! Cause: {}.", new Object[]{transactionId.toString(), commitOrAbort ? "commit" : "abort", response.errorString()});
            }
        });
    }

    Collection<JournalKeeperTransactionContext> getOpeningTransactions() {
        return this.openingTransactionMap.values().stream().map(TransactionState::getContext).collect(Collectors.toSet());
    }

    int getPartition(UUID transactionId) {
        TransactionState transactionState = this.openingTransactionMap.get(transactionId);
        return transactionState != null ? transactionState.getPartition() : -1;
    }

    void ensureTransactionOpen(UUID transactionId) {
        if (!this.openingTransactionMap.containsKey(transactionId)) {
            throw new IllegalStateException(String.format("Transaction %s is not open!", transactionId.toString()));
        }
    }

    boolean isTransactionPartition(int partition) {
        return partition >= 30000 && partition < 30032;
    }

    private static class TransactionState {
        private final int partition;
        private final JournalKeeperTransactionContext context;

        public TransactionState(int partition, JournalKeeperTransactionContext context) {
            this.partition = partition;
            this.context = context;
        }

        public int getPartition() {
            return this.partition;
        }

        public JournalKeeperTransactionContext getContext() {
            return this.context;
        }
    }

    private static class CompleteTransactionRetry
    implements Delayed {
        private final UUID transactionId;
        private final int partition;
        private final long expireTimeMs;

        CompleteTransactionRetry(UUID transactionId, int partition) {
            this.transactionId = transactionId;
            this.partition = partition;
            this.expireTimeMs = System.currentTimeMillis() + 10000L;
        }

        public UUID getTransactionId() {
            return this.transactionId;
        }

        public int getPartition() {
            return this.partition;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CompleteTransactionRetry that = (CompleteTransactionRetry)o;
            return this.partition == that.partition && this.transactionId.equals(that.transactionId);
        }

        public int hashCode() {
            return Objects.hash(this.transactionId, this.partition);
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.expireTimeMs - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            return (int)(this.getDelay(TimeUnit.MILLISECONDS) - o.getDelay(TimeUnit.MILLISECONDS));
        }
    }
}

