/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.store.file;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.joyqueue.store.PositionOverflowException;
import org.joyqueue.store.PositionUnderflowException;
import org.joyqueue.store.file.CorruptedLogException;
import org.joyqueue.store.file.LogSerializer;
import org.joyqueue.store.file.RollBackException;
import org.joyqueue.store.file.StoreFile;
import org.joyqueue.store.file.StoreFileImpl;
import org.joyqueue.store.file.WriteException;
import org.joyqueue.store.utils.PreloadBufferPool;
import org.joyqueue.toolkit.format.Format;
import org.joyqueue.toolkit.time.SystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PositioningStore<T>
implements Closeable {
    private final Logger logger = LoggerFactory.getLogger(PositioningStore.class);
    private final int fileHeaderSize;
    private final int fileDataSize;
    private final int diskFullRatio;
    private final int maxMessageLength;
    private final boolean loadOnRead;
    private final File base;
    private final LogSerializer<T> serializer;
    private final PreloadBufferPool bufferPool;
    private final NavigableMap<Long, StoreFile<T>> storeFileMap = new ConcurrentSkipListMap<Long, StoreFile<T>>();
    private final AtomicLong flushPosition = new AtomicLong(0L);
    private final AtomicLong rightPosition = new AtomicLong(0L);
    private final AtomicLong leftPosition = new AtomicLong(0L);
    private final Lock writeLock = new ReentrantLock();
    private final Lock flushLock = new ReentrantLock();
    private final ReentrantLock deleteLock = new ReentrantLock();
    private StoreFile<T> writeStoreFile = null;

    public PositioningStore(File base, Config config, PreloadBufferPool bufferPool, LogSerializer<T> serializer) {
        this.base = base;
        this.fileHeaderSize = config.fileHeaderSize;
        this.fileDataSize = config.fileDataSize;
        this.maxMessageLength = config.maxMessageLength;
        this.loadOnRead = config.loadOnRead;
        if (config.diskFullRatio <= 0 || config.diskFullRatio > 100) {
            this.logger.warn("Invalid config diskFullRatio: {}, using default: {}.", (Object)config.diskFullRatio, (Object)90);
            this.diskFullRatio = 90;
        } else {
            this.diskFullRatio = config.diskFullRatio;
        }
        this.bufferPool = bufferPool;
        this.serializer = serializer;
    }

    public long left() {
        return this.leftPosition.get();
    }

    public long right() {
        return this.rightPosition.get();
    }

    public long flushPosition() {
        return this.flushPosition.get();
    }

    public void clear(long position) {
        this.logger.info("Clear store, new position: {}, store: {}...", (Object)Format.formatWithComma((long)position), (Object)this.base.getAbsolutePath());
        try {
            this.flushLock.lock();
            this.writeLock.lock();
            this.deleteLock.lock();
            this.clear();
            this.leftPosition.set(position);
            this.rightPosition.set(position);
            this.flushPosition.set(position);
            this.resetWriteStoreFile();
        }
        finally {
            this.deleteLock.unlock();
            this.writeLock.unlock();
            this.flushLock.unlock();
        }
    }

    public void setRight(long position) throws IOException {
        if (position == this.right()) {
            return;
        }
        this.logger.info("Rollback to position: {}, left: {}, right: {}, flushPosition: {}, store: {}...", new Object[]{Format.formatWithComma((long)position), Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), this.base.getAbsolutePath()});
        try {
            this.flushLock.lock();
            this.writeLock.lock();
            this.deleteLock.lock();
            if (position <= this.left() || position > this.right()) {
                this.clear();
                this.leftPosition.set(position);
                this.rightPosition.set(position);
                this.flushPosition.set(position);
            } else if (position < this.right()) {
                this.rollbackFiles(position);
                this.rightPosition.set(position);
                if (this.flushPosition() > position) {
                    this.flushPosition.set(position);
                }
            }
            this.resetWriteStoreFile();
            this.deleteLock.unlock();
            this.writeLock.unlock();
            this.flushLock.unlock();
        }
        catch (Throwable throwable) {
            this.deleteLock.unlock();
            this.writeLock.unlock();
            this.flushLock.unlock();
            this.logger.info("Rollback finished, left: {}, right: {}, flushPosition: {}, store: {}.", new Object[]{Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), this.base.getAbsolutePath()});
            throw throwable;
        }
        this.logger.info("Rollback finished, left: {}, right: {}, flushPosition: {}, store: {}.", new Object[]{Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), this.base.getAbsolutePath()});
    }

    private void clear() {
        try {
            while (!this.storeFileMap.isEmpty()) {
                StoreFile storeFile = (StoreFile)this.storeFileMap.remove(this.storeFileMap.firstKey());
                this.forceDeleteStoreFile(storeFile);
                if (storeFile != this.writeStoreFile) continue;
                this.writeStoreFile = null;
            }
        }
        catch (IOException e) {
            throw new RollBackException(e);
        }
    }

    private void rollbackFiles(long position) throws IOException {
        if (!this.storeFileMap.isEmpty()) {
            Map.Entry<Long, StoreFile<T>> entry = this.storeFileMap.floorEntry(position);
            StoreFile<T> storeFile = entry.getValue();
            if (position > storeFile.position()) {
                int relPos = (int)(position - storeFile.position());
                this.logger.info("Truncate store file {} to relative position {}.", (Object)storeFile.file().getAbsolutePath(), (Object)relPos);
                storeFile.rollback(relPos);
            }
            SortedMap<Long, StoreFile<T>> toBeRemoved = this.storeFileMap.tailMap(position);
            while (!toBeRemoved.isEmpty()) {
                StoreFile removedStoreFile = (StoreFile)toBeRemoved.remove(toBeRemoved.firstKey());
                this.forceDeleteStoreFile(removedStoreFile);
                if (this.writeStoreFile != removedStoreFile) continue;
                this.writeStoreFile = null;
            }
        }
    }

    private void resetWriteStoreFile() {
        if (!this.storeFileMap.isEmpty()) {
            StoreFile<T> storeFile = this.storeFileMap.lastEntry().getValue();
            this.writeStoreFile = storeFile.position() + (long)storeFile.size() > this.right() ? storeFile : null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void recover() throws IOException {
        this.logger.info("Recovering store file: {}...", (Object)this.base.getAbsolutePath());
        try {
            this.flushLock.lock();
            this.writeLock.lock();
            this.deleteLock.lock();
            this.recoverFileMap();
            long recoverPosition = this.storeFileMap.isEmpty() ? 0L : (Long)this.storeFileMap.lastKey() + (long)this.storeFileMap.lastEntry().getValue().fileDataSize();
            this.flushPosition.set(recoverPosition);
            this.rightPosition.set(recoverPosition);
            this.leftPosition.set(this.storeFileMap.isEmpty() ? 0L : (Long)this.storeFileMap.firstKey());
            if (recoverPosition > 0L) {
                long lastLogTail = this.toLogTail(recoverPosition - 1L);
                if (lastLogTail < 0L) {
                    throw new CorruptedLogException(String.format("Unable to read any valid log. Corrupted log files: %s.", this.base.getAbsolutePath()));
                }
                if (lastLogTail < recoverPosition) {
                    this.rollbackFiles(lastLogTail);
                    this.flushPosition.set(lastLogTail);
                    this.rightPosition.set(lastLogTail);
                }
            }
            this.resetWriteStoreFile();
        }
        finally {
            this.deleteLock.unlock();
            this.writeLock.unlock();
            this.flushLock.unlock();
        }
        this.logger.info("Store recovered, leftPosition: {}, rightPosition: {}, flushPosition: {},  base: {}.", new Object[]{Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), this.base.getAbsolutePath()});
    }

    private void recoverFileMap() throws IOException {
        File[] files = this.base.listFiles(file -> file.isFile() && file.getName().matches("\\d+"));
        if (null != files) {
            for (File file2 : files) {
                long filePosition = Long.parseLong(file2.getName());
                this.storeFileMap.put(filePosition, new StoreFileImpl<T>(filePosition, this.base, this.fileHeaderSize, this.serializer, this.bufferPool, this.fileDataSize, this.loadOnRead));
            }
        }
        while (!this.storeFileMap.isEmpty() && this.storeFileMap.lastEntry().getValue().file().length() <= (long)this.fileHeaderSize) {
            Map.Entry<Long, StoreFile<T>> lastEntry = this.storeFileMap.pollLastEntry();
            StoreFile<T> storeFile = lastEntry.getValue();
            File file3 = storeFile.file();
            if (!file3.exists()) continue;
            if (file3.delete()) {
                this.logger.info("Store file deleted: {}.", (Object)file3.getAbsolutePath());
                continue;
            }
            throw new IOException(String.format("Delete file %s failed!", file3.getAbsolutePath()));
        }
        if (!this.storeFileMap.isEmpty()) {
            long position = (Long)this.storeFileMap.firstKey();
            for (Map.Entry fileEntry : this.storeFileMap.entrySet()) {
                if (position != (Long)fileEntry.getKey()) {
                    throw new CorruptedLogException(String.format("Files are not continuous! expect: %d, actual file name: %d, store: %s.", position, fileEntry.getKey(), this.base.getAbsolutePath()));
                }
                position += ((StoreFile)fileEntry.getValue()).file().length() - (long)this.fileHeaderSize;
            }
        }
    }

    private long toLogTail(long position) {
        Object t = null;
        long seekEndPosition = Math.max(position - (long)(2 * this.maxMessageLength), this.left());
        while (position >= seekEndPosition) {
            try {
                t = this.tryRead(position--);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            if (null == t) continue;
            return position + 1L + (long)this.serializer.size(t);
        }
        return -1L;
    }

    public long toLogStart(long position) {
        Object t = null;
        long seekEndPosition = Math.max(position - (long)(2 * this.maxMessageLength), this.left());
        while (position >= seekEndPosition) {
            try {
                t = this.tryRead(position--);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            if (null == t) continue;
            return position + 1L;
        }
        return -1L;
    }

    public long appendByteBuffer(ByteBuffer byteBuffer) throws IOException {
        if (null == this.writeStoreFile) {
            this.writeStoreFile = this.createStoreFile(this.right());
        }
        if (this.writeStoreFile.size() - this.writeStoreFile.writePosition() < byteBuffer.remaining()) {
            this.writeStoreFile.closeWrite();
            this.writeStoreFile = this.createStoreFile(this.right());
        }
        return this.rightPosition.addAndGet(this.writeStoreFile.appendByteBuffer(byteBuffer));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long append(T t) throws IOException {
        try {
            this.writeLock.lock();
            if (null == this.writeStoreFile) {
                this.writeStoreFile = this.createStoreFile(this.right());
            }
            if (this.writeStoreFile.size() - this.writeStoreFile.writePosition() < this.serializer.size(t)) {
                this.writeStoreFile.closeWrite();
                this.writeStoreFile = this.createStoreFile(this.right());
            }
            long l = this.rightPosition.addAndGet(this.writeStoreFile.append(t));
            return l;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long append(List<T> ts) throws IOException {
        try {
            this.writeLock.lock();
            if (null == ts || ts.isEmpty()) {
                throw new WriteException("Parameter list is empty!");
            }
            long position = 0L;
            for (T t : ts) {
                position = this.append(t);
            }
            long l = position;
            return l;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean flush() throws IOException {
        if (this.flushPosition() < this.right()) {
            try {
                Map.Entry<Long, StoreFile<T>> entry;
                this.flushLock.lock();
                if (this.flushPosition() < this.left()) {
                    this.flushPosition.set(this.left());
                }
                if (null == (entry = this.storeFileMap.floorEntry(this.flushPosition()))) {
                    if (this.storeFileMap.isEmpty()) {
                        this.flushPosition.set(this.right());
                    } else {
                        this.flushPosition.set((Long)this.storeFileMap.firstKey());
                    }
                    boolean bl = true;
                    return bl;
                }
                StoreFile<T> storeFile = entry.getValue();
                if (!storeFile.isClean()) {
                    Map.Entry<Long, StoreFile<T>> prevEntry;
                    if (storeFile.flushPosition() == 0 && null != (prevEntry = this.storeFileMap.floorEntry(entry.getKey() - 1L))) {
                        prevEntry.getValue().force();
                    }
                    storeFile.flush();
                }
                if (this.flushPosition() < storeFile.position() + (long)storeFile.flushPosition()) {
                    this.flushPosition.set(storeFile.position() + (long)storeFile.flushPosition());
                }
                boolean bl = true;
                return bl;
            }
            finally {
                this.flushLock.unlock();
            }
        }
        return false;
    }

    private StoreFile<T> createStoreFile(long position) {
        StoreFile<T> storeFile = new StoreFileImpl<T>(position, this.base, this.fileHeaderSize, this.serializer, this.bufferPool, this.fileDataSize, this.loadOnRead);
        StoreFile present = this.storeFileMap.putIfAbsent(position, storeFile);
        if (present != null) {
            storeFile = present;
        }
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Store file created, leftPosition: {}, rightPosition: {}, flushPosition: {}, base: {}.", new Object[]{Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), this.base.getAbsolutePath()});
        }
        return storeFile;
    }

    public boolean isDiskFull() {
        return (this.base.getTotalSpace() - this.base.getFreeSpace()) * 100L > this.base.getTotalSpace() * (long)this.diskFullRatio;
    }

    public T read(long position) throws IOException {
        this.checkReadPosition(position);
        try {
            return this.tryRead(position);
        }
        catch (Throwable t) {
            this.logger.warn("Exception on read position {} of store {}.", new Object[]{position, this.base.getAbsolutePath(), t});
            throw t;
        }
    }

    public T read(long position, int length) throws IOException {
        this.checkReadPosition(position);
        try {
            StoreFile<T> storeFile = this.storeFileMap.floorEntry(position).getValue();
            int relPosition = (int)(position - storeFile.position());
            return storeFile.read(relPosition, length);
        }
        catch (Throwable t) {
            this.logger.warn("Exception on readByteBuffer position {} of store {}, leftPosition: {}, rightPosition: {}, flushPosition: {}.", new Object[]{position, this.base.getAbsolutePath(), Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), t});
            throw t;
        }
    }

    private T tryRead(long position) throws IOException {
        this.checkReadPosition(position);
        StoreFile<T> storeFile = this.storeFileMap.floorEntry(position).getValue();
        int relPosition = (int)(position - storeFile.position());
        return storeFile.read(relPosition, -1);
    }

    public List<T> batchRead(long position, int count) throws IOException {
        long pointer;
        this.checkReadPosition(position);
        ArrayList<T> list = new ArrayList<T>(count);
        StoreFile<T> storeFile = null;
        try {
            T t;
            for (pointer = position; list.size() < count && pointer < this.right(); pointer += (long)this.serializer.size(t)) {
                if (null == storeFile || (long)storeFile.writePosition() + storeFile.position() <= pointer) {
                    storeFile = this.storeFileMap.floorEntry(pointer).getValue();
                }
                int relPosition = (int)(pointer - storeFile.position());
                t = storeFile.read(relPosition, -1);
                list.add(t);
            }
            return list;
        }
        catch (Throwable t) {
            this.logger.warn("Exception on batchRead position {} of store {}, leftPosition: {}, rightPosition: {}, flushPosition: {}.", new Object[]{pointer, this.base.getAbsolutePath(), Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), t});
            throw t;
        }
    }

    public ByteBuffer readByteBuffer(long position, int length) throws IOException {
        this.checkReadPosition(position);
        try {
            StoreFile<T> storeFile = this.storeFileMap.floorEntry(position).getValue();
            int relPosition = (int)(position - storeFile.position());
            ByteBuffer byteBuffer = storeFile.readByteBuffer(relPosition, length);
            byteBuffer.limit(byteBuffer.position() + this.serializer.trim(byteBuffer, length));
            return byteBuffer;
        }
        catch (Throwable t) {
            this.logger.warn("Exception on readByteBuffer position {} of store {}, leftPosition: {}, rightPosition: {}, flushPosition: {}.", new Object[]{position, this.base.getAbsolutePath(), Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), t});
            throw t;
        }
    }

    private void checkReadPosition(long position) {
        long p = this.left();
        if (p > position) {
            throw new PositionUnderflowException(position, p);
        }
        p = this.right();
        if (position >= p) {
            throw new PositionOverflowException(position, p);
        }
    }

    public long physicalSize() {
        return this.right() - this.left();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long physicalDeleteTo(long position) throws IOException {
        long deleteSize = 0L;
        try {
            StoreFile<T> storeFile;
            long fileDataSize;
            Map.Entry<Long, StoreFile<T>> entry;
            this.deleteLock.lock();
            while (this.storeFileMap.size() > 0 && (entry = this.storeFileMap.firstEntry()) != null && entry.getKey() + (fileDataSize = (storeFile = entry.getValue()).hasPage() ? (long)storeFile.writePosition() : (long)storeFile.fileDataSize()) <= position) {
                if (storeFile == this.writeStoreFile) {
                    this.deleteLock.unlock();
                    try {
                        this.flushLock.lock();
                        this.writeLock.lock();
                        this.deleteLock.lock();
                        if (this.storeFileMap.remove(entry.getKey(), storeFile)) {
                            this.writeStoreFile = null;
                            this.forceDeleteStoreFile(storeFile);
                            deleteSize += fileDataSize;
                            this.leftPosition.addAndGet(fileDataSize);
                            if (this.flushPosition.get() >= this.leftPosition.get()) continue;
                            this.flushPosition.set(this.leftPosition.get());
                            continue;
                        }
                        break;
                    }
                    finally {
                        this.deleteLock.unlock();
                        this.writeLock.unlock();
                        this.flushLock.unlock();
                        continue;
                    }
                }
                if (this.storeFileMap.remove(entry.getKey(), storeFile)) {
                    this.leftPosition.addAndGet(fileDataSize);
                    this.forceDeleteStoreFile(storeFile);
                    deleteSize += fileDataSize;
                    continue;
                }
                break;
            }
        }
        finally {
            if (this.deleteLock.isHeldByCurrentThread()) {
                this.deleteLock.unlock();
            }
        }
        return deleteSize;
    }

    public boolean isClean() {
        return this.flushPosition() == this.right();
    }

    public long physicalDeleteLeftFile() throws IOException {
        if (this.storeFileMap.isEmpty()) {
            return 0L;
        }
        StoreFile<T> storeFile = this.storeFileMap.firstEntry().getValue();
        return this.physicalDeleteTo(storeFile.position() + (long)(storeFile.hasPage() ? storeFile.writePosition() : storeFile.fileDataSize()));
    }

    private void forceDeleteStoreFile(StoreFile storeFile) throws IOException {
        storeFile.forceUnload();
        File file = storeFile.file();
        if (file.exists()) {
            if (file.delete()) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("Store file deleted, leftPosition: {}, rightPosition: {}, flushPosition: {}, store: {}.", new Object[]{Format.formatWithComma((long)this.left()), Format.formatWithComma((long)this.right()), Format.formatWithComma((long)this.flushPosition()), file.getAbsolutePath()});
                }
            } else {
                throw new IOException(String.format("Delete file %s failed!", file.getAbsolutePath()));
            }
        }
    }

    public long position(long position, int offsetCount) {
        long t1;
        long t0 = SystemClock.now();
        int offset = 0;
        long pos = position;
        if (pos < this.left()) {
            pos = this.left();
        } else if (pos > this.right()) {
            pos = this.right();
        } else if (this.left() < pos && pos < this.right()) {
            pos = this.toLogStart(position);
        }
        if (offsetCount > 0) {
            while (offset < offsetCount && pos < this.right()) {
                pos = this.toLogTail(pos);
                ++offset;
            }
        } else if (offsetCount < 0) {
            while (offset > offsetCount && pos > this.left()) {
                pos = this.toLogStart(pos - 1L);
                --offset;
            }
        }
        if ((t1 = SystemClock.now()) - t0 > 300L) {
            this.logger.info("Find log position takes: {}ms, store: {}", (Object)(t1 - t0), (Object)this.base.getAbsolutePath());
        }
        return pos;
    }

    public File base() {
        return this.base;
    }

    @Override
    public void close() {
        for (StoreFile storeFile : this.storeFileMap.values()) {
            storeFile.unload();
        }
    }

    public byte[] readBytes(long position, int length) throws IOException {
        this.checkReadPosition(position);
        StoreFile<T> storeFile = this.storeFileMap.floorEntry(position).getValue();
        int relPosition = (int)(position - storeFile.position());
        return storeFile.readByteBuffer(relPosition, length).array();
    }

    public int fileCount() {
        return this.storeFileMap.size();
    }

    public int meetMinStoreFile(long minIndexedPhysicalPosition) {
        return this.storeFileMap.headMap(minIndexedPhysicalPosition).size();
    }

    public static class Config {
        public static final int DEFAULT_FILE_HEADER_SIZE = 128;
        public static final int DEFAULT_FILE_DATA_SIZE = 0x8000000;
        public static final int DEFAULT_DISK_FULL_RATIO = 90;
        public static final int DEFAULT_MAX_MESSAGE_LENGTH = 0x400000;
        public static final boolean DEFAULT_LOAD_ON_READ = false;
        private final int fileHeaderSize;
        private final int fileDataSize;
        private final int diskFullRatio;
        private final int maxMessageLength;
        private final boolean loadOnRead;

        public Config() {
            this(0x8000000, 128);
        }

        public Config(int fileDataSize) {
            this(fileDataSize, 128);
        }

        public Config(int fileDataSize, int fileHeaderSize) {
            this(fileDataSize, fileHeaderSize, 90);
        }

        public Config(int fileDataSize, int fileHeaderSize, int diskFullRatio) {
            this(fileDataSize, fileHeaderSize, diskFullRatio, 0x400000, false);
        }

        public Config(int fileDataSize, int fileHeaderSize, int diskFullRatio, int maxMessageLength, boolean loadOnRead) {
            this.fileDataSize = fileDataSize;
            this.fileHeaderSize = fileHeaderSize;
            this.diskFullRatio = diskFullRatio;
            this.maxMessageLength = maxMessageLength;
            this.loadOnRead = loadOnRead;
        }

        public Config(int fileDataSize, boolean loadOnRead) {
            this(fileDataSize, 128, 90, 0x400000, loadOnRead);
        }

        public Config(int messageFileSize, int fileHeaderSize, int diskFullRatio, int maxMessageLength) {
            this(messageFileSize, fileHeaderSize, diskFullRatio, maxMessageLength, false);
        }
    }
}

