/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.store.utils;

import com.sun.management.OperatingSystemMXBean;
import java.lang.management.ManagementFactory;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.joyqueue.monitor.BufferPoolMonitorInfo;
import org.joyqueue.store.utils.BufferHolder;
import org.joyqueue.toolkit.concurrent.LoopThread;
import org.joyqueue.toolkit.format.Format;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.Cleaner;
import sun.misc.VM;
import sun.nio.ch.DirectBuffer;

public class PreloadBufferPool {
    private static final long INTERVAL_MS = 50L;
    private static final Logger logger = LoggerFactory.getLogger(PreloadBufferPool.class);
    private static final double CACHE_RATIO = 0.9;
    private static final double EVICT_RATIO = 0.9;
    private static final double CORE_RATIO = 0.8;
    private static final long DEFAULT_WRITE_PAGE_EXTRA_WEIGHT_MS = 60000L;
    public static final String PRINT_METRIC_INTERVAL_MS_KEY = "PreloadBufferPool.PrintMetricIntervalMs";
    public static final String MAX_MEMORY_KEY = "PreloadBufferPool.MaxMemory";
    private static final String WRITE_PAGE_EXTRA_WEIGHT_MS_KEY = "PreloadBufferPool.WritePageExtraWeightMs";
    private final LoopThread preloadThread;
    private final LoopThread metricThread;
    private final LoopThread evictThread;
    private final long maxMemorySize;
    private final long coreMemorySize;
    private final long evictMemorySize;
    private final long writePageExtraWeightMs;
    private final AtomicLong usedSize = new AtomicLong(0L);
    private final Set<BufferHolder> directBufferHolders = ConcurrentHashMap.newKeySet();
    private final Set<BufferHolder> mMapBufferHolders = ConcurrentHashMap.newKeySet();
    private final Map<Integer, PreLoadCache> bufferCache = new ConcurrentHashMap<Integer, PreLoadCache>();
    private static PreloadBufferPool instance = null;

    public static PreloadBufferPool getInstance() {
        if (null == instance) {
            instance = new PreloadBufferPool();
            Runtime.getRuntime().addShutdownHook(new Thread(instance::close));
        }
        return instance;
    }

    private PreloadBufferPool() {
        long printMetricInterval = Long.parseLong(System.getProperty(PRINT_METRIC_INTERVAL_MS_KEY, "30000"));
        this.maxMemorySize = this.getMaxMemorySize();
        this.evictMemorySize = Math.round((double)this.maxMemorySize * 0.9);
        this.coreMemorySize = Math.round((double)this.maxMemorySize * 0.8);
        this.writePageExtraWeightMs = Long.parseLong(System.getProperty(WRITE_PAGE_EXTRA_WEIGHT_MS_KEY, String.valueOf(60000L)));
        this.preloadThread = this.buildPreloadThread();
        this.preloadThread.start();
        if (printMetricInterval > 0L) {
            this.metricThread = this.buildMetricThread(printMetricInterval);
            this.metricThread.start();
        } else {
            this.metricThread = null;
        }
        this.evictThread = this.buildEvictThread();
        this.evictThread.start();
        logger.info("Max direct memory: {}, core direct memory: {}, evict direct memory: {}.", new Object[]{Format.formatSize((long)this.maxMemorySize), Format.formatSize((long)this.coreMemorySize), Format.formatSize((long)this.evictMemorySize)});
    }

    private long getMaxMemorySize() {
        String mmsString = System.getProperty(MAX_MEMORY_KEY);
        int pct = Format.getPercentage((String)mmsString);
        if (pct > 0 && pct < 100) {
            long mms;
            long physicalMemorySize = PreloadBufferPool.getPhysicalMemorySize();
            long reservedHeapMemorySize = Runtime.getRuntime().maxMemory();
            if (Long.MAX_VALUE == reservedHeapMemorySize) {
                logger.warn("Runtime.getRuntime().maxMemory() returns unlimited!");
                reservedHeapMemorySize = physicalMemorySize / 2L;
            }
            if ((mms = physicalMemorySize * (long)pct / 100L - reservedHeapMemorySize) > 0L) {
                return mms;
            }
            return Math.round((double)VM.maxDirectMemory() * 0.9);
        }
        return Format.parseSize((String)System.getProperty(MAX_MEMORY_KEY), (long)Math.round((double)VM.maxDirectMemory() * 0.9));
    }

    private LoopThread buildMetricThread(long printMetricInterval) {
        return LoopThread.builder().name("DirectBufferPrintThread").sleepTime(printMetricInterval, printMetricInterval).doWork(this::printMetric).daemon(true).build();
    }

    private void printMetric() {
        long totalUsed = this.usedSize.get();
        long plUsed = this.bufferCache.values().stream().mapToLong(preLoadCache -> {
            long cached = preLoadCache.cache.size();
            long usedPreLoad = preLoadCache.onFlyCounter.get();
            long totalSize = (long)preLoadCache.bufferSize * (cached + usedPreLoad);
            logger.info("PreloadCache usage: cached: {} * {} = {}, used: {} * {} = {}, total: {}", new Object[]{Format.formatSize((long)preLoadCache.bufferSize), cached, Format.formatSize((long)((long)preLoadCache.bufferSize * cached)), Format.formatSize((long)preLoadCache.bufferSize), usedPreLoad, Format.formatSize((long)((long)preLoadCache.bufferSize * usedPreLoad)), Format.formatSize((long)totalSize)});
            return totalSize;
        }).sum();
        long mmpUsed = this.mMapBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        long directUsed = this.directBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        logger.info("Direct memory usage: preload/direct/mmp/used/max: {}/{}/{}/{}/{}.", new Object[]{Format.formatSize((long)plUsed), Format.formatSize((long)directUsed), Format.formatSize((long)mmpUsed), Format.formatSize((long)totalUsed), Format.formatSize((long)this.maxMemorySize)});
    }

    private LoopThread buildPreloadThread() {
        return LoopThread.builder().name("PreloadBufferPoolThread").sleepTime(50L, 50L).doWork(this::preLoadBuffer).onException(e -> logger.warn("PreloadBufferPoolThread exception:", e)).daemon(true).build();
    }

    private LoopThread buildEvictThread() {
        return LoopThread.builder().name("EvictThread").sleepTime(50L, 50L).condition(this::needEviction).doWork(this::evict).onException(e -> logger.warn("EvictThread exception:", e)).daemon(true).build();
    }

    private void evict() {
        for (PreLoadCache preLoadCache : this.bufferCache.values()) {
            if (!this.needEviction()) break;
            while (preLoadCache.cache.size() > preLoadCache.maxCount && !this.needEviction()) {
                try {
                    this.destroyOne(preLoadCache.cache.remove());
                }
                catch (NoSuchElementException noSuchElementException) {}
            }
        }
        if (this.needEviction()) {
            List sorted = Stream.concat(this.directBufferHolders.stream(), this.mMapBufferHolders.stream()).filter(BufferHolder::isFree).map(bufferHolder -> new LruWrapper<BufferHolder>((BufferHolder)bufferHolder, bufferHolder.lastAccessTime(), bufferHolder.writable() ? this.writePageExtraWeightMs : 0L)).sorted(Comparator.comparing(rec$ -> ((LruWrapper)rec$).getWeight())).collect(Collectors.toList());
            while (this.needEviction() && !sorted.isEmpty()) {
                LruWrapper wrapper = (LruWrapper)sorted.remove(0);
                BufferHolder holder = (BufferHolder)wrapper.get();
                if (holder.lastAccessTime() != wrapper.getLastAccessTime()) continue;
                holder.evict();
            }
        }
    }

    private boolean needEviction() {
        return this.usedSize.get() > this.evictMemorySize;
    }

    private boolean isOutOfMemory() {
        return this.usedSize.get() > this.maxMemorySize;
    }

    private boolean isHungry() {
        return this.usedSize.get() < this.coreMemorySize;
    }

    public synchronized boolean addPreLoad(int bufferSize, int coreCount, int maxCount) {
        return this.bufferCache.putIfAbsent(bufferSize, new PreLoadCache(bufferSize, coreCount, maxCount)) == null;
    }

    private void close() {
        this.preloadThread.stop();
        this.evictThread.stop();
        if (this.metricThread != null) {
            this.metricThread.stop();
        }
        this.bufferCache.values().forEach(p -> {
            while (!p.cache.isEmpty()) {
                this.destroyOne(p.cache.remove());
            }
        });
    }

    private void destroyOne(ByteBuffer byteBuffer) {
        this.usedSize.getAndAdd(-1 * byteBuffer.capacity());
        this.releaseIfDirect(byteBuffer);
    }

    private void preLoadBuffer() {
        for (PreLoadCache preLoadCache : this.bufferCache.values()) {
            if (preLoadCache.cache.size() >= preLoadCache.coreCount) continue;
            if (this.isHungry()) {
                try {
                    while (preLoadCache.cache.size() < preLoadCache.coreCount && this.usedSize.get() + (long)preLoadCache.bufferSize < this.maxMemorySize) {
                        preLoadCache.cache.add(this.createOne(preLoadCache.bufferSize));
                    }
                    continue;
                }
                catch (OutOfMemoryError ignored) {
                    return;
                }
            }
            List outdated = this.directBufferHolders.stream().filter(b -> b.size() == preLoadCache.bufferSize).filter(BufferHolder::isFree).map(bufferHolder -> new LruWrapper<BufferHolder>((BufferHolder)bufferHolder, bufferHolder.lastAccessTime(), bufferHolder.writable() ? this.writePageExtraWeightMs : 0L)).sorted(Comparator.comparing(rec$ -> ((LruWrapper)rec$).getWeight())).collect(Collectors.toList());
            while (preLoadCache.cache.size() < preLoadCache.coreCount && !outdated.isEmpty()) {
                LruWrapper wrapper = (LruWrapper)outdated.remove(0);
                BufferHolder holder = (BufferHolder)wrapper.get();
                if (holder.lastAccessTime() != wrapper.getLastAccessTime()) continue;
                holder.evict();
            }
        }
    }

    private ByteBuffer createOne(int size) {
        this.reserveMemory(size);
        return ByteBuffer.allocateDirect(size);
    }

    private void reserveMemory(int size) {
        this.usedSize.addAndGet(size);
        try {
            PreLoadCache preLoadCache;
            while (this.isOutOfMemory() && null != (preLoadCache = (PreLoadCache)this.bufferCache.values().stream().filter(p -> p.cache.size() > 0).findAny().orElse(null))) {
                this.destroyOne(preLoadCache.cache.remove());
            }
            if (this.isOutOfMemory()) {
                this.evictThread.wakeup();
                for (int i = 0; i < 5 && this.isOutOfMemory(); ++i) {
                    try {
                        Thread.sleep(10L);
                        continue;
                    }
                    catch (InterruptedException e) {
                        logger.warn("Interrupted: ", (Throwable)e);
                    }
                }
                if (this.isOutOfMemory()) {
                    throw new OutOfMemoryError();
                }
            }
        }
        catch (Throwable t) {
            this.usedSize.getAndAdd(-1 * size);
        }
    }

    private void releaseIfDirect(ByteBuffer byteBuffer) {
        if (byteBuffer instanceof DirectBuffer) {
            try {
                Method getCleanerMethod = byteBuffer.getClass().getMethod("cleaner", new Class[0]);
                getCleanerMethod.setAccessible(true);
                Cleaner cleaner = (Cleaner)getCleanerMethod.invoke((Object)byteBuffer, new Object[0]);
                cleaner.clean();
            }
            catch (Exception e) {
                logger.warn("Exception: ", (Throwable)e);
            }
        }
    }

    public void allocateMMap(BufferHolder bufferHolder) {
        this.reserveMemory(bufferHolder.size());
        this.mMapBufferHolders.add(bufferHolder);
    }

    public ByteBuffer allocateDirect(BufferHolder bufferHolder) {
        ByteBuffer buffer = this.allocateDirect(bufferHolder.size());
        this.directBufferHolders.add(bufferHolder);
        return buffer;
    }

    private ByteBuffer allocateDirect(int bufferSize) {
        try {
            PreLoadCache preLoadCache = this.bufferCache.get(bufferSize);
            if (null != preLoadCache) {
                try {
                    ByteBuffer byteBuffer = preLoadCache.cache.remove();
                    preLoadCache.onFlyCounter.getAndIncrement();
                    return byteBuffer;
                }
                catch (NoSuchElementException e) {
                    logger.debug("Pool is empty, create ByteBuffer: {}", (Object)bufferSize);
                    ByteBuffer byteBuffer = this.createOne(bufferSize);
                    preLoadCache.onFlyCounter.getAndIncrement();
                    return byteBuffer;
                }
            }
            logger.warn("No cached buffer in pool, create ByteBuffer: {}", (Object)bufferSize);
            return this.createOne(bufferSize);
        }
        catch (OutOfMemoryError outOfMemoryError) {
            logger.debug("OOM: {}/{}.", (Object)Format.formatSize((long)this.usedSize.get()), (Object)Format.formatSize((long)this.maxMemorySize));
            throw outOfMemoryError;
        }
    }

    public void releaseDirect(ByteBuffer byteBuffer, BufferHolder bufferHolder) {
        this.directBufferHolders.remove(bufferHolder);
        int size = byteBuffer.capacity();
        PreLoadCache preLoadCache = this.bufferCache.get(size);
        if (null != preLoadCache) {
            if (this.needEviction() && preLoadCache.cache.size() >= preLoadCache.maxCount) {
                this.destroyOne(byteBuffer);
            } else {
                byteBuffer.clear();
                preLoadCache.cache.add(byteBuffer);
            }
            preLoadCache.onFlyCounter.getAndDecrement();
        } else {
            this.destroyOne(byteBuffer);
        }
    }

    public void releaseMMap(BufferHolder bufferHolder) {
        this.mMapBufferHolders.remove(bufferHolder);
        this.usedSize.getAndAdd(-1 * bufferHolder.size());
    }

    private static long getPhysicalMemorySize() {
        OperatingSystemMXBean os = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
        return os.getTotalPhysicalMemorySize();
    }

    public BufferPoolMonitorInfo monitorInfo() {
        BufferPoolMonitorInfo bufferPoolMonitorInfo = new BufferPoolMonitorInfo();
        ArrayList plMonitorInfos = new ArrayList();
        long totalUsed = this.usedSize.get();
        long plUsed = this.bufferCache.values().stream().mapToLong(preLoadCache -> {
            long cached = preLoadCache.cache.size();
            long usedPreLoad = preLoadCache.onFlyCounter.get();
            long totalSize = (long)preLoadCache.bufferSize * (cached + usedPreLoad);
            BufferPoolMonitorInfo.PLMonitorInfo plMonitorInfo = new BufferPoolMonitorInfo.PLMonitorInfo();
            plMonitorInfo.setBufferSize(Format.formatSize((long)preLoadCache.bufferSize));
            plMonitorInfo.setCached(Format.formatSize((long)((long)preLoadCache.bufferSize * cached)));
            plMonitorInfo.setUsedPreLoad(Format.formatSize((long)((long)preLoadCache.bufferSize * usedPreLoad)));
            plMonitorInfo.setTotalSize(Format.formatSize((long)totalSize));
            plMonitorInfos.add(plMonitorInfo);
            return totalSize;
        }).sum();
        long mmpUsed = this.mMapBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        long directUsed = this.directBufferHolders.stream().mapToLong(BufferHolder::size).sum();
        bufferPoolMonitorInfo.setPlMonitorInfos(plMonitorInfos);
        bufferPoolMonitorInfo.setPlUsed(Format.formatSize((long)plUsed));
        bufferPoolMonitorInfo.setUsed(Format.formatSize((long)totalUsed));
        bufferPoolMonitorInfo.setMaxMemorySize(Format.formatSize((long)this.maxMemorySize));
        bufferPoolMonitorInfo.setMmpUsed(Format.formatSize((long)mmpUsed));
        bufferPoolMonitorInfo.setDirectUsed(Format.formatSize((long)directUsed));
        return bufferPoolMonitorInfo;
    }

    private static class LruWrapper<V> {
        private final long lastAccessTime;
        private final long extraWeight;
        private final V t;

        LruWrapper(V t, long lastAccessTime, long extraWeight) {
            this.lastAccessTime = lastAccessTime;
            this.t = t;
            this.extraWeight = extraWeight;
        }

        private long getLastAccessTime() {
            return this.lastAccessTime;
        }

        private V get() {
            return this.t;
        }

        private long getWeight() {
            return this.lastAccessTime + this.extraWeight;
        }
    }

    static class PreLoadCache {
        final int bufferSize;
        final int coreCount;
        final int maxCount;
        final Queue<ByteBuffer> cache = new ConcurrentLinkedQueue<ByteBuffer>();
        final AtomicLong onFlyCounter = new AtomicLong(0L);

        PreLoadCache(int bufferSize, int coreCount, int maxCount) {
            this.bufferSize = bufferSize;
            this.coreCount = coreCount;
            this.maxCount = maxCount;
        }
    }
}

