/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.broker.archive;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.joyqueue.broker.Plugins;
import org.joyqueue.broker.archive.ArchiveConfig;
import org.joyqueue.broker.archive.ArchiveSerializer;
import org.joyqueue.broker.cluster.ClusterManager;
import org.joyqueue.exception.JoyQueueException;
import org.joyqueue.message.MessageLocation;
import org.joyqueue.monitor.PointTracer;
import org.joyqueue.network.session.Connection;
import org.joyqueue.server.archive.store.api.ArchiveStore;
import org.joyqueue.server.archive.store.model.ConsumeLog;
import org.joyqueue.toolkit.concurrent.LoopThread;
import org.joyqueue.toolkit.lang.Close;
import org.joyqueue.toolkit.lang.LifeCycle;
import org.joyqueue.toolkit.security.Md5;
import org.joyqueue.toolkit.service.Service;
import org.joyqueue.toolkit.time.SystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsumeArchiveService
extends Service {
    private static final Logger logger = LoggerFactory.getLogger(ConsumeArchiveService.class);
    private ArchiveMappedFileRepository repository;
    private ArchiveStore archiveStore;
    private ClusterManager clusterManager;
    private ArchiveConfig archiveConfig;
    private AtomicInteger readByteCounter;
    private LoopThread readConsumeLogThread;
    private LoopThread cleanConsumeLogFileThread;
    private PointTracer tracer;

    public ConsumeArchiveService(ArchiveConfig archiveConfig, ClusterManager clusterManager) {
        this.clusterManager = clusterManager;
        this.archiveConfig = archiveConfig;
    }

    protected void validate() throws Exception {
        super.validate();
        if (this.archiveStore == null) {
            this.archiveStore = (ArchiveStore)Plugins.ARCHIVESTORE.get();
        }
        this.archiveStore.setNameSpace(this.archiveConfig.getNamespace());
        logger.info("Get archive store namespace [{}] by archive config.", (Object)this.archiveConfig.getNamespace());
        Preconditions.checkArgument((this.archiveStore != null ? 1 : 0) != 0, (Object)"archive store can not be null.");
        this.repository = new ArchiveMappedFileRepository(this.archiveConfig.getArchivePath());
        this.readByteCounter = new AtomicInteger(0);
        this.tracer = (PointTracer)Plugins.TRACERERVICE.get((Object)this.archiveConfig.getTracerType());
        this.readConsumeLogThread = LoopThread.builder().sleepTime(1L, 10L).name("ReadAndPutHBase-ConsumeLog-Thread").onException(e -> {
            logger.error(e.getMessage(), e);
            this.repository.rollBack(this.readByteCounter.get());
            logger.info("finish rollback.");
        }).doWork(this::readAndWrite).build();
        this.cleanConsumeLogFileThread = LoopThread.builder().sleepTime(10000L, 10000L).name("CleanArchiveFile-ConsumeLog-Thread").onException(e -> logger.error(e.getMessage(), e)).doWork(this::cleanAndRollWriteFile).build();
    }

    protected void doStart() throws Exception {
        super.doStart();
        this.archiveStore.start();
        this.readConsumeLogThread.start();
        this.cleanConsumeLogFileThread.start();
    }

    protected void doStop() {
        super.doStop();
        Close.close((LifeCycle)this.readConsumeLogThread);
        Close.close((LifeCycle)this.cleanConsumeLogFileThread);
        Close.close((Closeable)this.repository);
        Close.close((LifeCycle)this.archiveStore);
    }

    private void readAndWrite() throws JoyQueueException, InterruptedException {
        int readBatchSize;
        int batchSize = this.archiveConfig.getConsumeBatchNum();
        do {
            List<ConsumeLog> list;
            if ((readBatchSize = (list = this.readConsumeLog(batchSize)).size()) > 0) {
                long startTime = SystemClock.now();
                this.archiveStore.putConsumeLog(list, this.tracer);
                long endTime = SystemClock.now();
                logger.debug("Write consumeLogs size:{} to archive store, and elapsed time {}ms", (Object)list.size(), (Object)(endTime - startTime));
                int consumeWriteDelay = this.archiveConfig.getConsumeWriteDelay();
                if (consumeWriteDelay <= 0) continue;
                Thread.sleep(consumeWriteDelay);
                continue;
            }
            if (this.repository.rFile != null && this.repository.rMap != null) {
                logger.debug("read file name {}, read position {}", (Object)this.repository.rFile.getName(), (Object)this.repository.rMap.toString());
                break;
            }
            logger.debug("read file is null.");
            break;
        } while (readBatchSize == batchSize);
    }

    private void cleanAndRollWriteFile() {
        this.repository.delArchivedFile();
        this.repository.tryFinishCurWriteFile();
    }

    private List<ConsumeLog> readConsumeLog(int count) {
        byte[] bytes;
        this.readByteCounter.set(0);
        LinkedList<ConsumeLog> list = new LinkedList<ConsumeLog>();
        for (int i = 0; i < count && (bytes = this.repository.readOne()).length > 0; ++i) {
            this.readByteCounter.addAndGet(5 + bytes.length);
            list.add(ArchiveSerializer.read(ByteBuffer.wrap(bytes)));
        }
        return list;
    }

    public long getRemainConsumeLogFileNum() {
        if (!this.archiveConfig.isReamingEnable()) {
            return 0L;
        }
        int localFileNum = this.repository.getLocalFileNum();
        return (long)localFileNum * this.repository.getPageSize();
    }

    public void appendConsumeLog(Connection connection, MessageLocation[] locations) throws JoyQueueException {
        if (!this.isStarted()) {
            logger.debug("ConsumeArchiveService not be started.");
            return;
        }
        List<ConsumeLog> logList = this.convert(connection, locations);
        logList.stream().forEach(log -> {
            ByteBuffer buffer = ArchiveSerializer.write(log);
            this.appendLog(buffer);
            ArchiveSerializer.release(buffer);
        });
    }

    private List<ConsumeLog> convert(Connection connection, MessageLocation[] locations) throws JoyQueueException {
        LinkedList<ConsumeLog> list = new LinkedList<ConsumeLog>();
        for (MessageLocation location : locations) {
            ConsumeLog log = new ConsumeLog();
            byte[] bytesMsgId = this.buildMessageId(location);
            log.setBytesMessageId(bytesMsgId);
            log.setApp(connection.getApp());
            log.setBrokerId(this.clusterManager.getBrokerId().intValue());
            log.setClientIp(connection.getAddress());
            log.setConsumeTime(SystemClock.now());
            list.add(log);
        }
        return list;
    }

    private byte[] buildMessageId(MessageLocation location) {
        String messageId = location.getTopic() + location.getPartition() + location.getIndex();
        byte[] messageIdBytes = new byte[]{};
        try {
            messageIdBytes = Md5.INSTANCE.encrypt(messageId.getBytes(), null);
        }
        catch (GeneralSecurityException e) {
            logger.error("topic:{}, partition:{}, index:{}, exception:{}", new Object[]{location.getTopic(), location.getPartition(), location.getIndex(), e});
        }
        return messageIdBytes;
    }

    private synchronized void appendLog(ByteBuffer buffer) {
        this.repository.append(buffer);
    }

    static class ArchiveMappedFileRepository
    implements Closeable {
        private String baseDir;
        private File rwFile;
        private MappedByteBuffer rwMap;
        private RandomAccessFile rwRaf;
        private FileChannel rwFileChannel;
        private File rFile;
        private MappedByteBuffer rMap;
        private RandomAccessFile rRaf;
        private FileChannel rFileChannel;
        private volatile long position = 0L;
        private final long pageSize = 0x1000000L;

        ArchiveMappedFileRepository(String baseDir) {
            this.baseDir = baseDir;
            this.recover();
        }

        private void recover() {
            File lastFile = this.LastFile();
            if (lastFile == null) {
                return;
            }
            this.rwFile = lastFile;
            try {
                this.rwRaf = new RandomAccessFile(this.rwFile, "rw");
                this.rwFileChannel = this.rwRaf.getChannel();
                this.rwMap = this.rwFileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, this.rwFile.length());
            }
            catch (IOException e) {
                logger.error(e.getMessage(), (Throwable)e);
            }
            while (this.checkStartFlag(this.rwMap)) {
                int len = this.rwMap.getInt();
                this.rwMap.get(new byte[len]);
            }
            if (this.checkFileEndFlag(this.rwMap)) {
                this.close();
                return;
            }
            this.position = this.rwMap.position();
        }

        public synchronized void append(ByteBuffer buffer) {
            this.position += (long)buffer.limit();
            if (this.rwMap == null) {
                this.newMappedRWFile();
                this.position = 0L;
                this.append(buffer);
            } else if (1L + this.position >= 0x1000000L) {
                this.rwMap.put((byte)127);
                this.rwMap = null;
                this.append(buffer);
            } else {
                if (buffer.limit() == 0) {
                    logger.debug("append buffer limit is zero.");
                    return;
                }
                this.rwMap.put((byte)-128);
                ++this.position;
                this.rwMap.put(buffer);
            }
        }

        public void newMappedRWFile() {
            try {
                if (this.rwFileChannel != null) {
                    this.rwFileChannel.close();
                }
                this.rwFileChannel = null;
                if (this.rwRaf != null) {
                    this.rwRaf.close();
                }
                this.rwRaf = null;
                this.rwFile = null;
                File parent = new File(this.baseDir);
                if (!parent.exists()) {
                    parent.mkdirs();
                }
                this.rwFile = FileUtils.getFile((String[])new String[]{this.baseDir, SystemClock.now() + ""});
                this.rwRaf = new RandomAccessFile(this.rwFile, "rw");
                this.rwFileChannel = this.rwRaf.getChannel();
                this.rwMap = this.rwFileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, 0x1000000L);
            }
            catch (Exception ex) {
                logger.error("create and mapped file error.", (Throwable)ex);
            }
        }

        private void mappedReadOnlyFile() {
            try {
                this.closeCurrentReadFile();
                this.rRaf = new RandomAccessFile(this.rFile, "r");
                this.rFileChannel = this.rRaf.getChannel();
                this.rMap = this.rFileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, 0x1000000L);
            }
            catch (IOException e) {
                logger.error(e.getMessage(), (Throwable)e);
            }
        }

        public byte[] readOne() {
            if (this.rMap == null) {
                if (this.nextFile() != null) {
                    this.mappedReadOnlyFile();
                } else {
                    return new byte[0];
                }
            }
            if (this.checkStartFlag(this.rMap)) {
                int msgLen = this.rMap.getInt();
                byte[] bytes = new byte[msgLen];
                this.rMap.get(bytes);
                return bytes;
            }
            if (this.checkFileEndFlag(this.rMap)) {
                logger.debug("Finish reading the file {}.{}", (Object)this.rFile, (Object)this.rMap.toString());
                if (this.nextFile() != null) {
                    this.mappedReadOnlyFile();
                    return this.readOne();
                }
            }
            return new byte[0];
        }

        private boolean checkStartFlag(MappedByteBuffer rMap) {
            if ((long)(rMap.position() + 1) < 0x1000000L) {
                if (rMap.get() == -128) {
                    return true;
                }
                rMap.position(rMap.position() - 1);
            }
            return false;
        }

        private boolean checkFileEndFlag(MappedByteBuffer rMap) {
            if ((long)(rMap.position() + 1) <= 0x1000000L) {
                if (rMap.get() == 127) {
                    rMap.position(rMap.position() - 1);
                    return true;
                }
                rMap.position(rMap.position() - 1);
            }
            return false;
        }

        private void rollBack(int interval) {
            if (this.rMap != null) {
                int position = this.rMap.position();
                int newPosition = position - interval;
                this.rMap.position(newPosition);
            }
        }

        @Override
        public void close() {
            try {
                this.closeCurrentReadFile();
                this.closeCurrentWriteFile();
            }
            catch (IOException e) {
                logger.error("delete read file error.", (Throwable)e);
            }
        }

        public void closeCurrentReadFile() throws IOException {
            if (this.rFileChannel != null) {
                this.rFileChannel.close();
            }
            if (this.rRaf != null) {
                this.rRaf.close();
            }
        }

        public void closeCurrentWriteFile() throws IOException {
            if (this.rwFileChannel != null) {
                this.rwFileChannel.close();
            }
            if (this.rwRaf != null) {
                this.rwRaf.close();
            }
        }

        private File nextFile() {
            File file = new File(this.baseDir);
            Object[] list = file.list();
            if (list == null || list.length == 1) {
                logger.debug("only one write file.");
                return null;
            }
            logger.debug("archive file list {}", (Object)Arrays.toString(list));
            String concurrentFileName = this.rFile == null ? "" : this.rFile.getName();
            List sorted = Arrays.asList(list).stream().filter(name -> name.compareTo(concurrentFileName) > 0).sorted(Comparator.naturalOrder()).collect(Collectors.toList());
            if (sorted.size() > 1) {
                File tempFile;
                String fileName = (String)sorted.get(0);
                this.rFile = tempFile = new File(this.baseDir + fileName);
                logger.debug("current read consume event file {}", (Object)tempFile);
                return this.rFile;
            }
            logger.debug("only one write file.");
            return null;
        }

        private File LastFile() {
            File file = new File(this.baseDir);
            String[] list = file.list();
            if (list == null) {
                return null;
            }
            Optional first = Arrays.asList(list).stream().sorted(Comparator.reverseOrder()).findFirst();
            if (first.isPresent()) {
                File tempFile;
                String fileName = (String)first.get();
                this.rwFile = tempFile = new File(this.baseDir + fileName);
                return this.rwFile;
            }
            return null;
        }

        private void delArchivedFile() {
            List<String> archivedFileList = this.getArchivedFileList();
            if (CollectionUtils.isNotEmpty(archivedFileList)) {
                archivedFileList.stream().forEach(fileName -> new File(this.baseDir + fileName).delete());
            }
        }

        private List<String> getArchivedFileList() {
            if (this.rFile == null) {
                logger.debug("Can not get archive file list cause by consume archive read file have no init.");
                return null;
            }
            File file = new File(this.baseDir);
            String[] list = file.list();
            if (list == null) {
                return null;
            }
            return Arrays.asList(list).stream().filter(name -> name.compareTo(this.rFile.getName()) < 0).collect(Collectors.toList());
        }

        public int getLocalFileNum() {
            File file = new File(this.baseDir);
            String[] list = file.list();
            if (list == null) {
                return 0;
            }
            return list.length;
        }

        public long getReadPosition() {
            return this.rMap.position();
        }

        public long getWritePosition() {
            return this.rwMap.position();
        }

        public long getPageSize() {
            return 0x1000000L;
        }

        public synchronized void tryFinishCurWriteFile() {
            if (this.rwFile == null) {
                return;
            }
            String name = this.rwFile.getName();
            long now = SystemClock.now();
            if (this.position > 0L && now - Long.parseLong(name) >= 60000L) {
                this.position = 0x1000000L;
                this.append(ByteBuffer.wrap(new byte[0]));
                logger.info("reset write file {} position {} to pageSize.", (Object)this.rwFile.getName(), (Object)this.rwMap.toString());
            }
        }
    }
}

