/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.broker.producer;

import com.google.common.base.Preconditions;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.joyqueue.broker.BrokerContext;
import org.joyqueue.broker.BrokerContextAware;
import org.joyqueue.broker.buffer.Serializer;
import org.joyqueue.broker.cluster.ClusterManager;
import org.joyqueue.broker.monitor.BrokerMonitor;
import org.joyqueue.broker.producer.Produce;
import org.joyqueue.broker.producer.ProduceConfig;
import org.joyqueue.broker.producer.PutResult;
import org.joyqueue.broker.producer.transaction.TransactionManager;
import org.joyqueue.domain.PartitionGroup;
import org.joyqueue.domain.QosLevel;
import org.joyqueue.domain.TopicName;
import org.joyqueue.exception.JoyQueueCode;
import org.joyqueue.exception.JoyQueueException;
import org.joyqueue.message.BrokerCommit;
import org.joyqueue.message.BrokerMessage;
import org.joyqueue.message.BrokerPrepare;
import org.joyqueue.message.BrokerRollback;
import org.joyqueue.message.JoyQueueLog;
import org.joyqueue.network.session.Producer;
import org.joyqueue.network.session.TransactionId;
import org.joyqueue.store.PartitionGroupStore;
import org.joyqueue.store.StoreService;
import org.joyqueue.store.WriteRequest;
import org.joyqueue.store.WriteResult;
import org.joyqueue.toolkit.concurrent.EventListener;
import org.joyqueue.toolkit.concurrent.LoopThread;
import org.joyqueue.toolkit.lang.Close;
import org.joyqueue.toolkit.lang.LifeCycle;
import org.joyqueue.toolkit.metric.Metric;
import org.joyqueue.toolkit.service.Service;
import org.joyqueue.toolkit.time.SystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProduceManager
extends Service
implements Produce,
BrokerContextAware {
    private static final Logger logger = LoggerFactory.getLogger(ProduceManager.class);
    private ProduceConfig config;
    private TransactionManager transactionManager;
    private ClusterManager clusterManager;
    private StoreService store;
    private BrokerMonitor brokerMonitor;
    private BrokerContext brokerContext;
    private Metric metrics = null;
    private Metric.MetricInstance metric = null;
    private LoopThread metricThread = null;

    public ProduceManager() {
    }

    public ProduceManager(ProduceConfig config, ClusterManager clusterManager, StoreService store, BrokerMonitor brokerMonitor) {
        this.config = config;
        this.clusterManager = clusterManager;
        this.store = store;
        this.brokerMonitor = brokerMonitor;
    }

    protected void doStart() throws Exception {
        super.doStart();
        this.transactionManager.start();
        if (null != this.metricThread) {
            this.metricThread.start();
        }
        logger.info("ProduceManager is started.");
    }

    protected void validate() throws Exception {
        super.validate();
        if (this.config == null) {
            this.config = new ProduceConfig(this.brokerContext == null ? null : this.brokerContext.getPropertySupplier());
        }
        if (this.store == null && this.brokerContext != null) {
            this.store = this.brokerContext.getStoreService();
        }
        if (this.clusterManager == null && this.brokerContext != null) {
            this.clusterManager = this.brokerContext.getClusterManager();
        }
        if (this.brokerMonitor == null && this.brokerContext != null) {
            this.brokerMonitor = this.brokerContext.getBrokerMonitor();
        }
        Preconditions.checkArgument((this.store != null ? 1 : 0) != 0, (Object)"store service can not be null");
        Preconditions.checkArgument((this.clusterManager != null ? 1 : 0) != 0, (Object)"cluster manager can not be null");
        if (this.brokerMonitor == null) {
            logger.warn("broker monitor is null.");
        }
        if (!this.clusterManager.isStarted()) {
            logger.warn("The clusterManager is not started, try to start it!");
            this.clusterManager.start();
        }
        this.transactionManager = new TransactionManager(this.config, this.store, this.clusterManager, this.brokerMonitor);
        if (this.config.getPrintMetricIntervalMs() > 0L) {
            this.metrics = new Metric("input", 1, new String[]{"callback", "async"}, new String[]{"tps"}, new String[]{"traffic"});
            this.metric = (Metric.MetricInstance)this.metrics.getMetricInstances().get(0);
            this.metricThread = LoopThread.builder().sleepTime(this.config.getPrintMetricIntervalMs(), this.config.getPrintMetricIntervalMs()).name("Metric-Thread").onException(e -> logger.warn("Exception:", e)).doWork(() -> this.metrics.reportAndReset()).build();
        }
    }

    protected void doStop() {
        super.doStop();
        Close.close((LifeCycle)this.transactionManager);
        if (null != this.metricThread) {
            this.metricThread.stop();
        }
        logger.info("ProduceManager is stopped.");
    }

    @Override
    public PutResult putMessage(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel) throws JoyQueueException {
        int timeout = this.clusterManager.getProducerPolicy(TopicName.parse((String)producer.getTopic()), producer.getApp()).getTimeOut();
        return this.putMessage(producer, msgs, qosLevel, timeout);
    }

    @Override
    public PutResult putMessage(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel, int timeout) throws JoyQueueException {
        long startWritePoint = SystemClock.now();
        long endTime = (long)timeout + startWritePoint;
        qosLevel = this.getConfigQosLevel(producer, qosLevel);
        String txId = msgs.get(0).getTxId();
        if (StringUtils.isNotEmpty((CharSequence)txId)) {
            return this.writeTxMessage(producer, msgs, txId, endTime);
        }
        return this.writeMessages(producer, msgs, qosLevel, endTime);
    }

    @Override
    public void putMessageAsync(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel, EventListener<WriteResult> eventListener) throws JoyQueueException {
        this.putMessageAsync(producer, msgs, qosLevel, this.clusterManager.getProducerPolicy(TopicName.parse((String)producer.getTopic()), producer.getApp()).getTimeOut(), eventListener);
    }

    @Override
    public void putMessageAsync(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel, int timeout, EventListener<WriteResult> eventListener) throws JoyQueueException {
        long startWritePoint = SystemClock.now();
        long endTime = (long)timeout + startWritePoint;
        String txId = msgs.get(0).getTxId();
        qosLevel = this.getConfigQosLevel(producer, qosLevel);
        if (StringUtils.isNotEmpty((CharSequence)txId)) {
            this.writeTxMessageAsync(producer, msgs, txId, timeout, eventListener);
        } else {
            this.writeMessagesAsync(producer, msgs, qosLevel, endTime, eventListener);
        }
    }

    private PutResult writeTxMessage(Producer producer, List<BrokerMessage> msgs, String txId, long endTime) throws JoyQueueException {
        ByteBuffer[] byteBuffers = this.generateRByteBufferList(msgs);
        Future<WriteResult> writeResultFuture = this.transactionManager.putMessage(producer, txId, byteBuffers);
        WriteResult writeResult = this.syncWait(writeResultFuture, endTime - SystemClock.now());
        PutResult putResult = new PutResult();
        putResult.addWriteResult(msgs.get(0).getPartition(), writeResult);
        return putResult;
    }

    private void writeTxMessageAsync(Producer producer, List<BrokerMessage> msgs, String txId, long timeout, EventListener<WriteResult> eventListener) throws JoyQueueException {
        ByteBuffer[] byteBuffers = this.generateRByteBufferList(msgs);
        try {
            WriteResult writeResult = this.transactionManager.putMessage(producer, txId, byteBuffers).get(timeout, TimeUnit.MILLISECONDS);
            eventListener.onEvent((Object)writeResult);
        }
        catch (Exception e) {
            logger.error("writeTxMessageAsync exception, producer: {}", (Object)producer, (Object)e);
            eventListener.onEvent((Object)new WriteResult(JoyQueueCode.CN_UNKNOWN_ERROR, ArrayUtils.EMPTY_LONG_ARRAY));
        }
    }

    private PutResult writeMessages(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel, long endTime) throws JoyQueueException {
        PutResult putResult = new PutResult();
        String topic = producer.getTopic();
        List<Short> partitions = this.clusterManager.getLocalPartitions(TopicName.parse((String)topic));
        if (partitions == null || partitions.size() == 0) {
            logger.error("no partitions available topic:%s", (Object)topic);
            throw new JoyQueueException(JoyQueueCode.CN_NO_PERMISSION, new Object[0]);
        }
        long startTime = SystemClock.now();
        Map<PartitionGroup, List<WriteRequest>> dispatchedMsgs = this.dispatchPartition(msgs, partitions);
        for (Map.Entry<PartitionGroup, List<WriteRequest>> dispatchEntry : dispatchedMsgs.entrySet()) {
            PartitionGroup partitionGroup = dispatchEntry.getKey();
            List<WriteRequest> writeRequests = dispatchEntry.getValue();
            PartitionGroupStore partitionStore = this.store.getStore(topic, partitionGroup.getGroup(), qosLevel);
            Future writeResultFuture = partitionStore.asyncWrite(writeRequests.toArray(new WriteRequest[0]));
            WriteResult writeResult = this.syncWait(writeResultFuture, endTime - SystemClock.now());
            if (writeResult.getCode().equals((Object)JoyQueueCode.SUCCESS)) {
                this.onPutMessage(topic, producer.getApp(), partitionGroup.getGroup(), startTime, writeRequests);
            }
            putResult.addWriteResult((short)partitionGroup.getGroup(), writeResult);
            if (!this.config.getLogDetail(producer.getApp())) continue;
            logger.info("writeMessages, topic: {}, app: {}, partitionGroup: {}, qosLevel: {}, size: {}, result: {}", new Object[]{producer.getTopic(), producer.getApp(), partitionGroup.getGroup(), qosLevel, writeRequests.size(), writeResult.getCode()});
        }
        return putResult;
    }

    private void writeMessagesAsync(Producer producer, List<BrokerMessage> msgs, QosLevel qosLevel, long endTime, EventListener<WriteResult> eventListener) throws JoyQueueException {
        String topic = producer.getTopic();
        String app = producer.getApp();
        List<Short> partitions = this.clusterManager.getLocalPartitions(TopicName.parse((String)topic));
        if (partitions == null || partitions.size() == 0) {
            logger.error("no partitions available topic:%s", (Object)topic);
            throw new JoyQueueException(JoyQueueCode.CN_NO_PERMISSION, new Object[0]);
        }
        Map<PartitionGroup, List<WriteRequest>> dispatchedMsgs = this.dispatchPartition(msgs, partitions);
        for (Map.Entry<PartitionGroup, List<WriteRequest>> dispatchEntry : dispatchedMsgs.entrySet()) {
            PartitionGroup partitionGroup = dispatchEntry.getKey();
            if (logger.isDebugEnabled()) {
                logger.debug("ProduceManager writeMessageAsync topic:[{}], partitionGroup:[{}]]", (Object)topic, (Object)partitionGroup);
            }
            List<WriteRequest> writeRequests = dispatchEntry.getValue();
            PartitionGroupStore partitionStore = this.store.getStore(topic, partitionGroup.getGroup(), qosLevel);
            long startTime = SystemClock.now();
            if (null != this.metric) {
                long t0 = System.nanoTime();
                partitionStore.asyncWrite((EventListener)new MetricEventListener(t0, startTime, this.metric, eventListener, topic, app, partitionGroup.getGroup(), writeRequests), writeRequests.toArray(new WriteRequest[0]));
                long t1 = System.nanoTime();
                this.metric.addCounter("tps", writeRequests.stream().map(WriteRequest::getBuffer).count());
                this.metric.addTraffic("traffic", (long)writeRequests.stream().map(WriteRequest::getBuffer).mapToInt(Buffer::remaining).sum());
                this.metric.addLatency("async", t1 - t0);
            } else {
                partitionStore.asyncWrite(event -> {
                    if (event.getCode().equals((Object)JoyQueueCode.SUCCESS)) {
                        this.onPutMessage(topic, app, partitionGroup.getGroup(), startTime, writeRequests);
                    }
                    if (this.config.getLogDetail(producer.getApp())) {
                        logger.info("writeMessagesAsync, topic: {}, app: {}, partitionGroup: {}, qosLevel: {}, size: {}, result: {}", new Object[]{producer.getTopic(), producer.getApp(), partitionGroup.getGroup(), qosLevel, writeRequests.size(), event.getCode()});
                    }
                    eventListener.onEvent(event);
                }, writeRequests.toArray(new WriteRequest[0]));
            }
            if (!qosLevel.equals((Object)QosLevel.ONE_WAY)) continue;
            this.onPutMessage(topic, app, partitionGroup.getGroup(), startTime, writeRequests);
        }
    }

    protected void onPutMessage(String topic, String app, int partitionGroup, long startTime, List<WriteRequest> writeRequests) {
        long now = SystemClock.now();
        writeRequests.forEach(writeRequest -> this.brokerMonitor.onPutMessage(topic, app, partitionGroup, writeRequest.getPartition(), writeRequest.getBatchSize(), writeRequest.getBuffer().limit(), now - startTime));
    }

    protected QosLevel getConfigQosLevel(Producer producer, QosLevel qosLevel) {
        if (this.config.getTopicQosLevel(producer.getTopic()) != -1) {
            return QosLevel.valueOf((int)this.config.getTopicQosLevel(producer.getTopic()));
        }
        if (this.config.getAppQosLevel(producer.getApp()) != -1) {
            return QosLevel.valueOf((int)this.config.getAppQosLevel(producer.getApp()));
        }
        if (this.config.getBrokerQosLevel() != -1) {
            return QosLevel.valueOf((int)this.config.getBrokerQosLevel());
        }
        return qosLevel;
    }

    private WriteResult syncWait(Future<WriteResult> writeResultFuture, long timeout) throws JoyQueueException {
        try {
            return writeResultFuture.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException | ExecutionException e) {
            logger.error("Write message error", (Throwable)e);
            throw new JoyQueueException(JoyQueueCode.CN_THREAD_INTERRUPTED, new Object[0]);
        }
        catch (TimeoutException e) {
            throw new JoyQueueException(JoyQueueCode.SE_WRITE_TIMEOUT, new Object[0]);
        }
    }

    private Map<PartitionGroup, List<WriteRequest>> dispatchPartition(List<BrokerMessage> messageList, List<Short> partitionList) throws JoyQueueException {
        int index = (int)Math.floor(Math.random() * (double)partitionList.size());
        short partition = partitionList.get(index);
        int partitionSize = partitionList.size();
        HashMap<PartitionGroup, List<WriteRequest>> resultMap = new HashMap<PartitionGroup, List<WriteRequest>>();
        for (BrokerMessage msg : messageList) {
            short writePartition = this.selectPartition(partition, msg, partitionList, partitionSize);
            PartitionGroup writePartitionGroup = this.clusterManager.getPartitionGroup(TopicName.parse((String)msg.getTopic()), writePartition);
            ArrayList<WriteRequest> writeRequestList = (ArrayList<WriteRequest>)resultMap.get(writePartitionGroup);
            if (writeRequestList == null) {
                writeRequestList = new ArrayList<WriteRequest>();
                resultMap.put(writePartitionGroup, writeRequestList);
            }
            short batchCount = 1;
            if (msg.isBatch()) {
                batchCount = msg.getFlag();
            }
            writeRequestList.add(new WriteRequest(writePartition, this.convertBrokerMessage2RByteBuffer(msg), (int)batchCount));
        }
        return resultMap;
    }

    private short selectPartition(short defaultPartition, BrokerMessage msg, List<Short> partitionList, int partitionSize) {
        if (msg.getPartition() >= 0) {
            return msg.getPartition();
        }
        short writePartition = defaultPartition;
        if (msg.isOrdered()) {
            String businessId = msg.getBusinessId();
            if (StringUtils.isEmpty((CharSequence)businessId)) {
                writePartition = partitionList.get(0);
            } else {
                int hashCode = businessId.hashCode();
                hashCode = hashCode > Integer.MIN_VALUE ? hashCode : -2147483647;
                int orderIndex = Math.abs(hashCode) % partitionSize;
                writePartition = partitionList.get(orderIndex);
            }
        }
        return writePartition;
    }

    private ByteBuffer convertBrokerMessage2RByteBuffer(BrokerMessage brokerMessage) throws JoyQueueException {
        int msgSize = Serializer.sizeOf(brokerMessage);
        ByteBuffer allocate = ByteBuffer.allocate(msgSize);
        try {
            Serializer.write(brokerMessage, allocate, msgSize);
        }
        catch (Exception e) {
            logger.error("Serialize message error! topic:{},app:{}", new Object[]{brokerMessage.getTopic(), brokerMessage.getApp(), e});
            throw new JoyQueueException(JoyQueueCode.SE_SERIALIZER_ERROR, new Object[0]);
        }
        return allocate;
    }

    private ByteBuffer[] generateRByteBufferList(List<BrokerMessage> msgs) throws JoyQueueException {
        int size = msgs.size();
        ByteBuffer[] byteBuffers = new ByteBuffer[size];
        for (int i = 0; i < size; ++i) {
            ByteBuffer byteBuffer;
            BrokerMessage message = msgs.get(i);
            byteBuffers[i] = byteBuffer = this.convertBrokerMessage2RByteBuffer(message);
        }
        return byteBuffers;
    }

    @Override
    public TransactionId putTransactionMessage(Producer producer, JoyQueueLog tx) throws JoyQueueException {
        if (tx.getType() == 3) {
            return this.transactionManager.prepare(producer, (BrokerPrepare)tx);
        }
        if (tx.getType() == 4) {
            return this.transactionManager.commit(producer, (BrokerCommit)tx);
        }
        if (tx.getType() == 5) {
            return this.transactionManager.rollback(producer, (BrokerRollback)tx);
        }
        throw new JoyQueueException(JoyQueueCode.CN_COMMAND_UNSUPPORTED, new Object[0]);
    }

    @Override
    public TransactionId getTransaction(Producer producer, String txId) {
        return this.transactionManager.getTransaction(producer.getTopic(), producer.getApp(), txId);
    }

    @Override
    public List<TransactionId> getFeedback(Producer producer, int count) {
        return this.transactionManager.getFeedback(producer, count);
    }

    @Override
    public void setBrokerContext(BrokerContext brokerContext) {
        this.brokerContext = brokerContext;
    }

    class MetricEventListener
    implements EventListener<WriteResult> {
        final long t0;
        final long startTime;
        final Metric.MetricInstance metric;
        final EventListener<WriteResult> eventListener;
        final String topic;
        final String app;
        final int partitionGroup;
        final List<WriteRequest> writeRequests;

        MetricEventListener(long t0, long startTime, Metric.MetricInstance metric, EventListener<WriteResult> eventListener, String topic, String app, int partitionGroup, List<WriteRequest> writeRequests) {
            this.t0 = t0;
            this.startTime = startTime;
            this.metric = metric;
            this.eventListener = eventListener;
            this.topic = topic;
            this.app = app;
            this.partitionGroup = partitionGroup;
            this.writeRequests = writeRequests;
        }

        public void onEvent(WriteResult event) {
            long elapse = System.nanoTime() - this.t0;
            this.metric.addLatency("callback", elapse);
            if (event.getCode().equals((Object)JoyQueueCode.SUCCESS)) {
                ProduceManager.this.onPutMessage(this.topic, this.app, this.partitionGroup, this.startTime, this.writeRequests);
            }
            this.eventListener.onEvent((Object)event);
        }
    }
}

