/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.toolkit.io.snappy;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.OutputStream;
import org.joyqueue.toolkit.io.snappy.BufferRecycler;
import org.joyqueue.toolkit.io.snappy.SnappyCompressor;
import org.joyqueue.toolkit.io.snappy.SnappyFramed;
import org.joyqueue.toolkit.security.Crc32C;

public final class SnappyFramedOutputStream
extends OutputStream {
    public static final int MAX_BLOCK_SIZE = 65536;
    public static final int DEFAULT_BLOCK_SIZE = 65536;
    public static final double DEFAULT_MIN_COMPRESSION_RATIO = 0.85;
    private final BufferRecycler recycler;
    private final int blockSize;
    private final byte[] buffer;
    private final byte[] outputBuffer;
    private final double minCompressionRatio;
    private final OutputStream out;
    private int position;
    private boolean closed;

    public SnappyFramedOutputStream(OutputStream out) throws IOException {
        this(out, 65536, 0.85);
    }

    public SnappyFramedOutputStream(OutputStream out, int blockSize, double minCompressionRatio) throws IOException {
        Preconditions.checkNotNull((Object)out, (Object)"output is null");
        Preconditions.checkArgument((blockSize > 0 && blockSize <= 65536 ? 1 : 0) != 0, (String)"blockSize must be in (0, 65536]", (int)blockSize);
        Preconditions.checkArgument((minCompressionRatio > 0.0 && minCompressionRatio <= 1.0 ? 1 : 0) != 0, (String)"minCompressionRatio %1s must be between (0,1.0].", (Object)minCompressionRatio);
        this.out = out;
        this.minCompressionRatio = minCompressionRatio;
        this.recycler = BufferRecycler.instance();
        this.blockSize = blockSize;
        this.buffer = this.recycler.allocOutputBuffer(blockSize);
        this.outputBuffer = this.recycler.allocEncodingBuffer(SnappyCompressor.maxCompressedLength(blockSize));
        this.writeHeader(out);
    }

    protected void writeHeader(OutputStream out) throws IOException {
        out.write(SnappyFramed.HEADER_BYTES);
    }

    protected void writeBlock(OutputStream out, byte[] data, int offset, int length, boolean compressed, int crc32c) throws IOException {
        out.write(compressed ? 0 : 1);
        int headerLength = length + 4;
        out.write(headerLength);
        out.write(headerLength >>> 8);
        out.write(headerLength >>> 16);
        out.write(crc32c);
        out.write(crc32c >>> 8);
        out.write(crc32c >>> 16);
        out.write(crc32c >>> 24);
        out.write(data, offset, length);
    }

    @Override
    public void write(int b) throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        if (this.position >= this.blockSize) {
            this.flushBuffer();
        }
        this.buffer[this.position++] = (byte)b;
    }

    @Override
    public void write(byte[] input, int offset, int length) throws IOException {
        Preconditions.checkNotNull((Object)input, (Object)"input is null");
        Preconditions.checkPositionIndexes((int)offset, (int)(offset + length), (int)input.length);
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        int free = this.blockSize - this.position;
        if (free >= length) {
            this.copyToBuffer(input, offset, length);
            return;
        }
        if (this.position > 0) {
            this.copyToBuffer(input, offset, free);
            this.flushBuffer();
            offset += free;
            length -= free;
        }
        while (length >= this.blockSize) {
            this.writeCompressed(input, offset, this.blockSize);
            offset += this.blockSize;
            length -= this.blockSize;
        }
        this.copyToBuffer(input, offset, length);
    }

    @Override
    public void flush() throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        this.flushBuffer();
        this.out.flush();
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        try {
            this.flush();
            this.out.close();
        }
        finally {
            this.closed = true;
            this.recycler.releaseOutputBuffer(this.outputBuffer);
            this.recycler.releaseEncodeBuffer(this.buffer);
        }
    }

    private void copyToBuffer(byte[] input, int offset, int length) {
        System.arraycopy(input, offset, this.buffer, this.position, length);
        this.position += length;
    }

    private void flushBuffer() throws IOException {
        if (this.position > 0) {
            this.writeCompressed(this.buffer, 0, this.position);
            this.position = 0;
        }
    }

    private void writeCompressed(byte[] input, int offset, int length) throws IOException {
        int crc32c = this.calculateCRC32C(input, offset, length);
        int compressed = SnappyCompressor.compress(input, offset, length, this.outputBuffer, 0);
        if ((double)compressed / (double)length <= this.minCompressionRatio) {
            this.writeBlock(this.out, this.outputBuffer, 0, compressed, true, crc32c);
        } else {
            this.writeBlock(this.out, input, offset, length, false, crc32c);
        }
    }

    protected int calculateCRC32C(byte[] data, int offset, int length) {
        return Crc32C.mask(data, offset, length);
    }
}

