/*
 * Decompiled with CFR 0.152.
 */
package org.joyqueue.toolkit.io.snappy;

import com.google.common.base.Preconditions;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import org.joyqueue.toolkit.io.Files;
import org.joyqueue.toolkit.io.snappy.BufferRecycler;
import org.joyqueue.toolkit.io.snappy.SnappyDecompressor;
import org.joyqueue.toolkit.io.snappy.SnappyFramed;
import org.joyqueue.toolkit.security.Crc32C;

public class SnappyFramedInputStream
extends InputStream {
    private final InputStream in;
    private final byte[] frameHeader;
    private final boolean verifyChecksums;
    private final BufferRecycler recycler;
    private byte[] input;
    private byte[] uncompressed;
    private boolean closed;
    private boolean eof;
    private int valid;
    private int position;
    private byte[] buffer;

    public SnappyFramedInputStream(InputStream in, boolean verifyChecksums) throws IOException {
        this(in, 65536, 4, verifyChecksums, SnappyFramed.HEADER_BYTES);
    }

    public SnappyFramedInputStream(InputStream in, int maxBlockSize, int frameHeaderSize, boolean verifyChecksums, byte[] expectedHeader) throws IOException {
        this.in = in;
        this.verifyChecksums = verifyChecksums;
        this.recycler = BufferRecycler.instance();
        this.input = this.recycler.allocInputBuffer(maxBlockSize + 5);
        this.uncompressed = this.recycler.allocDecodeBuffer(maxBlockSize + 5);
        this.frameHeader = new byte[frameHeaderSize];
        byte[] actualHeader = new byte[expectedHeader.length];
        int read = Files.readBytes(in, actualHeader, 0, actualHeader.length);
        if (read < expectedHeader.length) {
            throw new EOFException("encountered EOF while reading stream header");
        }
        if (!Arrays.equals(expectedHeader, actualHeader)) {
            throw new IOException("invalid stream header");
        }
    }

    @Override
    public int read() throws IOException {
        if (this.closed) {
            return -1;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        return this.buffer[this.position++] & 0xFF;
    }

    @Override
    public int read(byte[] output, int offset, int length) throws IOException {
        Preconditions.checkNotNull((Object)output, (Object)"output is null");
        Preconditions.checkPositionIndexes((int)offset, (int)(offset + length), (int)output.length);
        if (this.closed) {
            return -1;
        }
        if (length == 0) {
            return 0;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        int size = Math.min(length, this.available());
        System.arraycopy(this.buffer, this.position, output, offset, size);
        this.position += size;
        return size;
    }

    @Override
    public int available() throws IOException {
        if (this.closed) {
            return 0;
        }
        return this.valid - this.position;
    }

    @Override
    public void close() throws IOException {
        try {
            this.in.close();
        }
        finally {
            if (!this.closed) {
                this.closed = true;
                this.recycler.releaseInputBuffer(this.input);
                this.recycler.releaseDecodeBuffer(this.uncompressed);
            }
        }
    }

    private boolean ensureBuffer() throws IOException {
        int actualCrc32c;
        int actualRead;
        if (this.available() > 0) {
            return true;
        }
        if (this.eof) {
            return false;
        }
        if (!this.readBlockHeader()) {
            this.eof = true;
            return false;
        }
        FrameMetaData frameMetaData = this.getFrameMetaData(this.frameHeader);
        if (FrameAction.SKIP == frameMetaData.frameAction) {
            Files.skip(this.in, frameMetaData.length);
            return this.ensureBuffer();
        }
        if (frameMetaData.length > this.input.length) {
            this.input = this.recycler.allocInputBuffer(frameMetaData.length);
            this.uncompressed = this.recycler.allocDecodeBuffer(frameMetaData.length);
        }
        if ((actualRead = Files.readBytes(this.in, this.input, 0, frameMetaData.length)) != frameMetaData.length) {
            throw new EOFException("unexpectd EOF when reading frame");
        }
        FrameData frameData = this.getFrameData(this.frameHeader, this.input, actualRead);
        if (FrameAction.UNCOMPRESS == frameMetaData.frameAction) {
            int uncompressedLength = SnappyDecompressor.getUncompressedLength(this.input, frameData.offset);
            if (uncompressedLength > this.uncompressed.length) {
                this.uncompressed = this.recycler.allocDecodeBuffer(uncompressedLength);
            }
            this.valid = SnappyDecompressor.uncompress(this.input, frameData.offset, actualRead - frameData.offset, this.uncompressed, 0);
            this.buffer = this.uncompressed;
            this.position = 0;
        } else {
            this.position = frameData.offset;
            this.buffer = this.input;
            this.valid = actualRead;
        }
        if (this.verifyChecksums && frameData.checkSum != (actualCrc32c = Crc32C.mask(this.buffer, this.position, this.valid - this.position))) {
            throw new IOException("Corrupt input: invalid checksum");
        }
        return true;
    }

    private boolean readBlockHeader() throws IOException {
        int read = Files.readBytes(this.in, this.frameHeader, 0, this.frameHeader.length);
        if (read == -1) {
            return false;
        }
        if (read < this.frameHeader.length) {
            throw new EOFException("encountered EOF while reading block header");
        }
        return true;
    }

    protected FrameMetaData getFrameMetaData(byte[] frameHeader) throws IOException {
        int minLength;
        FrameAction frameAction;
        int length = frameHeader[1] & 0xFF;
        length |= (frameHeader[2] & 0xFF) << 8;
        length |= (frameHeader[3] & 0xFF) << 16;
        int flag = frameHeader[0] & 0xFF;
        switch (flag) {
            case 0: {
                frameAction = FrameAction.UNCOMPRESS;
                minLength = 5;
                break;
            }
            case 1: {
                frameAction = FrameAction.RAW;
                minLength = 5;
                break;
            }
            case 255: {
                if (length != 6) {
                    throw new IOException("stream identifier chunk with invalid length: " + length);
                }
                frameAction = FrameAction.SKIP;
                minLength = 6;
                break;
            }
            default: {
                if (flag <= 127) {
                    throw new IOException("unsupported unskippable chunk: " + Integer.toHexString(flag));
                }
                frameAction = FrameAction.SKIP;
                minLength = 0;
            }
        }
        if (length < minLength) {
            throw new IOException("invalid length: " + length + " for chunk flag: " + Integer.toHexString(flag));
        }
        return new FrameMetaData(frameAction, length);
    }

    protected FrameData getFrameData(byte[] frameHeader, byte[] content, int length) {
        int crc32c = (content[3] & 0xFF) << 24 | (content[2] & 0xFF) << 16 | (content[1] & 0xFF) << 8 | content[0] & 0xFF;
        return new FrameData(crc32c, 4);
    }

    protected static final class FrameData {
        final int checkSum;
        final int offset;

        public FrameData(int checkSum, int offset) {
            this.checkSum = checkSum;
            this.offset = offset;
        }
    }

    protected static final class FrameMetaData {
        final int length;
        final FrameAction frameAction;

        public FrameMetaData(FrameAction frameAction, int length) {
            this.frameAction = frameAction;
            this.length = length;
        }
    }

    static enum FrameAction {
        RAW,
        SKIP,
        UNCOMPRESS;

    }
}

