/*
 * Decompiled with CFR 0.152.
 */
package org.jsimpledb.kv.caching;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import org.jsimpledb.kv.CloseableKVStore;
import org.jsimpledb.kv.KVException;
import org.jsimpledb.kv.KVPair;
import org.jsimpledb.kv.KVPairIterator;
import org.jsimpledb.kv.KVStore;
import org.jsimpledb.kv.KeyRange;
import org.jsimpledb.kv.caching.CachingConfig;
import org.jsimpledb.kv.caching.RingEntry;
import org.jsimpledb.kv.util.CloseableForwardingKVStore;
import org.jsimpledb.util.ByteUtil;
import org.jsimpledb.util.CloseableIterator;
import org.jsimpledb.util.MovingAverage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingKVStore
extends CloseableForwardingKVStore
implements CachingConfig {
    public static final int DEFAULT_MAX_RANGES = 256;
    public static final long DEFAULT_MAX_RANGE_BYTES = 0xA00000L;
    public static final long DEFAULT_MAX_TOTAL_BYTES = 0x6400000L;
    private static final byte[][] EMPTY_BYTES = new byte[0][];
    private static final double RTT_DECAY_FACTOR = 0.025;
    private static final double WAIT_VS_RTT_FACTOR = 1.5;
    private static final int INITIAL_ARRAY_CAPACITY = 32;
    private static final float ARRAY_GROWTH_FACTOR = 1.5f;
    private static final Comparator<KVRange> SORT_BY_MIN = Comparator.comparing(KVRange::getMin, ByteUtil.COMPARATOR);
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final MovingAverage rtt;
    private final ExecutorService executor;
    private final TreeSet<KVRange> ranges = new TreeSet<KVRange>(SORT_BY_MIN);
    private final RingEntry<KVRange> lru = new RingEntry<Object>(null);
    private int maxRanges = 256;
    private long maxRangeBytes = 0xA00000L;
    private long maxTotalBytes = 0x6400000L;
    private boolean readAhead = true;
    private long totalBytes;
    private KVException error;

    public CachingKVStore(KVStore kvstore, ExecutorService executor, long rttEstimate) {
        this(kvstore, null, executor, rttEstimate);
    }

    public CachingKVStore(CloseableKVStore kvstore, ExecutorService executor, long rttEstimate) {
        this((KVStore)kvstore, (Closeable)kvstore, executor, rttEstimate);
    }

    private CachingKVStore(KVStore kvstore, Closeable closeable, ExecutorService executor, long rttEstimate) {
        super(kvstore, closeable);
        Preconditions.checkArgument((executor != null ? 1 : 0) != 0, (Object)"null executor");
        Preconditions.checkArgument((rttEstimate >= 0L ? 1 : 0) != 0, (Object)"rttEstimate < 0");
        this.executor = executor;
        this.rtt = new MovingAverage(0.025, (double)rttEstimate);
    }

    @Override
    public synchronized long getMaxRangeBytes() {
        return this.maxRangeBytes;
    }

    @Override
    public synchronized void setMaxRangeBytes(long maxRangeBytes) {
        Preconditions.checkArgument((this.maxRanges > 0 ? 1 : 0) != 0, (Object)"maxRangeBytes <= 0");
        this.maxRangeBytes = maxRangeBytes;
    }

    @Override
    public synchronized long getMaxTotalBytes() {
        return this.maxTotalBytes;
    }

    @Override
    public synchronized void setMaxTotalBytes(long maxTotalBytes) {
        Preconditions.checkArgument((maxTotalBytes > 0L ? 1 : 0) != 0, (Object)"maxTotalBytes <= 0");
        this.maxTotalBytes = maxTotalBytes;
    }

    @Override
    public synchronized int getMaxRanges() {
        return this.maxRanges;
    }

    @Override
    public synchronized void setMaxRanges(int maxRanges) {
        Preconditions.checkArgument((maxRanges > 0 ? 1 : 0) != 0, (Object)"maxRanges <= 0");
        this.maxRanges = maxRanges;
    }

    @Override
    public synchronized boolean isReadAhead() {
        return this.readAhead;
    }

    @Override
    public synchronized void setReadAhead(boolean readAhead) {
        this.readAhead = readAhead;
    }

    public byte[] get(byte[] key) {
        KVPair pair = this.getAtLeast(key, ByteUtil.getNextKey((byte[])key));
        return pair != null ? pair.getValue() : null;
    }

    public CloseableIterator<KVPair> getRange(byte[] minKey, byte[] maxKey, boolean reverse) {
        if (minKey == null) {
            minKey = ByteUtil.EMPTY;
        }
        return new KVPairIterator((KVStore)this, new KeyRange(minKey, maxKey), null, reverse);
    }

    public KVPair getAtLeast(byte[] minKey, byte[] maxKey) {
        if (minKey == null) {
            minKey = ByteUtil.EMPTY;
        }
        if (KeyRange.compare((byte[])minKey, (byte[])maxKey) >= 0) {
            return null;
        }
        return this.find(minKey, maxKey, false);
    }

    public KVPair getAtMost(byte[] maxKey, byte[] minKey) {
        if (minKey == null) {
            minKey = ByteUtil.EMPTY;
        }
        if (KeyRange.compare((byte[])minKey, (byte[])maxKey) >= 0) {
            return null;
        }
        return this.find(maxKey, minKey, true);
    }

    public void put(byte[] key, byte[] value) {
        throw new UnsupportedOperationException();
    }

    public void remove(byte[] key) {
        throw new UnsupportedOperationException();
    }

    public void removeRange(byte[] minKey, byte[] maxKey) {
        throw new UnsupportedOperationException();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        CachingKVStore cachingKVStore = this;
        synchronized (cachingKVStore) {
            if (this.log.isTraceEnabled()) {
                this.trace("closing with ranges={}", this.ranges);
            }
            for (KVRange range : this.ranges) {
                this.discard(range, false);
            }
            this.ranges.clear();
        }
        super.close();
    }

    public double getRttEstimate() {
        return this.rtt.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private KVPair find(byte[] start, byte[] limit, boolean reverse) {
        long waitTimeRemain;
        assert (start != null || reverse);
        assert (limit != null || !reverse);
        CachingKVStore cachingKVStore = this;
        synchronized (cachingKVStore) {
            assert (this.sanityCheck());
            waitTimeRemain = (long)(1.5 * this.rtt.get());
        }
        assert (waitTimeRemain >= 0L);
        long lastLoopTime = System.nanoTime();
        boolean interrupted = false;
        while (true) {
            CompletableFuture<?> future;
            CachingKVStore cachingKVStore2 = this;
            synchronized (cachingKVStore2) {
                if (this.error != null) {
                    this.error.rethrow();
                }
                assert (this.sanityCheck());
                long now = System.nanoTime();
                long elapsed = now - lastLoopTime;
                waitTimeRemain = Math.max(0L, waitTimeRemain - elapsed);
                lastLoopTime = now;
                if (this.log.isTraceEnabled()) {
                    this.trace("find: start={} limit={} {} remain={}ms ranges={}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), reverse ? "reverse" : "forward", waitTimeRemain / 1000000L, this.ranges);
                }
                KVRange range = this.last(start != null ? this.ranges.headSet(this.key(start), true) : this.ranges);
                if (reverse && range != null && KeyRange.compare((byte[])range.getMax(), (byte[])start) < 0) {
                    range = this.first(this.ranges.tailSet(this.key(start), true));
                    assert (range == null || KeyRange.compare((byte[])range.getMax(), (byte[])start) >= 0);
                }
                Loader loader = null;
                if (range != null) {
                    boolean rangeContainsStart;
                    assert (!reverse ? KeyRange.compare((byte[])start, (byte[])range.getMin()) >= 0 : KeyRange.compare((byte[])start, (byte[])range.getMax()) <= 0);
                    boolean bl = reverse ? KeyRange.compare((byte[])start, (byte[])range.getMin()) > 0 : (rangeContainsStart = KeyRange.compare((byte[])start, (byte[])range.getMax()) < 0);
                    if (rangeContainsStart) {
                        boolean rangeContainsLimit;
                        KVPair pair;
                        KVPair kVPair = pair = reverse ? range.getAtMost(start) : range.getAtLeast(start);
                        boolean bl2 = reverse ? KeyRange.compare((byte[])limit, (byte[])range.getMin()) >= 0 : (rangeContainsLimit = KeyRange.compare((byte[])limit, (byte[])range.getMax()) <= 0);
                        if (pair != null && rangeContainsLimit) {
                            boolean pairWithinLimit;
                            boolean bl3 = reverse ? KeyRange.compare((byte[])pair.getKey(), (byte[])limit) >= 0 : (pairWithinLimit = KeyRange.compare((byte[])pair.getKey(), (byte[])limit) < 0);
                            if (!pairWithinLimit) {
                                pair = null;
                            }
                        }
                        if (pair != null || rangeContainsLimit) {
                            if (this.log.isTraceEnabled()) {
                                if (pair != null) {
                                    this.trace("find: start={} limit={} found {}={} in range={}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), ByteUtil.toString((byte[])pair.getKey()), ByteUtil.toString((byte[])pair.getValue()), range);
                                } else {
                                    this.trace("find: start={} limit={} not found in range={}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range);
                                }
                            }
                            this.touch(range);
                            return pair;
                        }
                        if (this.log.isTraceEnabled()) {
                            this.trace("find: start={} limit={} not found in range={} -> new start={}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range, reverse ? range.getMin() : range.getMax());
                        }
                        start = reverse ? range.getMin() : range.getMax();
                    }
                    boolean startOnEdge = KeyRange.compare((byte[])start, (byte[])(reverse ? range.getMin() : range.getMax())) == 0;
                    loader = range.getLoader(reverse);
                    if (loader == null) {
                        if (!startOnEdge) {
                            if (this.log.isTraceEnabled()) {
                                this.trace("find: start={} limit={} range={} not adjacent, no loader => create my own range", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range);
                            }
                            range = null;
                        } else if (this.log.isTraceEnabled()) {
                            this.trace("find: start={} limit={} range={} adjacent but no loader => create new loader", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range);
                        }
                    } else if (!startOnEdge) {
                        long eta;
                        if (range.size() < 2) {
                            eta = (long)((double)System.nanoTime() - ((double)loader.getStartTime() + this.rtt.get()));
                        } else {
                            double arrivalRate = loader.getArrivalRate();
                            byte[] loaderBase = reverse ? range.getMin() : range.getMax();
                            long l = eta = arrivalRate > 0.0 ? (long)(Math.abs(this.measure(start) - this.measure(loaderBase)) / arrivalRate) : -1L;
                        }
                        if (eta >= waitTimeRemain) {
                            range = null;
                            loader = null;
                            if (this.log.isTraceEnabled()) {
                                this.trace("find: start={} limit={} range={} eta={} >= waitTimeRemain={} => don't wait for it", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range, eta, waitTimeRemain);
                            }
                        } else if (this.log.isTraceEnabled()) {
                            this.trace("find: start={} limit={} range={} eta={} < waitTimeRemain={} => wait for it", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range, eta, waitTimeRemain);
                        }
                    }
                } else if (this.log.isTraceEnabled()) {
                    this.trace("find: start={} limit={} no containing or adjacent range found", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit));
                }
                assert (range != null || loader == null);
                if (range == null) {
                    range = new KVRange(start);
                    this.ranges.add(range);
                    range.getLruEntry().attachAfter(this.lru);
                    if (this.log.isTraceEnabled()) {
                        this.trace("find: start={} limit={} created new {}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), range);
                    }
                } else {
                    this.touch(range);
                }
                if (loader == null) {
                    assert (range.getLoader(reverse) == null) : "" + range;
                    byte[] actualLimit = limit;
                    if (reverse && limit.length > 0) {
                        KVRange nextRange = this.ranges.lower(range);
                        if (nextRange != null && !nextRange.isPrimordial()) {
                            actualLimit = nextRange.getMax();
                        } else if (this.readAhead) {
                            actualLimit = ByteUtil.EMPTY;
                        }
                    } else if (!reverse && limit != null) {
                        KVRange nextRange = this.ranges.higher(range);
                        if (nextRange != null && !nextRange.isPrimordial()) {
                            actualLimit = nextRange.getMin();
                        } else if (this.readAhead) {
                            actualLimit = null;
                        }
                    }
                    loader = new Loader(range, reverse, actualLimit);
                    if (this.log.isTraceEnabled()) {
                        this.trace("find: start={} limit={} created new {}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), loader);
                    }
                }
                future = loader.getFuture();
                assert (future != null);
                this.scrub();
                assert (this.sanityCheck());
                if (this.log.isTraceEnabled()) {
                    this.trace("going to sleep, ranges={}", this.ranges);
                }
            }
            try {
                future.get();
                if (!this.log.isTraceEnabled()) continue;
                this.trace("find: start={} limit={} woke up", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit));
                continue;
            }
            catch (CancellationException e) {
                if (!this.log.isTraceEnabled()) continue;
                this.trace("find: start={} limit={} woke up - {}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), "canceled");
                continue;
            }
            catch (ExecutionException e) {
                if (this.log.isTraceEnabled()) {
                    this.trace("find: start={} limit={} woke up - {}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), e.getCause());
                }
                CachingKVStore cachingKVStore3 = this;
                synchronized (cachingKVStore3) {
                    if (this.error == null) {
                        Throwable cause = e.getCause();
                        this.error = cause instanceof KVException ? (KVException)cause : new KVException(cause);
                    }
                    throw this.error.rethrow();
                }
            }
            catch (InterruptedException e) {
                if (this.log.isTraceEnabled()) {
                    this.trace("find: start={} limit={} woke up - {}", ByteUtil.toString((byte[])start), ByteUtil.toString((byte[])limit), "interrupted");
                }
                Thread.currentThread().interrupt();
                continue;
            }
            break;
        }
    }

    private void touch(KVRange range) {
        assert (Thread.holdsLock(this));
        assert (range.getLruEntry().isAttached());
        assert (this.ranges.contains(range)) : "range " + range + " not found in " + this.ranges;
        if (this.log.isTraceEnabled()) {
            this.trace("touch: renew range={}", range);
        }
        range.getLruEntry().attachAfter(this.lru);
    }

    private void scrub() {
        KVRange range;
        assert (Thread.holdsLock(this));
        while (this.totalBytes > this.maxTotalBytes && (range = this.lru.prev().getOwner()) != null) {
            this.discard(range, true);
        }
    }

    private void discard(KVRange range, boolean removeFromRanges) {
        assert (Thread.holdsLock(this));
        if (this.log.isTraceEnabled()) {
            this.trace("discarding range={} age={}ms", range, (System.nanoTime() - range.getCreationTime()) / 1000000L);
        }
        range.stopLoader(false);
        range.stopLoader(true);
        range.getLruEntry().detach();
        this.totalBytes -= range.getTotalBytes();
        if (removeFromRanges) {
            boolean removed = this.ranges.remove(range);
            assert (removed);
        }
    }

    private boolean sanityCheck() {
        assert (Thread.holdsLock(this));
        for (KVRange kVRange : this.ranges) {
            assert (kVRange.sanityCheck());
        }
        KVRange prev = null;
        for (KVRange next : this.ranges) {
            assert (KeyRange.compare((byte[])next.getMin(), (byte[])next.getMax()) <= 0);
            assert (KeyRange.compare((byte[])next.getMin(), (byte[])next.getMax()) != 0 || next.getLoader(false) != null || next.getLoader(true) != null);
            assert (prev == null || KeyRange.compare((byte[])prev.max, (byte[])next.min) < 0) : "range overlap : " + this.ranges;
            prev = next;
        }
        ArrayList<KVRange> arrayList = new ArrayList<KVRange>(this.ranges);
        ArrayList<KVRange> lruList = new ArrayList<KVRange>(this.ranges.size());
        RingEntry<KVRange> entry = this.lru.next();
        while (entry.getOwner() != null) {
            lruList.add(entry.getOwner());
            entry = entry.next();
        }
        Collections.sort(lruList, SORT_BY_MIN);
        assert (lruList.equals(arrayList)) : "lru=" + lruList + ", ranges=" + arrayList;
        return true;
    }

    private KVRange key(byte[] key) {
        return new KVRange(key, key, EMPTY_BYTES, EMPTY_BYTES, 0, 0, 0L);
    }

    private double measure(byte[] value) {
        return value != null ? ByteUtil.toDouble((byte[])value) : 1.0;
    }

    private void debug(String msg, Object ... params) {
        this.log.debug(this.getClass().getSimpleName() + ": " + msg, params);
    }

    private void trace(String msg, Object ... params) {
        this.log.trace(this.getClass().getSimpleName() + ": " + msg, params);
    }

    private <T> T first(SortedSet<T> set) {
        try {
            return set.first();
        }
        catch (NoSuchElementException e) {
            return null;
        }
    }

    private <T> T last(SortedSet<T> set) {
        try {
            return set.last();
        }
        catch (NoSuchElementException e) {
            return null;
        }
    }

    private class KVRange {
        private final long creationTime = System.nanoTime();
        private final RingEntry<KVRange> lruEntry = new RingEntry<KVRange>(this);
        private final Loader[] loaders = new Loader[2];
        private byte[] min;
        private byte[] max;
        private byte[][] keys;
        private byte[][] vals;
        private int minIndex;
        private int maxIndex;
        private long totalBytes;
        private int lastKnownRangesIndex;

        KVRange(byte[] start) {
            this(start, start, new byte[32][], new byte[32][], 0, 0, 0L);
        }

        KVRange(byte[] min, byte[] max, byte[][] keys, byte[][] vals, int minIndex, int maxIndex, long totalBytes) {
            this.min = min;
            this.max = max;
            this.keys = keys;
            this.vals = vals;
            this.minIndex = minIndex;
            this.maxIndex = maxIndex;
            this.totalBytes = totalBytes;
            assert (this.sanityCheckInternal());
        }

        public byte[] getMin() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.min;
        }

        public void setMin(byte[] min) {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (min != null);
            assert (ByteUtil.compare((byte[])min, (byte[])this.min) <= 0);
            this.min = min;
        }

        public byte[] getMax() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.max;
        }

        public void setMax(byte[] max) {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (KeyRange.compare((byte[])max, (byte[])this.max) >= 0);
            this.max = max;
        }

        public KeyRange getKeyRange() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return new KeyRange(this.min, this.max);
        }

        public int getLastKnownRangesIndex() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.lastKnownRangesIndex;
        }

        public void setLastKnownRangesIndex(int lastKnownRangesIndex) {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (lastKnownRangesIndex >= 0);
            this.lastKnownRangesIndex = lastKnownRangesIndex;
        }

        public Loader getLoader(boolean reverse) {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.loaders[reverse ? 1 : 0];
        }

        public void setLoader(boolean reverse, Loader loader) {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (this.loaders[reverse ? 1 : 0] == null || this.loaders[reverse ? 1 : 0].isStopped());
            this.loaders[reverse ? 1 : 0] = loader;
        }

        public void stopLoader(boolean reverse) {
            assert (Thread.holdsLock(CachingKVStore.this));
            Loader loader = this.getLoader(reverse);
            if (loader != null) {
                loader.stop();
            }
        }

        public KVPair getAtLeast(byte[] key) {
            assert (Thread.holdsLock(CachingKVStore.this));
            int index = Arrays.binarySearch(this.keys, this.minIndex, this.maxIndex, key, ByteUtil.COMPARATOR);
            if (index < 0 && (index ^= 0xFFFFFFFF) >= this.maxIndex) {
                return null;
            }
            return new KVPair((byte[])this.keys[index].clone(), (byte[])this.vals[index].clone());
        }

        public KVPair getAtMost(byte[] key) {
            assert (Thread.holdsLock(CachingKVStore.this));
            int index = Arrays.binarySearch(this.keys, this.minIndex, this.maxIndex, key, KeyRange::compare);
            if (index < 0) {
                index ^= 0xFFFFFFFF;
            }
            if (--index < this.minIndex) {
                return null;
            }
            return new KVPair((byte[])this.keys[index].clone(), (byte[])this.vals[index].clone());
        }

        public int size() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.maxIndex - this.minIndex;
        }

        public boolean isEmpty() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.maxIndex == this.minIndex;
        }

        public boolean isPrimordial() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return KeyRange.compare((byte[])this.min, (byte[])this.max) == 0;
        }

        public long getTotalBytes() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.totalBytes;
        }

        public long getCreationTime() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.creationTime;
        }

        public RingEntry<KVRange> getLruEntry() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.lruEntry;
        }

        public CachingKVStore getKVStore() {
            return CachingKVStore.this;
        }

        public void add(byte[] key, byte[] val) {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (!this.getKeyRange().contains(key));
            assert (key != null && val != null);
            if (ByteUtil.compare((byte[])key, (byte[])this.min) < 0) {
                if (this.minIndex == 0) {
                    this.growArrays();
                }
                assert (this.minIndex > 0);
                --this.minIndex;
                this.keys[this.minIndex] = key;
                this.vals[this.minIndex] = val;
            } else {
                assert (KeyRange.compare((byte[])key, (byte[])this.max) >= 0);
                if (this.maxIndex == this.keys.length) {
                    this.growArrays();
                }
                assert (this.maxIndex < this.keys.length);
                this.keys[this.maxIndex] = key;
                this.vals[this.maxIndex] = val;
                ++this.maxIndex;
            }
            this.totalBytes += (long)(key.length + val.length);
        }

        public KVRange merge(KVRange next) {
            int newMinIndex;
            Object newVals;
            Object newKeys;
            int shift;
            assert (this.getKVStore() == next.getKVStore());
            assert (Thread.holdsLock(this.getKVStore()));
            assert (this.sanityCheck());
            assert (next.sanityCheck());
            assert (KeyRange.compare((byte[])this.getMax(), (byte[])next.getMin()) == 0);
            int thisSize = this.size();
            int nextSize = next.size();
            int newSize = thisSize + nextSize;
            long newTotalBytes = this.totalBytes + next.totalBytes;
            if (newSize <= this.keys.length) {
                shift = nextSize - (this.keys.length - this.maxIndex);
                if (shift > 0) {
                    System.arraycopy(this.keys, this.minIndex, this.keys, this.minIndex - shift, thisSize);
                    System.arraycopy(this.vals, this.minIndex, this.vals, this.minIndex - shift, thisSize);
                    this.minIndex -= shift;
                    this.maxIndex -= shift;
                }
                System.arraycopy(next.keys, next.minIndex, this.keys, this.maxIndex, nextSize);
                System.arraycopy(next.vals, next.minIndex, this.vals, this.maxIndex, nextSize);
                newKeys = this.keys;
                newVals = this.vals;
                newMinIndex = this.minIndex;
            } else if (newSize <= next.keys.length) {
                shift = thisSize - next.minIndex;
                if (shift > 0) {
                    System.arraycopy(next.keys, next.minIndex, next.keys, next.minIndex + shift, nextSize);
                    System.arraycopy(next.vals, next.minIndex, next.vals, next.minIndex + shift, nextSize);
                    next.minIndex += shift;
                    next.maxIndex += shift;
                }
                System.arraycopy(this.keys, this.minIndex, next.keys, next.minIndex - thisSize, thisSize);
                System.arraycopy(this.vals, this.minIndex, next.vals, next.minIndex - thisSize, thisSize);
                newKeys = next.keys;
                newVals = next.vals;
                newMinIndex = next.minIndex - thisSize;
            } else {
                newKeys = new byte[(int)((float)newSize * 1.5f) + 30][];
                newVals = new byte[((byte[][])newKeys).length][];
                newMinIndex = (((byte[][])newKeys).length - newSize) / 2;
                System.arraycopy(this.keys, this.minIndex, newKeys, newMinIndex, thisSize);
                System.arraycopy(this.vals, this.minIndex, newVals, newMinIndex, thisSize);
                System.arraycopy(next.keys, next.minIndex, newKeys, newMinIndex + thisSize, nextSize);
                System.arraycopy(next.vals, next.minIndex, newVals, newMinIndex + thisSize, nextSize);
            }
            CachingKVStore cachingKVStore = this.getKVStore();
            cachingKVStore.getClass();
            KVRange range = cachingKVStore.new KVRange(this.min, next.max, (byte[][])newKeys, (byte[][])newVals, newMinIndex, newMinIndex + newSize, newTotalBytes);
            assert (this.invalidate());
            assert (next.invalidate());
            return range;
        }

        private void growArrays() {
            assert (Thread.holdsLock(CachingKVStore.this));
            int size = this.size();
            byte[][] newKeys = new byte[(int)((float)size * 1.5f) + 30][];
            byte[][] newVals = new byte[newKeys.length][];
            int newMinIndex = (newKeys.length - size) / 2;
            System.arraycopy(this.keys, this.minIndex, newKeys, newMinIndex, size);
            System.arraycopy(this.vals, this.minIndex, newVals, newMinIndex, size);
            this.keys = newKeys;
            this.vals = newVals;
            this.minIndex = newMinIndex;
            this.maxIndex = newMinIndex + size;
            assert (this.sanityCheck());
        }

        public boolean sanityCheck() {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (this.sanityCheckInternal());
            assert (CachingKVStore.this.ranges.contains(this)) : this + " not in " + CachingKVStore.access$800(CachingKVStore.this);
            assert (this.lruEntry.isAttached()) : this + " not in LRU";
            return true;
        }

        private boolean sanityCheckInternal() {
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (KeyRange.compare((byte[])this.min, (byte[])this.max) <= 0);
            assert (this.keys != null);
            assert (this.vals != null);
            assert (this.size() >= 0);
            assert (this.lastKnownRangesIndex >= 0) : "range=" + this + " ranges=" + CachingKVStore.access$800(CachingKVStore.this);
            assert (this.keys.length == this.vals.length);
            assert (this.size() <= this.keys.length);
            for (int i = this.minIndex; i < this.maxIndex; ++i) {
                assert (this.keys[i] != null);
                assert (this.vals[i] != null);
                byte[] key = this.keys[i];
                assert (KeyRange.compare((byte[])key, (byte[])this.getMin()) >= 0);
                assert (KeyRange.compare((byte[])key, (byte[])this.getMax()) < 0);
                assert (i == this.minIndex || ByteUtil.compare((byte[])key, (byte[])this.keys[i - 1]) > 0);
            }
            return true;
        }

        private boolean invalidate() {
            assert (Thread.holdsLock(CachingKVStore.this));
            this.keys = null;
            this.vals = null;
            this.minIndex = -1;
            this.maxIndex = -1;
            this.totalBytes = -1L;
            this.lastKnownRangesIndex = -1;
            return true;
        }

        public boolean containsOnlyKeyInRange(byte[] key, byte[] min, byte[] max) {
            int searchMax;
            assert (Thread.holdsLock(CachingKVStore.this));
            if (KeyRange.compare((byte[])key, (byte[])this.min) < 0 || KeyRange.compare((byte[])key, (byte[])this.max) >= 0) {
                key = null;
            }
            assert (KeyRange.compare((byte[])min, (byte[])max) <= 0);
            int searchMin = Arrays.binarySearch(this.keys, this.minIndex, this.maxIndex, min, KeyRange::compare);
            if (searchMin < 0) {
                searchMin ^= 0xFFFFFFFF;
            }
            if ((searchMax = Arrays.binarySearch(this.keys, this.minIndex, this.maxIndex, max, KeyRange::compare)) < 0) {
                searchMax ^= 0xFFFFFFFF;
            }
            if (key == null) {
                return searchMin == searchMax;
            }
            return searchMax == searchMin + 1 && KeyRange.compare((byte[])this.keys[searchMin], (byte[])key) == 0;
        }

        private void debug(String msg, Object ... params) {
            CachingKVStore.this.log.debug(this + ": " + msg, params);
        }

        private void trace(String msg, Object ... params) {
            CachingKVStore.this.log.trace(this + ": " + msg, params);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public String toString() {
            CachingKVStore cachingKVStore = CachingKVStore.this;
            synchronized (cachingKVStore) {
                StringBuilder keysBuf = new StringBuilder();
                if (!this.isEmpty()) {
                    for (int i = this.minIndex; i < this.maxIndex; ++i) {
                        if (this.maxIndex - i > 5 && i - this.minIndex > 5) {
                            if (i - this.minIndex != 6) continue;
                            keysBuf.append("...");
                            continue;
                        }
                        keysBuf.append(',');
                        if (i == this.minIndex) {
                            keysBuf.append("keys=");
                        }
                        keysBuf.append(ByteUtil.toString((byte[])this.keys[i]));
                    }
                }
                return "KVRange[min=" + ByteUtil.toString((byte[])this.min) + ",max=" + ByteUtil.toString((byte[])this.max) + keysBuf + (this.loaders[0] != null ? ",fwdLoader=" + this.loaders[0] : "") + (this.loaders[1] != null ? ",revLoader=" + this.loaders[1] : "") + "]";
            }
        }
    }

    private class Loader
    implements Runnable {
        private static final double RATE_DECAY_FACTOR = 0.1;
        private final Logger log;
        private final long startTime;
        private final KVRange range;
        private final boolean reverse;
        private final byte[] start;
        private final byte[] limit;
        private Future<?> taskFuture;
        private CompletableFuture<?> future;
        private MovingAverage arrivalRate;

        Loader(KVRange range, boolean reverse, byte[] limit) {
            this.log = CachingKVStore.this.log;
            this.startTime = System.nanoTime();
            this.arrivalRate = new MovingAverage(0.1);
            assert (Thread.holdsLock(CachingKVStore.this));
            assert (range != null);
            this.range = range;
            this.reverse = reverse;
            this.start = this.getBase();
            assert (!reverse ? KeyRange.compare((byte[])limit, (byte[])this.start) > 0 : KeyRange.compare((byte[])limit, (byte[])this.start) < 0);
            this.future = new CompletableFuture();
            this.taskFuture = CachingKVStore.this.executor.submit(this);
            this.range.setLoader(this.reverse, this);
            this.limit = limit;
        }

        public byte[] getBase() {
            return this.reverse ? this.range.getMin() : this.range.getMax();
        }

        public boolean isReverse() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.reverse;
        }

        public double getArrivalRate() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.arrivalRate.get();
        }

        public long getStartTime() {
            return this.startTime;
        }

        public CompletableFuture<?> getFuture() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.future;
        }

        public boolean isStopped() {
            assert (Thread.holdsLock(CachingKVStore.this));
            return this.future == null;
        }

        public void stop() {
            assert (Thread.holdsLock(CachingKVStore.this));
            if (this.future != null) {
                this.future.cancel(true);
                this.future = null;
            }
            if (this.taskFuture != null) {
                if (this.log.isTraceEnabled()) {
                    this.trace("stopping task", new Object[0]);
                }
                this.taskFuture.cancel(true);
                this.taskFuture = null;
            }
            if (this.range.getLoader(this.reverse) == this) {
                this.range.setLoader(this.reverse, null);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Object iterator;
            try {
                if (this.log.isTraceEnabled()) {
                    this.trace("run() invoked, creating iterator {}...{}", ByteUtil.toString((byte[])this.start), ByteUtil.toString((byte[])this.limit));
                }
                iterator = this.reverse ? CachingKVStore.super.getRange(this.limit, this.start, true) : CachingKVStore.super.getRange(this.start, this.limit, false);
                Throwable throwable = null;
                try {
                    this.load((Iterator<KVPair>)iterator);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (iterator != null) {
                        if (throwable != null) {
                            try {
                                iterator.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                        } else {
                            iterator.close();
                        }
                    }
                }
            }
            catch (Throwable t) {
                if (this.log.isTraceEnabled()) {
                    this.trace("exception in task", t);
                }
                CachingKVStore cachingKVStore = CachingKVStore.this;
                synchronized (cachingKVStore) {
                    if (!this.isStopped()) {
                        if (CachingKVStore.this.error == null) {
                            CachingKVStore.this.error = t instanceof KVException ? (KVException)t : new KVException(t);
                            CachingKVStore.this.close();
                        }
                        throw CachingKVStore.this.error.rethrow();
                    }
                }
            }
            finally {
                iterator = CachingKVStore.this;
                synchronized (iterator) {
                    this.stop();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void load(Iterator<KVPair> iterator) {
            if (this.log.isTraceEnabled()) {
                this.trace("starting load", new Object[0]);
            }
            int numKeysLoaded = 0;
            long lastMeasureTime = 0L;
            double lastKeySpaceValue = 0.0;
            while (true) {
                CachingKVStore cachingKVStore;
                byte[] val;
                KVPair pair = iterator.hasNext() ? iterator.next() : null;
                byte[] key = pair != null ? pair.getKey() : null;
                byte[] byArray = val = pair != null ? pair.getValue() : null;
                if (key != null) {
                    if (this.reverse) {
                        cachingKVStore = CachingKVStore.this;
                        synchronized (cachingKVStore) {
                            assert (KeyRange.compare((byte[])key, (byte[])this.getBase()) < 0) : "key=" + ByteUtil.toString((byte[])key) + " >= " + ByteUtil.toString((byte[])this.getBase());
                            assert (KeyRange.compare((byte[])key, (byte[])this.limit) >= 0) : "key=" + ByteUtil.toString((byte[])key) + " < " + ByteUtil.toString((byte[])this.limit);
                        }
                    }
                    cachingKVStore = CachingKVStore.this;
                    synchronized (cachingKVStore) {
                        assert (KeyRange.compare((byte[])key, (byte[])this.getBase()) >= 0) : "key=" + ByteUtil.toString((byte[])key) + " < " + ByteUtil.toString((byte[])this.getBase());
                        assert (KeyRange.compare((byte[])key, (byte[])this.limit) < 0) : "key=" + ByteUtil.toString((byte[])key) + " >= " + ByteUtil.toString((byte[])this.limit);
                    }
                }
                if (this.log.isTraceEnabled()) {
                    if (key != null) {
                        this.trace("got next key={}", ByteUtil.toString((byte[])key));
                    } else {
                        this.trace("got end of iteration after {}ms", (System.nanoTime() - this.startTime) / 1000000L);
                    }
                }
                cachingKVStore = CachingKVStore.this;
                synchronized (cachingKVStore) {
                    KVRange neighbor;
                    double keySpaceValue;
                    KVRange overlap;
                    byte[] extentMin;
                    assert (CachingKVStore.this.sanityCheck());
                    if (this.log.isTraceEnabled()) {
                        this.trace("handle key={} ranges={}", ByteUtil.toString((byte[])key), CachingKVStore.this.ranges);
                    }
                    if (this.isStopped()) {
                        if (this.log.isTraceEnabled()) {
                            this.trace("stopped", new Object[0]);
                        }
                        break;
                    }
                    assert (this.range.getLoader(this.reverse) == this);
                    assert (this.future != null);
                    byte[] byArray2 = !this.reverse ? this.getBase() : (extentMin = key != null ? key : this.limit);
                    byte[] extentMax = this.reverse ? this.getBase() : (key != null ? ByteUtil.getNextKey((byte[])key) : this.limit);
                    boolean stopLoading = key == null;
                    Iterator<KVRange> i = CachingKVStore.this.ranges.tailSet(CachingKVStore.this.key(extentMin), true).iterator();
                    while (i.hasNext()) {
                        KVRange contained = i.next();
                        assert (ByteUtil.compare((byte[])contained.getMin(), (byte[])extentMin) >= 0);
                        if (KeyRange.compare((byte[])contained.getMax(), (byte[])extentMax) > 0) break;
                        if (contained == this.range) {
                            assert (KeyRange.compare((byte[])this.range.getMin(), (byte[])this.range.getMax()) == 0);
                            continue;
                        }
                        assert (contained.containsOnlyKeyInRange(key, ByteUtil.EMPTY, null)) : "contained=" + contained + " key=" + ByteUtil.toString((byte[])key);
                        if (this.log.isTraceEnabled()) {
                            this.trace("discarding contained range {}", contained);
                        }
                        i.remove();
                        CachingKVStore.this.discard(contained, false);
                    }
                    if (this.reverse) {
                        overlap = this.last(CachingKVStore.this.ranges.headSet(CachingKVStore.this.key(extentMin), false));
                        assert (overlap == null || overlap == this.range || KeyRange.compare((byte[])overlap.getMin(), (byte[])extentMin) < 0);
                        if (overlap != null && overlap != this.range && KeyRange.compare((byte[])overlap.getMax(), (byte[])extentMin) > 0) {
                            if (this.log.isTraceEnabled()) {
                                this.trace("overlap range {}, truncate {} -> {}", overlap, ByteUtil.toString((byte[])extentMin), ByteUtil.toString((byte[])overlap.getMax()));
                            }
                            assert (overlap.containsOnlyKeyInRange(key, extentMin, extentMax));
                            key = null;
                            extentMin = overlap.getMax();
                            stopLoading = true;
                        }
                    } else {
                        overlap = (KVRange)this.last(extentMax != null ? CachingKVStore.this.ranges.headSet(CachingKVStore.this.key(extentMax), false) : CachingKVStore.this.ranges);
                        assert (overlap == null || overlap == this.range || KeyRange.compare((byte[])overlap.getMin(), (byte[])extentMax) < 0);
                        if (overlap != null && overlap != this.range) {
                            assert (KeyRange.compare((byte[])overlap.getMax(), (byte[])extentMax) > 0);
                            if (this.log.isTraceEnabled()) {
                                this.trace("overlap range {}, truncate {} -> {}", overlap, ByteUtil.toString((byte[])extentMax), ByteUtil.toString((byte[])overlap.getMin()));
                            }
                            assert (overlap.containsOnlyKeyInRange(key, extentMin, extentMax));
                            key = null;
                            extentMax = overlap.getMin();
                            stopLoading = true;
                        }
                    }
                    if (key != null) {
                        if (this.log.isTraceEnabled()) {
                            this.trace("adding key {} to {}", ByteUtil.toString((byte[])key), this.range);
                        }
                        this.range.add(key, val);
                    }
                    if (this.reverse) {
                        if (this.log.isTraceEnabled()) {
                            this.trace("setting min of {} to {}", this.range, ByteUtil.toString((byte[])extentMin));
                        }
                        this.range.setMin(extentMin);
                    } else {
                        if (this.log.isTraceEnabled()) {
                            this.trace("setting max of {} to {}", this.range, ByteUtil.toString((byte[])extentMax));
                        }
                        this.range.setMax(extentMax);
                    }
                    long now = System.nanoTime();
                    double d = this.reverse ? ByteUtil.toDouble((byte[])extentMin) : (keySpaceValue = extentMax != null ? ByteUtil.toDouble((byte[])extentMax) : 1.0);
                    if (numKeysLoaded++ == 0) {
                        lastKeySpaceValue = keySpaceValue;
                        lastMeasureTime = now;
                        if (this.log.isTraceEnabled()) {
                            this.trace("first key @ {}ms", (now - this.startTime) / 1000000L);
                        }
                    } else {
                        double nanosElapsed = now - lastMeasureTime;
                        double keySpaceDiff = this.reverse ? lastKeySpaceValue - keySpaceValue : keySpaceValue - lastKeySpaceValue;
                        double oldArrivalRate = this.arrivalRate.get();
                        this.arrivalRate.add(keySpaceDiff / nanosElapsed);
                        if (this.log.isTraceEnabled()) {
                            this.trace("key #{} dif={} elapsed={} oldRate={} newRate={}", numKeysLoaded, keySpaceDiff, nanosElapsed, oldArrivalRate, this.arrivalRate.get());
                        }
                    }
                    KVRange mergedRange = null;
                    if (this.reverse && extentMin.length > 0) {
                        neighbor = this.last(CachingKVStore.this.ranges.headSet(CachingKVStore.this.key(extentMin), false));
                        if (neighbor != null && ByteUtil.compare((byte[])neighbor.getMax(), (byte[])extentMin) == 0) {
                            if (neighbor.isPrimordial()) {
                                assert (neighbor.isEmpty());
                                if (this.log.isTraceEnabled()) {
                                    this.trace("merging primordial {} -> {}", neighbor, this.range);
                                }
                                CachingKVStore.this.discard(neighbor, true);
                            } else {
                                if (this.log.isTraceEnabled()) {
                                    this.trace("merging {} + {}", neighbor, this.range);
                                }
                                mergedRange = neighbor.merge(this.range);
                                if (this.log.isTraceEnabled()) {
                                    this.trace("result of merge: {}", mergedRange);
                                }
                                CachingKVStore.this.discard(neighbor, true);
                                CachingKVStore.this.discard(this.range, true);
                                CachingKVStore.this.ranges.add(mergedRange);
                                mergedRange.getLruEntry().attachAfter(CachingKVStore.this.lru);
                                assert (mergedRange.sanityCheck());
                            }
                            assert (CachingKVStore.this.sanityCheck());
                        }
                    } else if (!this.reverse && extentMax != null && (neighbor = this.first(CachingKVStore.this.ranges.tailSet(CachingKVStore.this.key(extentMax), true))) != null && ByteUtil.compare((byte[])neighbor.getMin(), (byte[])extentMax) == 0) {
                        if (neighbor.isPrimordial()) {
                            assert (neighbor.isEmpty());
                            if (this.log.isTraceEnabled()) {
                                this.trace("merging primordial {} <- {}", this.range, neighbor);
                            }
                            CachingKVStore.this.discard(neighbor, true);
                        } else {
                            if (this.log.isTraceEnabled()) {
                                this.trace("merging {} + {}", this.range, neighbor);
                            }
                            mergedRange = this.range.merge(neighbor);
                            if (this.log.isTraceEnabled()) {
                                this.trace("result of merge: {}", mergedRange);
                            }
                            CachingKVStore.this.discard(this.range, true);
                            CachingKVStore.this.discard(neighbor, true);
                            CachingKVStore.this.ranges.add(mergedRange);
                            mergedRange.getLruEntry().attachAfter(CachingKVStore.this.lru);
                            assert (mergedRange.sanityCheck());
                        }
                        assert (CachingKVStore.this.sanityCheck());
                    }
                    if (stopLoading |= mergedRange != null || this.range.getTotalBytes() > CachingKVStore.this.maxRangeBytes) {
                        this.stop();
                        assert (CachingKVStore.this.sanityCheck());
                        break;
                    }
                    assert (this.future != null);
                    this.future.complete(null);
                    this.future = new CompletableFuture();
                }
            }
        }

        private void debug(String msg, Object ... params) {
            this.log.debug(this + ": " + msg, params);
        }

        private void trace(String msg, Object ... params) {
            this.log.trace(this + ": " + msg, params);
        }

        private <T> T first(SortedSet<T> set) {
            try {
                return set.first();
            }
            catch (NoSuchElementException e) {
                return null;
            }
        }

        private <T> T last(SortedSet<T> set) {
            try {
                return set.last();
            }
            catch (NoSuchElementException e) {
                return null;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public String toString() {
            CachingKVStore cachingKVStore = CachingKVStore.this;
            synchronized (cachingKVStore) {
                return "Loader[" + (this.reverse ? "reverse" : "forward") + ",base=" + ByteUtil.toString((byte[])this.getBase()) + ",limit=" + ByteUtil.toString((byte[])this.limit) + "]";
            }
        }
    }
}

