package org.kevoree.modeling.util.maths.matrix.solvers;

import org.kevoree.modeling.util.maths.matrix.CommonOps;
import org.kevoree.modeling.util.maths.matrix.DenseMatrix64F;
import org.kevoree.modeling.util.maths.matrix.solvers.decomposition.BidiagonalDecomposition;
import org.kevoree.modeling.util.maths.matrix.solvers.decomposition.BidiagonalDecompositionRow_D64;
import org.kevoree.modeling.util.maths.matrix.solvers.decomposition.BidiagonalDecompositionTall_D64;

public class SvdImplicitQrDecompose_D64 {

    private int numRows;
    private int numCols;

    // dimensions of transposed matrix
    private int numRowsT;
    private int numColsT;

    // if true then it can use the special Bidiagonal decomposition
    private boolean canUseTallBidiagonal;

    // If U is not being computed and the input matrix is 'tall' then a special decomposition decomposition
    // can be used which is faster.
    private BidiagonalDecomposition<DenseMatrix64F> bidiag;
    private SvdImplicitQrAlgorithm qralg = new SvdImplicitQrAlgorithm();

    double diag[];
    double off[];

    private DenseMatrix64F Ut;
    private DenseMatrix64F Vt;

    private double singularValues[];
    private int numSingular;

    // compute a compact SVD
    private boolean compact;
    // What is actually computed
    private boolean computeU;
    private boolean computeV;

    // What the user requested to be computed
    // If the transpose is computed instead then what is actually computed is swapped
    private boolean prefComputeU;
    private boolean prefComputeV;

    // Should it compute the transpose instead
    private boolean transposed;

    // Either a copy of the input matrix or a copy of it transposed
    private DenseMatrix64F A_mod = new DenseMatrix64F(1,1);

    /**
     * Configures the class
     *
     * @param compact Compute a compact SVD
     * @param computeU If true it will compute the U matrix
     * @param computeV If true it will compute the V matrix
     * @param canUseTallBidiagonal If true then it can choose to use a tall Bidiagonal decomposition to improve runtime performance.
     */
    public SvdImplicitQrDecompose_D64(boolean compact, boolean computeU, boolean computeV,
                                      boolean canUseTallBidiagonal)
    {
        this.compact = compact;
        this.prefComputeU = computeU;
        this.prefComputeV = computeV;
        this.canUseTallBidiagonal = canUseTallBidiagonal;
    }

    public double[] getSingularValues() {
        return singularValues;
    }


    public int numberOfSingularValues() {
        return numSingular;
    }


    public boolean isCompact() {
        return compact;
    }


    public DenseMatrix64F getU( DenseMatrix64F U , boolean transpose) {
        if( !prefComputeU )
            throw new RuntimeException("As requested U was not computed.");
        if( transpose ) {
            if( U == null )
                return Ut;
            else if( U.numRows != Ut.numRows || U.numCols != Ut.numCols )
                throw new RuntimeException("Unexpected shape of U");

            U.setMatrix(Ut);
        } else {
            if( U == null )
                U = new DenseMatrix64F(Ut.numCols,Ut.numRows);
            else if( U.numRows != Ut.numCols || U.numCols != Ut.numRows )
                throw new RuntimeException("Unexpected shape of U");

            CommonOps.transposeMatrix(Ut, U);
        }

        return U;
    }


    public DenseMatrix64F getV( DenseMatrix64F V , boolean transpose ) {
        if( !prefComputeV )
            throw new RuntimeException("As requested V was not computed.");
        if( transpose ) {
            if( V == null )
                return Vt;
            else if( V.numRows != Vt.numRows || V.numCols != Vt.numCols )
                throw new RuntimeException("Unexpected shape of V");

            V.setMatrix(Vt);
        } else {
            if( V == null )
                V = new DenseMatrix64F(Vt.numCols,Vt.numRows);
            else if( V.numRows != Vt.numCols || V.numCols != Vt.numRows )
                throw new RuntimeException("Unexpected shape of V");
            CommonOps.transposeMatrix(Vt,V);
        }

        return V;
    }


    public DenseMatrix64F getW( DenseMatrix64F W ) {
        int m = compact ? numSingular : numRows;
        int n = compact ? numSingular : numCols;

        if( W == null )
            W = new DenseMatrix64F(m,n);
        else {
            W.reshapeBoolean(m,n, false);
            W.zero();
        }

        for( int i = 0; i < numSingular; i++ ) {
            W.set(i,i, singularValues[i]);
        }

        return W;
    }


    public boolean decompose(DenseMatrix64F orig) {
        if( !setup(orig) )
            return false;

        if (bidiagonalization(orig))
            return false;

        if( computeUWV() )
            return false;

        // make sure all the singular values or positive
        makeSingularPositive();

        // if transposed undo the transposition
        undoTranspose();

        return true;
    }


    public boolean inputModified() {
        return false;
    }

    private boolean bidiagonalization(DenseMatrix64F orig) {
        // change the matrix to decomposition form
        if( transposed ) {
            A_mod.reshapeBoolean(orig.numCols, orig.numRows, false);
            CommonOps.transposeMatrix(orig,A_mod);
        } else {
            A_mod.reshapeBoolean(orig.numRows, orig.numCols, false);
            A_mod.setMatrix(orig);
        }
        return !bidiag.decompose(A_mod);
    }

    /**
     * If the transpose was computed instead do some additional computations
     */
    private void undoTranspose() {
        if( transposed ) {
            DenseMatrix64F temp = Vt;
            Vt = Ut;
            Ut = temp;
        }
    }

    /**
     * Compute singular values and U and V at the same time
     */
    private boolean computeUWV() {
        bidiag.getDiagonal(diag,off);
        qralg.setMatrix(numRowsT,numColsT,diag,off);

//        long pointA = System.currentTimeMillis();
        // compute U and V matrices
        if( computeU )
            Ut = bidiag.getU(Ut,true,compact);
        if( computeV )
            Vt = bidiag.getV(Vt,true,compact);

        qralg.setFastValues(false);
        if( computeU )
            qralg.setUt(Ut);
        else
            qralg.setUt(null);
        if( computeV )
            qralg.setVt(Vt);
        else
            qralg.setVt(null);

//        long pointB = System.currentTimeMillis();

        boolean ret = !qralg.process();

//        long pointC = System.currentTimeMillis();
//        System.out.println("  compute UV "+(pointB-pointA)+"  QR = "+(pointC-pointB));

        return ret;
    }

    private boolean setup(DenseMatrix64F orig) {
        transposed = orig.numCols > orig.numRows;

        // flag what should be computed and what should not be computed
        if( transposed ) {
            computeU = prefComputeV;
            computeV = prefComputeU;
            numRowsT = orig.numCols;
            numColsT = orig.numRows;
        } else {
            computeU = prefComputeU;
            computeV = prefComputeV;
            numRowsT = orig.numRows;
            numColsT = orig.numCols;
        }

        numRows = orig.numRows;
        numCols = orig.numCols;

        if( numRows == 0 || numCols == 0 )
            return false;

        if( diag == null || diag.length < numColsT ) {
            diag = new double[ numColsT ];
            off = new double[ numColsT-1 ];
        }

        // if it is a tall matrix and U is not needed then there is faster decomposition algorithm
        if( canUseTallBidiagonal && numRows > numCols * 2 && !computeU ) {
            if( bidiag == null || !(bidiag instanceof BidiagonalDecompositionTall_D64) ) {
                bidiag = new BidiagonalDecompositionTall_D64();
            }
        } else if( bidiag == null || !(bidiag instanceof BidiagonalDecompositionRow_D64) ) {
            bidiag = new BidiagonalDecompositionRow_D64(1);
        }

        return true;
    }

    /**
     * With the QR algorithm it is possible for the found singular values to be negative.  This
     * makes them all positive by multiplying it by a diagonal matrix that has
     */
    private void makeSingularPositive() {
        numSingular = qralg.getNumberOfSingularValues();
        singularValues = qralg.getSingularValues();

        for( int i = 0; i < numSingular; i++ ) {
            double val = qralg.getSingularValue(i);

            if( val < 0 ) {
                singularValues[i] = 0.0d - val;

                if( computeU ) {
                    // compute the results of multiplying it by an element of -1 at this location in
                    // a diagonal matrix.
                    int start = i* Ut.numCols;
                    int stop = start+ Ut.numCols;

                    for( int j = start; j < stop; j++ ) {
                        Ut.setValueAtIndex(j, 0.0d - Ut.getValueAtIndex(j));
                    }
                }
            } else {
                singularValues[i] = val;
            }
        }
    }


    public int getNumRows() {
        return numRows;
    }


    public int getNumCols() {
        return numCols;
    }
}