/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core;

import jmms.core.model.*;
import leap.lang.Strings;
import leap.lang.meta.*;
import leap.web.api.spec.InvalidSpecException;
import leap.web.api.spec.swagger.SwaggerJsonWriter;
import leap.web.api.spec.swagger.SwaggerSpecReader;
import leap.web.api.spec.swagger.SwaggerType;

public class Types {

    private static SwaggerSpecReader swaggerReader = new SwaggerSpecReader();
    private static SwaggerJsonWriter swaggerWriter = new SwaggerJsonWriter();

    public static MComplexTypeBuilder convert(MetaModel t) {
        MComplexTypeBuilder ct = new MComplexTypeBuilder();
        ct.setName(t.getName());
        ct.setTitle(t.getTitle());
        ct.setSummary(t.getSummary());
        ct.setDescription(t.getDescription());

        for(MetaProperty mp : t.getProperties().values()) {
            MPropertyBuilder p = new MPropertyBuilder();
            p.setName(mp.getName());
            p.setType(new MUnresolvedTypeRef(mp.getType(), mp.getType()));
            p.setTitle(mp.getTitle());
            p.setSummary(mp.getSummary());
            p.setDescription(mp.getDescription());
            ct.addProperty(p);
        }

        return ct;
    }

    public static String toTypeName(MType type) {
        if(type.isSimpleType()) {
            SwaggerType st = swaggerWriter.convertSimpleType(type.asSimpleType());
            if(null == st) {
                throw new IllegalStateException("Unsupported type '" + type + "'");
            }
            return st.name().toLowerCase();
        }

        if(type.isCollectionType()) {
            MType elementType = type.asCollectionType().getElementType();
            return "array<" + toTypeName(elementType) + ">";
        }

        if(type.isComplexType()) {
            return type.asComplexType().getName();
        }

        if(type.isTypeRef()) {
            return type.asTypeRef().getRefTypeName();
        }

        if(type.isDictionaryType()) {
            return "map<string," + toTypeName(type.asDictionaryType().getValueType()) + ">";
        }

        if(type.isObjectType()) {
            return "object";
        }

        if(type.isVoidType()) {
            return "";
        }

        throw new IllegalStateException("Unsupported type '" + type + "'");
    }

    public static MType resolveType(MetaApi api, MType type) {
        if(!hasUnresolvedType(type)) {
            return type;
        }
        return doResolveType(api, type);
    }

    private static MType doResolveType(MetaApi api, MType type) {
        if(type instanceof MUnresolvedTypeRef) {
            return resolveType(api, ((MUnresolvedTypeRef) type).getRefTypeName());
        }

        if(type instanceof MCollectionType) {
            return new MCollectionType(doResolveType(api, type.asCollectionType().getElementType()));
        }

        if(type instanceof MDictionaryType) {
            return new MDictionaryType(type.asDictionaryType().getKeyType(), doResolveType(api, type.asDictionaryType().getValueType()));
        }

        throw new IllegalStateException("Unsupported type '" + type + "'");
    }

    private static boolean hasUnresolvedType(MType type) {
        if(type instanceof MUnresolvedTypeRef) {
            return true;
        }

        if(type.isCollectionType()) {
            return hasUnresolvedType(type.asCollectionType().getElementType());
        }

        if(type.isDictionaryType()) {
            return hasUnresolvedType(type.asDictionaryType().getValueType());
        }

        return false;
    }

    public static boolean isSpecialType(String typeName) {
        if(typeName.endsWith("[]")) {
            return true;
        }
        if("int".equalsIgnoreCase(typeName)) {
            return true;
        }
        if("map".equalsIgnoreCase(typeName) || Strings.startsWithIgnoreCase(typeName, "map<")) {
            return true;
        }
        if(Strings.startsWithIgnoreCase(typeName, "array<")) {
            return true;
        }
        if(Strings.startsWithIgnoreCase(typeName, "partial<")) {
            return true;
        }
//        if(Strings.startsWithIgnoreCase(typeName, "as<")) {
//            return true;
//        }
        return false;
    }

    public static MType resolveType(MetaApi api, String typeName) {
        if(typeName.endsWith("[]")) {
            String elementTypeName = typeName.substring(0, typeName.length() - 2);

            MType elementType = resolveType(api, elementTypeName);

            return new MCollectionType(elementType);
        }

        if("object".equalsIgnoreCase(typeName)) {
            return MObjectType.TYPE;
        }

        //hack swagger type
        if("int".equalsIgnoreCase(typeName)) {
            return MSimpleTypes.INTEGER;
        }

        if(typeName.equalsIgnoreCase("map")) {
            return new MDictionaryType(MSimpleTypes.STRING, MObjectType.TYPE);
        }

        if(typeName.equalsIgnoreCase("array")) {
            return new MCollectionType(MObjectType.TYPE);
        }

        int index = typeName.indexOf('<');
        if(index > 0) {
            if(!typeName.endsWith(">")) {
                throw new IllegalStateException("Unsupported type '" + typeName + "', should be 'map<string,typeName>'");
            }

            String prefix = typeName.substring(0,index).trim();
            String s = typeName.substring(index+1,typeName.length()-1);
            String[] parts = Strings.split(s, ',');

            if(prefix.equalsIgnoreCase("map")) {
                String valueType;
                if(parts.length == 0) {
                    valueType = "object";
                }else if(parts.length == 1) {
                    valueType = parts[0];
                }else if(parts.length == 2) {
                    if(!parts[0].equalsIgnoreCase("string")) {
                        throw new IllegalStateException("The key type of '" + typeName + "' must be 'string'");
                    }
                    valueType = parts[1];
                }else {
                    throw new IllegalStateException("Invalid map type '" + typeName + "'");
                }
                return new MDictionaryType(MSimpleTypes.STRING, resolveType(api, valueType));
            }

            if(prefix.equalsIgnoreCase("array")) {
                String elementType;
                if(parts.length == 0) {
                    elementType = "object";
                }else if(parts.length == 1) {
                    elementType = parts[0];
                }else {
                    throw new IllegalStateException("Invalid array type '" + typeName + "'");
                }
                return new MCollectionType(resolveType(api, elementType));
            }

            if(prefix.equalsIgnoreCase("partial")) {
                if(parts.length != 1) {
                    throw new IllegalStateException("Invalid partial type '" + typeName + "', must be partial<EntityOrModelName>");
                }

                String name = parts[0];

                MetaEntity entity = api.getEntity(name);
                if(null != entity) {
                    return new MComplexTypeRef(entity.getName(), true);
                }

                MetaModel m = api.getModel(name);
                if(null != m) {
                    return new MComplexTypeRef(m.getName(), true);
                }

                throw new IllegalStateException("Unsupported type '" + name + "' at '" + typeName + "'");
            }

            //as<model.property> || as<entity.field>
//            if(prefix.equalsIgnoreCase("as")) {
//                if(parts.length != 1) {
//                    throw new IllegalStateException("Invalid type '" + typeName + "', must be as<model.property> or as<entity.field>");
//                }
//
//                String[] modelAndProp = Strings.split(parts[0], ".");
//                if(modelAndProp.length != 2) {
//                    throw new IllegalStateException("Invalid type '" + typeName + "', must be as<model.property> or as<entity.field>");
//                }
//
//                String modelOrEntityName   = modelAndProp[0];
//                String propertyOrFieldName = modelAndProp[1];
//
//                MetaEntity entity = api.getEntity(modelOrEntityName);
//                if(null != entity) {
//                    MetaField field = entity.getField(propertyOrFieldName);
//                    if(null == field) {
//                        throw new IllegalStateException("Field '" + propertyOrFieldName + "' not exists at entity '" + modelOrEntityName + "', check '" + typeName + "'");
//                    }
//                    return null != field.getResolvedType() ? field.getResolvedType() : resolveType(api, field.getType());
//                }
//
//                MetaModel model = api.getModel(modelOrEntityName);
//                if(null != model) {
//                    MetaProperty property = model.getProperty(propertyOrFieldName);
//                    if(null == property) {
//                        throw new IllegalStateException("Property '" + propertyOrFieldName + "' not exists at model '" + modelOrEntityName + "', check '" + typeName + "'");
//                    }
//                    return null != property.getResolvedType() ? property.getResolvedType() : resolveType(api, property.getType());
//                }
//
//                throw new IllegalStateException("Model or Entity '" + modelOrEntityName + "' not exists, check '" + typeName + "'");
//            }

            throw new IllegalStateException("Unsupported type '" + typeName);
        }

        MSimpleType simpleType = null;
        try {
            simpleType = swaggerReader.readSimpleType(typeName);
        }catch (InvalidSpecException e) {
            //do nothing.
        }

        if(null != simpleType) {
            return simpleType;
        }else{
            MetaEntity entity = api.getEntity(typeName);
            if(null != entity) {
                return new MComplexTypeRef(entity.getName());
            }

            MetaModel m = api.getModel(typeName);
            if(null != m) {
                return new MComplexTypeRef(m.getName());
            }

            throw new IllegalStateException("Unsupported type '" + typeName + "'");
        }
    }

    protected Types() {

    } 

}
