/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.parser.OpsParser;
import leap.core.validation.annotations.Required;
import leap.lang.Strings;
import leap.lang.collection.WrappedCaseInsensitiveMap;
import leap.lang.exception.ObjectExistsException;
import leap.lang.json.JsonIgnore;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class MetaApi extends MetaObjNamed {

    protected String         version;
    protected String         basePath;
    protected MetaDataSource dataSource;
    protected String         dataSourceName;
    protected boolean        migrateDatabase;
    protected Boolean        upgradeSchema;
    protected Boolean        cascadeDelete;
    protected boolean        readOnly;
    protected Integer        defaultPageSize;
    protected String[]       includedEntities;
    protected String[]       excludedEntities;
    protected String         generates;
    protected MetaSecurity   security;
    protected GlobalConfig   global;

    protected Map<String, MetaField>        domains       = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaEntity>       entities      = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaModel>        models        = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaSql>          sqls          = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaTag>          tags          = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaPermission>   permissions   = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaOptionSet>    optionSets    = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaParameterSet> parameterSets = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaResponseSet>  responseSets  = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaQueryFilterSet>  filterSets = new WrappedCaseInsensitiveMap<>();
    protected Map<String, MetaService>      services      = new WrappedCaseInsensitiveMap<>();
    protected List<MetaOperation>           operations    = new ArrayList<>();

    @JsonIgnore
    protected OpsParser.Ops ops = OpsParser.parse("none");

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    @Required
    public String getBasePath() {
        return basePath;
    }

    public void setBasePath(String basePath) {
        this.basePath = basePath;
    }

    public MetaDataSource getDataSource() {
        return dataSource;
    }

    public void setDataSource(MetaDataSource dataSource) {
        this.dataSource = dataSource;
    }

    public String getDataSourceName() {
        return dataSourceName;
    }

    public void setDataSourceName(String dataSourceName) {
        this.dataSourceName = dataSourceName;
    }

    public boolean isMigrateDatabase() {
        return migrateDatabase;
    }

    public void setMigrateDatabase(boolean migrateDatabase) {
        this.migrateDatabase = migrateDatabase;
    }

    public boolean isUpgradeSchema() {
        return null != upgradeSchema ? upgradeSchema : isMigrateDatabase();
    }

    public Boolean getUpgradeSchema() {
        return upgradeSchema;
    }

    public void setUpgradeSchema(Boolean upgradeSchema) {
        this.upgradeSchema = upgradeSchema;
    }

    public Boolean getCascadeDelete() {
        return cascadeDelete;
    }

    public void setCascadeDelete(Boolean cascadeDelete) {
        this.cascadeDelete = cascadeDelete;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public void setReadOnly(boolean readOnly) {
        this.readOnly = readOnly;
    }

    public Integer getDefaultPageSize() {
        return defaultPageSize;
    }

    public void setDefaultPageSize(Integer defaultPageSize) {
        this.defaultPageSize = defaultPageSize;
    }

    public MetaSecurity getSecurity() {
        return security;
    }

    public void setSecurity(MetaSecurity security) {
        this.security = security;
    }

    public GlobalConfig getGlobal() {
        return global;
    }

    public void setGlobal(GlobalConfig global) {
        this.global = global;
    }

    public EntityGlobal getEntityGlobal() {
        return null == global ? null : global.getEntity();
    }

    public String[] getIncludedEntities() {
        return includedEntities;
    }

    public void setIncludedEntities(String[] includedEntities) {
        this.includedEntities = includedEntities;
    }

    public String[] getExcludedEntities() {
        return excludedEntities;
    }

    public void setExcludedEntities(String[] excludedEntities) {
        this.excludedEntities = excludedEntities;
    }

    public Map<String, MetaField> getDomains() {
        return domains;
    }

    public void setDomains(Map<String, MetaField> domains) {
        this.domains = WrappedCaseInsensitiveMap.create(domains);
    }

    public MetaField getDomain(String name) {
        return domains.get(name);
    }

    public void addDomain(MetaField domain) {
        domains.put(domain.getName(), domain);
    }

    public Map<String, MetaEntity> getEntities() {
        return entities;
    }

    public void setEntities(Map<String, MetaEntity> entities) {
        this.entities = WrappedCaseInsensitiveMap.create(entities);
    }

    public MetaEntity getEntity(String name) {
        return entities.get(name);
    }

    public void addEntity(MetaEntity entity) throws ObjectExistsException {
        if (entities.containsKey(entity.getName())) {
            throw new ObjectExistsException("Entity '" + entity.getName() + "' already exists in api '" + name + "'");
        }

        entities.put(entity.getName(), entity);
    }

    public Map<String, MetaModel> getModels() {
        return models;
    }

    public void setModels(Map<String, MetaModel> models) {
        this.models = WrappedCaseInsensitiveMap.create(models);
    }

    public MetaModel getModel(String name) {
        return models.get(name);
    }

    public void addModel(MetaModel m) {
        models.put(m.getName(), m);
    }

    public List<MetaOperation> getOperations() {
        return operations;
    }

    public void setOperations(List<MetaOperation> operations) {
        this.operations = operations;
    }

    public void addOperation(MetaOperation operation) {
        operations.add(operation);
    }

    public MetaOperation getOperation(String method, String path) {
        for (MetaOperation o : operations) {
            if (Strings.equalsIgnoreCase(o.getMethod().name(), method)) {
                if (Strings.equals(o.getPath(), path)) {
                    return o;
                }
            }
        }
        return null;
    }

    public Map<String, MetaSql> getSqls() {
        return sqls;
    }

    public void setSqls(Map<String, MetaSql> sqls) {
        this.sqls = WrappedCaseInsensitiveMap.create(sqls);
    }

    public MetaSql getSql(String key) {
        return sqls.get(key);
    }

    public void putSql(String key, MetaSql sql) {
        sqls.put(key, sql);
    }

    public Map<String, MetaTag> getTags() {
        return tags;
    }

    public void setTags(Map<String, MetaTag> tags) {
        this.tags = WrappedCaseInsensitiveMap.create(tags);
    }

    public void addTag(MetaTag tag) {
        tags.put(tag.getName(), tag);
    }

    public Map<String, MetaPermission> getPermissions() {
        return permissions;
    }

    public void setPermissions(Map<String, MetaPermission> permissions) {
        this.permissions = WrappedCaseInsensitiveMap.create(permissions);
    }

    public void addPermission(MetaPermission permission) {
        permissions.put(permission.getName(), permission);
    }


    public Map<String, MetaOptionSet> getOptionSets() {
        return optionSets;
    }

    public void setOptionSets(Map<String, MetaOptionSet> optionSets) {
        this.optionSets = WrappedCaseInsensitiveMap.create(optionSets);
    }

    public MetaOptionSet getOptionSet(String name) {
        return optionSets.get(name);
    }

    public void addOptionSet(MetaOptionSet os) {
        optionSets.put(os.getName(), os);
    }

    public Map<String, MetaParameterSet> getParameterSets() {
        return parameterSets;
    }

    public void addParameterSet(MetaParameterSet paramSet) {
        this.parameterSets.put(paramSet.getName(), paramSet);
    }

    public void setParameterSets(Map<String, MetaParameterSet> parameterSets) {
        this.parameterSets = WrappedCaseInsensitiveMap.create(parameterSets);
    }

    public Map<String, MetaResponseSet> getResponseSets() {
        return responseSets;
    }

    public void addResponseSet(MetaResponseSet responseSet) {
        this.responseSets.put(responseSet.getName(), responseSet);
    }

    public void setResponseSets(Map<String, MetaResponseSet> responseSets) {
        this.responseSets = WrappedCaseInsensitiveMap.create(responseSets);
    }

    public MetaQueryFilterSet getFilterSet(String name) {
        return filterSets.get(name);
    }

    public Map<String, MetaQueryFilterSet> getFilterSets() {
        return filterSets;
    }

    public void setFilterSets(Map<String, MetaQueryFilterSet> filterSets) {
        this.filterSets = WrappedCaseInsensitiveMap.create(filterSets);
    }

    public void addFilterSet(MetaQueryFilterSet qfs) {
        filterSets.put(qfs.getName(), qfs);
    }

    public Map<String, MetaService> getServices() {
        return services;
    }

    public void setServices(Map<String, MetaService> services) {
        this.services = services;
    }

    public MetaService getService(String name) {
        return services.get(name);
    }

    public void addService(MetaService service) {
        services.put(service.getName(), service);
    }

    public String getGenerates() {
        return generates;
    }

    public void setGenerates(String generates) {
        this.generates = generates;
    }

    public OpsParser.Ops getOps() {
        return ops;
    }

    public void setOps(OpsParser.Ops ops) {
        this.ops = ops;
    }

    public MetaApiKey toKey() {
        return new MetaApiKey(name, version);
    }

    public String toKeyString() {
        return toKey().toKeyString();
    }

    public static class GlobalConfig {

        protected EntityGlobal entity;

        public EntityGlobal getEntity() {
            return entity;
        }

        public void setEntity(EntityGlobal entity) {
            this.entity = entity;
        }
    }

    public static class EntityGlobal {

        protected MetaSecurity security;

        public MetaSecurity getSecurity() {
            return security;
        }

        public void setSecurity(MetaSecurity security) {
            this.security = security;
        }
    }
}