/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.parser.OpsParser;
import leap.core.validation.annotations.Required;
import leap.lang.Arrays2;
import leap.lang.Strings;
import leap.lang.collection.WrappedCaseInsensitiveMap;
import leap.lang.exception.ObjectExistsException;
import leap.lang.json.JsonIgnore;

import java.util.*;

public class MetaEntity extends MetaModelBase {

    /**
     * Internal entity will not appear at swagger spec.
     */
    protected Boolean internal;

    /**
     * Is remote entity?
     */
    protected Boolean remote;

    /**
     * For remote entity. The name of {@link MetaService}.
     */
    protected String  serviceName;

    /**
     * For remote entity. The path relative to {@link MetaService#getUrl()}, starts with '/'.
     */
    protected String  servicePath;

    /**
     * The table name.
     */
    protected String  table;

    /**
     * The generated operations expr. See {@link OpsParser}.
     *
     * <p/>
     * Example: generates=none , or generates = * , or generates = create update delete
     */
    protected String  generates;

    /**
     * Supports cascade delete in generated delete operation?
     */
    protected Boolean cascadeDelete;

    /**
     * Allows anonymous access?
     */
    protected Boolean anonymous;

    /**
     * The security of entity.
     */
    protected MetaSecurity security;

    /**
     * The fields(columns) of entity.
     */
    @Required
    protected Map<String, MetaField>     fields     = new WrappedCaseInsensitiveMap<>();

    /**
     * The relations of entity.
     */
    protected Map<String, MetaRelation>  relations  = new WrappedCaseInsensitiveMap<>();

    /**
     * The alternate keys
     */
    protected Map<String, List<String>>  keys = new WrappedCaseInsensitiveMap<>();

    /**
     * The query filters of entity.
     */
    protected Map<String, MetaQueryFilter> filters = new WrappedCaseInsensitiveMap<>();

    /**
     * The query filter of entity.
     */
    protected QueryFilter queryFilter;

    /**
     * The sql view for query operation.
     */
    protected String queryView;

    @JsonIgnore
    protected MetaQueryFilterSet filterSet;

    @JsonIgnore
    protected boolean tableNameDeclared;

    @JsonIgnore
    protected String path;

    @JsonIgnore
    protected OpsParser.Ops ops = new OpsParser.Ops();

    @JsonIgnore
    protected Perms userPerms;

    @JsonIgnore
    protected Perms clientPerms;

    @JsonIgnore
    protected MetaService serviceObject;

    @JsonIgnore
    protected String serviceUrl;

    /**
     * Shallow to-one dependencies.
     */
    @JsonIgnore
    protected List<Dep> deps;

    @JsonIgnore
    protected List<MetaEntity> deepDeps;

    public String getPath() {
        return path;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public boolean isRemote() {
        return null != remote && remote;
    }

    public Boolean getRemote() {
        return remote;
    }

    public void setRemote(boolean remote) {
        this.remote = remote;
    }

    /**
     * The remote service name. Valid for remote entity only.
     */
    public String getServiceName() {
        return serviceName;
    }

    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    public String getServicePath() {
        return servicePath;
    }

    public void setServicePath(String servicePath) {
        this.servicePath = servicePath;
    }

    public String getGenerates() {
        return generates;
    }

    public void setGenerates(String generates) {
        this.generates = generates;
    }

    public boolean isCascadeDelete() {
        return null != cascadeDelete && cascadeDelete;
    }

    public Boolean getCascadeDelete() {
        return cascadeDelete;
    }

    public void setCascadeDelete(Boolean cascadeDelete) {
        this.cascadeDelete = cascadeDelete;
    }

    public MetaSecurity getSecurity() {
        return security;
    }

    public void setSecurity(MetaSecurity security) {
        this.security = security;
    }

    public boolean isInternal() {
        return null != internal && internal;
    }

    public Boolean getInternal() {
        return internal;
    }

    public void setInternal(Boolean internal) {
        this.internal = internal;
    }

    public boolean isAnonymous() {
        return null != anonymous && anonymous;
    }

    public Boolean getAnonymous() {
        return anonymous;
    }

    public void setAnonymous(Boolean anonymous) {
        this.anonymous = anonymous;
    }

    @Required
    public String getTable() {
        return table;
    }

    public void setTable(String table) {
        this.table = table;
    }

    public boolean isTableNameDeclared() {
        return tableNameDeclared;
    }

    public void setTableNameDeclared(boolean tableNameDeclared) {
        this.tableNameDeclared = tableNameDeclared;
    }

    public Map<String, MetaField> getIdentity() {
        Map<String, MetaField> map = new WrappedCaseInsensitiveMap<>();
        fields.values().forEach(f -> {
            if(f.isIdentity()) {
                map.put(f.getName(), f);
            }
        });
        return map;
    }

    @Required
    public Map<String, MetaField> getFields() {
        return fields;
    }

    public void setFields(Map<String, MetaField> fields) {
        this.fields = WrappedCaseInsensitiveMap.create(fields);
    }

    public MetaField getField(String name) {
        return fields.get(name);
    }

    public MetaField getFieldByColumn(String columnName) {
        for(MetaField field : fields.values()) {
            if(field.getColumn().equalsIgnoreCase(columnName)) {
                return field;
            }
        }
        return null;
    }

    public void addField(MetaField field) throws ObjectExistsException {
        if(fields.containsKey(field.getName())) {
            throw new ObjectExistsException("Field '" + field.getName() + "' already exists in entity '" + name + "'");
        }
        fields.put(field.getName(), field);
    }

    public Map<String, MetaRelation> getRelations() {
        return relations;
    }

    public MetaRelation findRelationByLocalFields(String... fields) {
        for(MetaRelation r : relations.values()) {
            if(r.isManyToOne() && r.getJoinFields().size() == fields.length) {
                boolean matches = true;
                for(String field : fields) {
                    if(!r.getJoinFields().stream().anyMatch(jf -> jf.local.equalsIgnoreCase(field))){
                        matches = false;
                        break;
                    }
                }
                if(matches) {
                    return r;
                }
            }
        }
        return null;
    }

    public List<MetaRelation> getTargetRelations(String targetEntity) {
        List<MetaRelation> list = new ArrayList<>();

        relations.values().forEach(r -> {
            if(r.getTargetEntity().equalsIgnoreCase(targetEntity)) {
                list.add(r);
            }
        });

        return list;
    }

    public void setRelations(Map<String, MetaRelation> relations) {
        this.relations = WrappedCaseInsensitiveMap.create(relations);
    }

    public MetaRelation getRelation(String name) {
        return relations.get(name);
    }

    public void addRelation(MetaRelation relation) throws ObjectExistsException {
        if(relations.containsKey(relation.getName())) {
            throw new ObjectExistsException("Relation '" + relation.getName() + "' already exists in entity '" + name + "'");
        }
        relations.put(relation.getName(), relation);
    }

    public Map<String, List<String>> getKeys() {
        return keys;
    }

    public void setKeys(Map<String, List<String>> keys) {
        this.keys = keys;
    }

    public void addKey(String name, List<String> fields) {
        keys.put(name, fields);
    }

    public Map<String, MetaQueryFilter> getFilters() {
        return filters;
    }

    public void setFilters(Map<String, MetaQueryFilter> filters) {
        this.filters = WrappedCaseInsensitiveMap.create(filters);
    }

    public void addFilter(MetaQueryFilter filter) {
        this.filters.put(filter.getName(), filter);
    }

    public QueryFilter getQueryFilter() {
        return queryFilter;
    }

    public void setQueryFilter(QueryFilter queryFilter) {
        this.queryFilter = queryFilter;
    }

    public String getQueryView() {
        return queryView;
    }

    public void setQueryView(String queryView) {
        this.queryView = queryView;
    }

    public MetaQueryFilterSet getFilterSet() {
        return filterSet;
    }

    public void setFilterSet(MetaQueryFilterSet filterSet) {
        this.filterSet = filterSet;
    }

    public MetaService getServiceObject() {
        return serviceObject;
    }

    public void setServiceObject(MetaService serviceObject) {
        this.serviceObject = serviceObject;
    }

    public String getServiceUrl() {
        return serviceUrl;
    }

    public void setServiceUrl(String serviceUrl) {
        this.serviceUrl = serviceUrl;
    }

    public List<Dep> getDeps() {
        return deps;
    }

    public void setDeps(List<Dep> deps) {
        this.deps = deps;
    }

    public List<MetaEntity> resolveDeepDeps() {
        if(null == deepDeps) {
            Map<String, MetaEntity> deps = new LinkedHashMap<>();
            resolveDeepDeps(this, deps);
            deepDeps = new ArrayList<>(deps.values());
        }
        return deepDeps;
    }

    protected void resolveDeepDeps(MetaEntity entity, Map<String, MetaEntity> deps) {
        for(MetaEntity.Dep dep : entity.getDeps()) {
            //todo: supports optional dep
            if(dep.isOptional()) {
                continue;
            }

            MetaEntity target = dep.getTarget();
            if(deps.containsKey(target.getName())) {
                continue;
            }

            resolveDeepDeps(target, deps);

            deps.put(target.getName(), target);
        }
    }

    public OpsParser.Ops getOps() {
        return ops;
    }

    public void setOps(OpsParser.Ops ops) {
        this.ops = ops;
    }

    public Perms getUserPerms() {
        return userPerms;
    }

    public void setUserPerms(Perms userPerms) {
        this.userPerms = userPerms;
    }

    public Perms getClientPerms() {
        return clientPerms;
    }

    public void setClientPerms(Perms clientPerms) {
        this.clientPerms = clientPerms;
    }

    public MetaField findUniqueIdentity() {
        List<MetaField> id = new ArrayList<>();
        fields.values().forEach(f -> {
            if(f.isIdentity()) {
                id.add(f);
            }
        });
        if(id.size() == 1) {
            return id.get(0);
        }else{
            return null;
        }
    }

    public void applyExtension(MetaEntity ex) {
        MetaUtils.applySimpleExtension(ex, this);

        if(null != ex.getSecurity()) {
            this.security = ex.getSecurity();
        }

        //fields
        ex.getFields().forEach((k,f) -> {
            MetaField target = fields.get(k);
            if(null != target) {
                target.applyExtension(f);
            }else {
                addField(f);
            }
        });

        //relations
        ex.getRelations().forEach((k, r) -> {
            MetaRelation target = relations.get(k);
            if(null != target) {
                throw new IllegalStateException("Can't extend exists relation");
            }
            addRelation(r);
        });

        //keys
        keys.putAll(ex.getKeys());

        //filters
        filters.putAll(ex.getFilters());
    }

    public static final class QueryFilter {

        protected String  expr;
        protected Boolean find;

        public String getExpr() {
            return expr;
        }

        public void setExpr(String expr) {
            this.expr = expr;
        }

        public Boolean getFind() {
            return find;
        }

        public void setFind(Boolean find) {
            this.find = find;
        }
    }

    public static final class Dep {
        protected final MetaRelation relation;
        protected final MetaEntity   target;

        public Dep(MetaRelation relation, MetaEntity target) {
            this.relation = relation;
            this.target = target;
        }

        public MetaRelation getRelation() {
            return relation;
        }

        public MetaEntity getTarget() {
            return target;
        }

        public boolean isOptional() {
            return null != relation.getOptional() && relation.getOptional();
        }
    }

    public static final class Perms {
        public static final String PLACE_HOLDER  = "{Entity}";

        public static final String READ          = "Read";
        public static final String WRITE         = "Write";
        public static final String CREATE        = "Create";
        public static final String UPDATE        = "Update";
        public static final String DELETE        = "Delete";
        public static final String FIND          = "Find";
        public static final String QUERY         = "Query";

        public static final String ENTITY_READ   = PLACE_HOLDER + "." + READ;
        public static final String ENTITY_WRITE  = PLACE_HOLDER + "." + WRITE;
        public static final String ENTITY_CREATE = PLACE_HOLDER + "." + CREATE;
        public static final String ENTITY_UPDATE = PLACE_HOLDER + "." + UPDATE;
        public static final String ENTITY_DELETE = PLACE_HOLDER + "." + DELETE;
        public static final String ENTITY_FIND   = PLACE_HOLDER + "." + FIND;
        public static final String ENTITY_QUERY  = PLACE_HOLDER + "." + QUERY;

        private static String[] DEFAULTS = new String[]{
                                                READ, WRITE, CREATE, UPDATE, DELETE, FIND, QUERY,
                                                ENTITY_READ, ENTITY_WRITE,
                                                ENTITY_CREATE, ENTITY_UPDATE, ENTITY_DELETE,
                                                ENTITY_FIND, ENTITY_QUERY};

        private List<String> read   = new ArrayList<>();
        private List<String> write  = new ArrayList<>();
        private List<String> create = new ArrayList<>();
        private List<String> update = new ArrayList<>();
        private List<String> delete = new ArrayList<>();
        private List<String> find   = new ArrayList<>();
        private List<String> query  = new ArrayList<>();

        public List<String> getRead() {
            return read;
        }

        public List<String> getWrite() {
            return write;
        }

        public List<String> getCreate() {
            return create;
        }

        public List<String> getUpdate() {
            return update;
        }

        public List<String> getDelete() {
            return delete;
        }

        public List<String> getFind() {
            return find;
        }

        public List<String> getQuery() {
            return query;
        }

        public String[] getReadScope() {
            return read.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getWriteScope() {
            return write.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getCreateScope() {
            List<String> a = create.isEmpty() ? write : create;
            return a.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getUpdateScope() {
            List<String> a = update.isEmpty() ? write : update;
            return a.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getDeleteScope() {
            List<String> a = delete.isEmpty() ? write : delete;
            return a.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getFindScope() {
            List<String> a = find.isEmpty() ? read : find;
            return a.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getQueryScope() {
            List<String> a = query.isEmpty() ? read : query;
            return a.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public String[] getAllScope() {
            Set<String> scope = new LinkedHashSet<>();
            scope.addAll(read);
            scope.addAll(write);
            scope.addAll(create);
            scope.addAll(update);
            scope.addAll(delete);
            scope.addAll(find);
            scope.addAll(query);
            return scope.toArray(Arrays2.EMPTY_STRING_ARRAY);
        }

        public Perms resolve(MetaEntity entity) {
            String name = entity.getName();

            for(int i=0;i<read.size();i++) {
                read.set(i, Strings.replaceIgnoreCase(read.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<write.size();i++) {
                write.set(i, Strings.replaceIgnoreCase(write.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<create.size();i++) {
                create.set(i, Strings.replaceIgnoreCase(create.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<update.size();i++) {
                update.set(i, Strings.replaceIgnoreCase(update.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<delete.size();i++) {
                delete.set(i, Strings.replaceIgnoreCase(delete.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<find.size();i++) {
                find.set(i, Strings.replaceIgnoreCase(find.get(i), PLACE_HOLDER, name));
            }

            for(int i=0;i<query.size();i++) {
                query.set(i, Strings.replaceIgnoreCase(query.get(i), PLACE_HOLDER, name));
            }

            return this;
        }

        public Perms parse(String p) {
            String[] parts = Strings.split(p, '=');

            if(parts.length == 1) {
                String type = parts[0];
                String code = perm(type);
                if(type.equalsIgnoreCase(READ) || type.equalsIgnoreCase(ENTITY_READ)) {
                    this.read.add(code);
                }else if(code.equalsIgnoreCase(WRITE) || type.equalsIgnoreCase(ENTITY_WRITE)) {
                    this.write.add(code);
                }else if(code.equalsIgnoreCase(CREATE) || type.equalsIgnoreCase(ENTITY_CREATE)) {
                    this.create.add(code);
                }else if(code.equalsIgnoreCase(UPDATE) || type.equalsIgnoreCase(ENTITY_UPDATE)) {
                    this.update.add(code);
                }else if(code.equalsIgnoreCase(DELETE) || type.equalsIgnoreCase(ENTITY_DELETE)) {
                    this.delete.add(code);
                }else if(code.equalsIgnoreCase(FIND) || type.equalsIgnoreCase(ENTITY_FIND)) {
                    this.find.add(code);
                }else if(code.equalsIgnoreCase(QUERY) || type.equalsIgnoreCase(ENTITY_QUERY)) {
                    this.query.add(code);
                }else {
                    this.read.add(code);
                    this.write.add(code);
                }
                return this;
            }

            if(parts.length == 2) {
                String type = parts[0];
                String code = parts[1];

                if(type.equalsIgnoreCase(READ)) {
                    for(String s : Strings.split(code, ',')) {
                        this.read.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(WRITE)) {
                    for(String s : Strings.split(code, ',')) {
                        this.write.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(CREATE)) {
                    for(String s : Strings.split(code, ',')) {
                        this.create.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(UPDATE)) {
                    for(String s : Strings.split(code, ',')) {
                        this.update.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(DELETE)) {
                    for(String s : Strings.split(code, ',')) {
                        this.delete.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(FIND)) {
                    for(String s : Strings.split(code, ',')) {
                        this.find.add(perm(s));
                    }
                }else if(type.equalsIgnoreCase(QUERY)) {
                    for(String s : Strings.split(code, ',')) {
                        this.query.add(perm(s));
                    }
                }else {
                    for(String s : Strings.split(code, ',')) {
                        String perm = perm(s);
                        this.read.add(perm);
                        this.write.add(perm);
                    }
                }
                return this;
            }

            throw new IllegalStateException("Invalid entity permission '" + p + "'");
        }

        private static String perm(String s) {
            for(String d : DEFAULTS) {
                if(d.equalsIgnoreCase(s)) {
                    return d;
                }
            }
            return s;
        }
    }

}