/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.parser.FieldDefParser;
import leap.core.validation.annotations.Required;
import leap.lang.Strings;
import leap.lang.convert.StringParsable;
import leap.lang.json.JsonIgnore;
import leap.lang.json.JsonName;
import leap.lang.meta.MType;
import leap.web.api.meta.model.MApiPropertyBuilder;

public class MetaField extends MetaObjFormatted implements StringParsable {

    /**
     * Column primary key
     */
    protected Boolean identity;

    /**
     * Column auto increment
     */
    protected Boolean increment;

    /**
     * Column unique
     */
    protected Boolean unique;

    /**
     * Swagger's data type, optional.
     */
    protected String dataType;

    /**
     * Column name
     */
    @Required
    protected String  column;

    /**
     * Column's jdbc type name.
     */
    @Required
    protected String  type;

    /**
     * Column length.
     */
    protected Integer length;

    /**
     * Column precision.
     */
    protected Integer precision;

    /**
     * Column scale.
     */
    protected Integer scale;

    /**
     * Column's default value.
     */
    protected String  columnDefault;

    /**
     * The fixed value of field (not column).
     */
    protected String  value;

    /**
     * The default value of field (not column).
     */
    @JsonName("default")
    protected String  defaults;

    /**
     * The name of {@link MetaFormat}.
     */
    protected String  format;

    /**
     * The name of {@link MetaOptionSet}.
     */
    protected String  options;

    /**
     * Don't generate column when persist=false.
     */
    protected Boolean persist;

    /**
     * Use the column when insert a new record?.
     */
    protected Boolean insert;

    /**
     * Use the column when update an exists record?
     */
    protected Boolean update;

    /**
     * Is creatable by user?
     */
    protected Boolean creatable;

    /**
     * Is updatable by user?
     */
    protected Boolean updatable;

    /**
     * Use the column at query and find operation?
     */
    protected Boolean selectable;

    /**
     * Use the column with aggregation functions at query operation?
     */
    protected Boolean aggregatable;

    /**
     * Use the column with groupby at query operation?
     */
    protected Boolean groupable;

    /**
     * Sortable at query operation?
     */
    protected Boolean sortable;

    /**
     * Filterable at query operation?
     */
    protected Boolean filterable;

    /**
     * Internal field will not become swagger's api property.
     */
    protected Boolean internal;

    /**
     * Creatable=false & Updatable=false will cause readOnly=true
     */
    protected Boolean readOnly;

    /**
     * Sets the validates expression. See {@link jmms.core.parser.ValidateExprParser}.
     */
    protected String  validate;

    /**
     * The meaning of field.
     */
    protected String  meaning;

    /**
     * The input type of field.
     */
    protected String  input;

    /**
     * The name of domain.
     */
    protected String domain;

    //Don't set the following fields annotated with @JsonIgnore.

    /**
     * Is foreign column?
     */
    @JsonIgnore
    protected boolean foreign;

    @JsonIgnore
    protected boolean columnNameDeclared;

    @JsonIgnore
    protected MType resolvedType;

    @JsonIgnore
    protected String referenceTo;

    @JsonIgnore
    protected String relationName;

    @JsonIgnore
    protected Boolean expandable;

    @JsonIgnore
    protected Boolean logical;

    @JsonIgnore
    protected MetaOptionSet optionSet;

    @JsonIgnore
    protected MApiPropertyBuilder apiProperty;

    public boolean isIdentity() {
        return null != identity && identity;
    }

    public Boolean getIdentity() {
        return identity;
    }

    public void setIdentity(Boolean identity) {
        this.identity = identity;
    }

    public boolean isIncrement() {
        return null != increment && increment;
    }

    public Boolean getIncrement() {
        return increment;
    }

    public void setIncrement(Boolean increment) {
        this.increment = increment;
        if(null != increment && increment) {
            this.identity = true;
        }
    }

    public boolean isUnique() {
        return null != unique && unique;
    }

    public Boolean getUnique() {
        return unique;
    }

    public void setUnique(Boolean unique) {
        this.unique = unique;
    }

    public String getDataType() {
        return dataType;
    }

    public void setDataType(String dataType) {
        this.dataType = dataType;
    }

    public String getColumn() {
        return column;
    }

    public void setColumn(String column) {
        this.column = column;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getFormat() {
        return format;
    }

    public void setFormat(String format) {
        this.format = format;
    }

    public Integer getLength() {
        return length;
    }

    public void setLength(Integer length) {
        this.length = length;
    }

    public Integer getPrecision() {
        return precision;
    }

    public void setPrecision(Integer precision) {
        this.precision = precision;
    }

    public Integer getScale() {
        return scale;
    }

    public void setScale(Integer scale) {
        this.scale = scale;
    }

    public String getColumnDefault() {
        return columnDefault;
    }

    public void setColumnDefault(String columnDefault) {
        this.columnDefault = columnDefault;
    }

    public String getValue() {
        return value;
    }

    public void setValue(String value) {
        this.value = value;
    }

    public String getDefaults() {
        return defaults;
    }

    public void setDefaults(String defaults) {
        this.defaults = defaults;
    }

    /**
     * The name of option set.
     */
    public String getOptions() {
        return options;
    }

    public void setOptions(String options) {
        this.options = options;
    }

    public boolean isPersistent() {
        return null == persist || persist;
    }

    public Boolean getPersist() {
        return persist;
    }

    public void setPersist(Boolean persist) {
        this.persist = persist;
    }

    public Boolean getInsert() {
        return insert;
    }

    public void setInsert(Boolean insert) {
        this.insert = insert;
    }

    public Boolean getUpdate() {
        return update;
    }

    public void setUpdate(Boolean update) {
        this.update = update;
    }

    public Boolean getCreatable() {
        return creatable;
    }

    public void setCreatable(Boolean creatable) {
        this.creatable = creatable;
    }

    public Boolean getUpdatable() {
        return updatable;
    }

    public void setUpdatable(Boolean updatable) {
        this.updatable = updatable;
    }

    public Boolean getSelectable() {
        return selectable;
    }

    public void setSelectable(Boolean selectable) {
        this.selectable = selectable;
    }

    public Boolean getAggregatable() {
        return aggregatable;
    }

    public void setAggregatable(Boolean aggregatable) {
        this.aggregatable = aggregatable;
    }

    public Boolean getGroupable() {
        return groupable;
    }

    public void setGroupable(Boolean groupable) {
        this.groupable = groupable;
    }

    public Boolean getSortable() {
        return sortable;
    }

    public void setSortable(Boolean sortable) {
        this.sortable = sortable;
    }

    public Boolean getFilterable() {
        return filterable;
    }

    public void setFilterable(Boolean filterable) {
        this.filterable = filterable;
    }

    public boolean isInternal() {
        return null != internal && internal;
    }

    public Boolean getInternal() {
        return internal;
    }

    public void setInternal(Boolean internal) {
        this.internal = internal;
    }

    public boolean isReadOnly() {
        return null != readOnly && readOnly;
    }

    public Boolean getReadOnly() {
        return readOnly;
    }

    public void setReadOnly(Boolean readOnly) {
        this.readOnly = readOnly;
    }

    public String getValidate() {
        return validate;
    }

    public void setValidate(String validate) {
        this.validate = validate;
    }

    public String getMeaning() {
        return meaning;
    }

    public void setMeaning(String meaning) {
        this.meaning = meaning;
    }

    public String getInput() {
        return input;
    }

    public void setInput(String input) {
        this.input = input;
    }

    public String getDomain() {
        return domain;
    }

    public void setDomain(String domain) {
        this.domain = domain;
    }

    public void tryCopyTypeTo(MetaField f) {
        if(Strings.isEmpty(f.getType())) {
            f.setType(type);
        }

        if(null == f.getLength()) {
            f.setLength(length);
        }

        if(null == f.getPrecision()) {
            f.setPrecision(precision);
        }

        if(null == f.getScale()) {
            f.setScale(scale);
        }
    }

    public void applyExtension(MetaField ex) {
        MetaUtils.applySimpleExtension(ex, this);
    }

    @Override
    public void parseString(String s) {
        FieldDefParser.parse(s, this);
    }

    public boolean isForeign() {
        return foreign;
    }

    public void setForeign(boolean foreign) {
        this.foreign = foreign;
    }

    public boolean isColumnNameDeclared() {
        return columnNameDeclared;
    }

    public void setColumnNameDeclared(boolean columnNameDeclared) {
        this.columnNameDeclared = columnNameDeclared;
    }

    public String getReferenceTo() {
        return referenceTo;
    }

    public void setReferenceTo(String referenceTo) {
        this.referenceTo = referenceTo;
    }

    public Boolean getExpandable() {
        return expandable;
    }

    public void setExpandable(Boolean expandable) {
        this.expandable = expandable;
    }

    public Boolean getLogical() {
        return logical;
    }

    public void setLogical(Boolean logical) {
        this.logical = logical;
    }

    public String getRelationName() {
        return relationName;
    }

    public void setRelationName(String relationName) {
        this.relationName = relationName;
    }

    public MType getResolvedType() {
        return resolvedType;
    }

    public void setResolvedType(MType resolvedType) {
        this.resolvedType = resolvedType;
    }

    public MetaOptionSet getOptionSet() {
        return optionSet;
    }

    public void setOptionSet(MetaOptionSet optionSet) {
        this.optionSet = optionSet;
    }

    public MApiPropertyBuilder getApiProperty() {
        return apiProperty;
    }

    public void setApiProperty(MApiPropertyBuilder apiProperty) {
        this.apiProperty = apiProperty;
    }

    public boolean isDefaultCreatable(MetaEntity e) {
        if(!Strings.isEmpty(value)) {
            return false;
        }
        if(isIncrement()) {
            return false;
        }
        if(null != insert && !insert) {
            return false;
        }
        return true;
    }

    public boolean isDefaultUpdatable(MetaEntity e) {
        if(!Strings.isEmpty(value)) {
            return false;
        }
        if(isIncrement()) {
            return false;
        }
        if(null != update && !update) {
            return false;
        }
        if(null != identity && identity) {
            return false;
        }
        return true;
    }
}