/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import leap.core.validation.annotations.Required;
import leap.lang.http.HTTP;

import java.util.*;

public class MetaOperation extends MetaObjNamed {

    protected String entityName;

    /**
     * Http method.
     */
    @Required
    protected HTTP.Method method;

    /**
     * The path of operation. starts with '/'
     */
    @Required
    protected String path;

    /**
     * Supported types: 'js', 'sql'
     */
    @Required
    protected String type;

    /**
     * The script of js or sql.
     */
    @Required
    protected String script;

    /**
     * Optional. The source path of script, such as '~/hello.js"
     */
    protected String scriptPath;

    /**
     * Prior=true will overrides java's controller action.
     */
    protected boolean prior;

    /**
     * Allow anonymous access?
     */
    protected Boolean anonymous;

    /**
     * Swagger's deprecated.
     */
    protected boolean deprecated;

    /**
     * Swagger's consumes.
     */
    protected Set<String> consumes;

    /**
     * Swagger's produces.
     */
    protected Set<String> produces;

    /**
     * Operation parameters.
     */
    protected List<MetaParameter> parameters = new ArrayList<>();

    /**
     * Operation responses.
     */
    protected List<MetaResponse> responses = new ArrayList<>();

    /**
     * The security.
     */
    protected MetaSecurity security;

    /**
     * Swagger tags, reference to {@link MetaTag}.
     */
    protected Set<String> tags;

    protected Map<String, Object> arguments;

    public String getEntityName() {
        return entityName;
    }

    public void setEntityName(String entityName) {
        this.entityName = entityName;
    }

    public HTTP.Method getMethod() {
        return method;
    }

    public void setMethod(HTTP.Method method) {
        this.method = method;
    }

    public String getPath() {
        return path;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getScript() {
        return script;
    }

    public void setScript(String script) {
        this.script = script;
    }

    public String getScriptPath() {
        return scriptPath;
    }

    public void setScriptPath(String scriptPath) {
        this.scriptPath = scriptPath;
    }

    public boolean isPrior() {
        return prior;
    }

    public void setPrior(boolean prior) {
        this.prior = prior;
    }

    public boolean isAnonymous() {
        return null != anonymous && anonymous;
    }

    public Boolean getAnonymous() {
        return anonymous;
    }

    public void setAnonymous(Boolean anonymous) {
        this.anonymous = anonymous;
    }

    public boolean isDeprecated() {
        return deprecated;
    }

    public void setDeprecated(boolean deprecated) {
        this.deprecated = deprecated;
    }

    public Set<String> getConsumes() {
        return consumes;
    }

    public void setConsumes(Set<String> consumes) {
        this.consumes = consumes;
    }

    public void addConsume(String consume) {
        if(null == consumes) {
            consumes = new LinkedHashSet<>();
        }
        consumes.add(consume);
    }

    public Set<String> getProduces() {
        return produces;
    }

    public void setProduces(Set<String> produces) {
        this.produces = produces;
    }

    public void addProduce(String produce) {
        if(null == produces) {
            produces = new LinkedHashSet<>();
        }
        produces.add(produce);
    }

    public List<MetaParameter> getParameters() {
        return parameters;
    }

    public void setParameters(List<MetaParameter> parameters) {
        this.parameters = parameters;
    }

    public MetaParameter getParameter(String name) {
        for(MetaParameter p : parameters) {
            if(p.getName().equals(name)) {
                return p;
            }
        }
        return null;
    }

    public void addParameter(MetaParameter p) {
        parameters.add(p);
    }

    public List<MetaResponse> getResponses() {
        return responses;
    }

    public void setResponses(List<MetaResponse> responses) {
        this.responses = responses;
    }

    public void addResponse(MetaResponse response) {
        responses.add(response);
    }

    public MetaResponse getResponse(int status) {
        for(MetaResponse r : responses) {
            if(r.getStatus() == status) {
                return r;
            }
        }
        return null;
    }

    public Set<String> getTags() {
        return tags;
    }

    public void addTag(String tag) {
        if(null == tags) {
            tags = new LinkedHashSet<>();
        }
        tags.add(tag);
    }

    public MetaSecurity getSecurity() {
        return security;
    }

    public void setSecurity(MetaSecurity security) {
        this.security = security;
    }

    public void setTags(Set<String> tags) {
        this.tags = tags;
    }

    public Map<String, Object> getArguments() {
        return arguments;
    }

    public void setArguments(Map<String, Object> arguments) {
        this.arguments = arguments;
    }

    @Override
    public String toString() {
        return method + " " + path;
    }
}