/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.parser.RelationDefParser;
import leap.core.validation.annotations.Required;
import leap.lang.Valued;
import leap.lang.convert.StringParsable;
import leap.lang.json.JsonIgnore;

import java.util.ArrayList;
import java.util.List;

public class MetaRelation extends MetaObjNamed implements StringParsable {

    public enum Type implements Valued<String> {
        MANY_TO_MANY("many-to-many"),
        MANY_TO_ONE("many-to-one"),
        ONE_TO_MANY("one-to-many");

        private final String value;

        Type(String value) {
            this.value = value;
        }

        @Override
        public String getValue() {
            return value;
        }

        public boolean isMany() {
            return this.equals(MANY_TO_MANY) || this.equals(ONE_TO_MANY);
        }
    }

    public static final class JoinField {
        public String local;
        public String target;

        public JoinField(String local, String target) {
            this.local = local;
            this.target = target;
        }
    }


    protected Type    type;
    protected String  targetEntity;
    protected String  joinEntity;
    protected Boolean optional;
    protected Boolean expandable;
    protected Boolean logical;
    protected Boolean internal;
    protected String  foreignKeyName;      //for many-to-one only
    protected String  inverseRelationName; //for one-to-many only
    protected List<JoinField> joinFields = new ArrayList<>();

    @JsonIgnore
    protected String[] inverseFields; //for one-to-many

    @Required
    public Type getType() {
        return type;
    }

    public void setType(Type type) {
        this.type = type;
    }

    @Required
    public String getTargetEntity() {
        return targetEntity;
    }

    public void setTargetEntity(String targetEntity) {
        this.targetEntity = targetEntity;
    }

    public String getJoinEntity() {
        return joinEntity;
    }

    public void setJoinEntity(String joinEntity) {
        this.joinEntity = joinEntity;
    }

    public boolean isOptional() {
        return null != optional && optional;
    }

    public Boolean getOptional() {
        return optional;
    }

    public void setOptional(Boolean optional) {
        this.optional = optional;
    }

    public boolean isLogical() {
        return null != logical && logical;
    }

    public Boolean getLogical() {
        return logical;
    }

    public void setLogical(Boolean logical) {
        this.logical = logical;
    }

    public boolean isExpandable() {
        return null != expandable && expandable;
    }

    public Boolean getExpandable() {
        return expandable;
    }

    public void setExpandable(Boolean expandable) {
        this.expandable = expandable;
    }

    public boolean isInternal() {
        return null != internal && internal;
    }

    public Boolean getInternal() {
        return internal;
    }

    public void setInternal(Boolean internal) {
        this.internal = internal;
    }

    public String getForeignKeyName() {
        return foreignKeyName;
    }

    public void setForeignKeyName(String foreignKeyName) {
        this.foreignKeyName = foreignKeyName;
    }

    public String getInverseRelationName() {
        return inverseRelationName;
    }

    public void setInverseRelationName(String inverseRelationName) {
        this.inverseRelationName = inverseRelationName;
    }

    public List<JoinField> getJoinFields() {
        return joinFields;
    }

    public void addJoinField(JoinField field) {
        if(null == joinFields) {
            joinFields = new ArrayList<>();
        }
        joinFields.add(field);
    }

    public void setJoinFields(List<JoinField> joinFields) {
        this.joinFields = joinFields;
    }

    public String[] getInverseFields() {
        return inverseFields;
    }

    public void setInverseFields(String[] inverseFields) {
        this.inverseFields = inverseFields;
    }

    public boolean isManyToOne() {
        return type == Type.MANY_TO_ONE;
    }

    public boolean isOneToMany() {
        return type == Type.ONE_TO_MANY;
    }

    public boolean isManyToMany() {
        return type == Type.MANY_TO_MANY;
    }

    @Override
    public void parseString(String s) {
        RelationDefParser.parse(s, this);
    }

    @Override
    public String toString() {
        return name + "(" + type + ")";
    }
}