/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.parser.SecurityParser;
import leap.lang.Arrays2;
import leap.lang.Strings;
import leap.lang.convert.StringParsable;
import leap.lang.json.JsonObject;
import leap.lang.json.JsonParsable;
import leap.lang.json.JsonValue;

import java.util.List;

public class MetaSecurity extends MetaObj implements StringParsable,JsonParsable {

    protected Rule user;
    protected Rule client;

    public Boolean getUserEnabled() {
        return null == user ? null : user.isEnabled();
    }

    public boolean getClientEnabled() {
        return null == client ? null : client.isEnabled();
    }

    public boolean isUserEnabled() {
        return null != user && user.isEnabled();
    }

    public boolean isClientEnabled() {
        return null != client && client.isEnabled();
    }

    public Rule getUser() {
        return user;
    }

    public void setUser(Rule user) {
        this.user = user;
    }

    public Rule getClient() {
        return client;
    }

    public void setClient(Rule client) {
        this.client = client;
    }

    @Override
    public void parseString(String s) {
        SecurityParser.parse(this, s);
    }

    @Override
    public void parseJson(JsonValue json) {
        if(json.isObject()) {
            JsonObject o = json.asJsonObject();

            this.user   = toRule(o.get("user"));
            this.client = toRule(o.get("client"));
            return;
        }

        Object v = json.raw();
        if(v instanceof String) {
            this.parseString((String)v);
            return;
        }

        throw new IllegalStateException("Can't parse json value '" + v + "'");
    }

    private static Rule toRule(Object v) {
        if(null == v) {
            return null;
        }

        if(v instanceof String) {
            Rule rule = new Rule();

            String s = (String)v;
            if(s.equalsIgnoreCase("enabled")) {
                rule.setEnabled(true);
                return rule;
            }

            if(s.equalsIgnoreCase("disabled")) {
                rule.setEnabled(false);
                return rule;
            }
        }

        return new Rule(toArray(v));
    }

    private static String[] toArray(Object v) {
        if(v == null) {
            return null;
        }
        if(v instanceof String) {
            return Strings.split((String)v, ',', ' ');
        }
        if(v instanceof List) {
            return ((List<String>)v).toArray(Arrays2.EMPTY_STRING_ARRAY);
        }
        throw new IllegalStateException("Invalid array value " + v);
    }

    public static class Rule  {
        protected boolean  enabled;
        protected String[] permissions = Arrays2.EMPTY_STRING_ARRAY;

        public Rule() {

        }

        public Rule(boolean enabled) {
            this.enabled = enabled;
        }

        public Rule(String[] permissions) {
            this.enabled     = true;
            this.permissions = permissions;
        }

        public boolean isEnabled() {
            return enabled;
        }

        public void setEnabled(boolean enabled) {
            this.enabled = enabled;
        }

        public String[] getPermissions() {
            return permissions;
        }

        public void setPermissions(String[] permissions) {
            this.permissions = permissions;
        }
    }
}