/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import leap.lang.collection.WrappedCaseInsensitiveMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;

public class MetaTestSuite extends MetaObjNamed {

    protected List<MetaTestAction>       before = new ArrayList<>();
    protected Map<String, MetaTestCase>  tests  = WrappedCaseInsensitiveMap.create();
    protected Map<String, MetaTestSuite> suites = WrappedCaseInsensitiveMap.create();
    protected List<MetaTestAction>       after  = new ArrayList<>();

    public MetaTestSuite() {
    }

    public MetaTestSuite(String name) {
        this.name  = name;
        this.title = MetaUtils.nameToTitle(name);
    }

    public List<MetaTestAction> getBefore() {
        return before;
    }

    public void addBefore(MetaTestAction action) {
        before.add(action);
    }

    public void setBefore(List<MetaTestAction> before) {
        this.before = before;
    }

    public Map<String, MetaTestCase> getTests() {
        return tests;
    }

    public MetaTestCase getTest(String name) {
        return tests.get(name);
    }

    public void addTest(MetaTestCase test) {
        tests.put(test.getName(), test);
    }

    public void setTests(Map<String, MetaTestCase> tests) {
        this.tests = WrappedCaseInsensitiveMap.create(tests);
    }

    public Map<String, MetaTestSuite> getSuites() {
        return suites;
    }

    public MetaTestSuite getSuite(String name) {
        return suites.get(name);
    }

    public void addSuite(MetaTestSuite suite) {
        suites.put(suite.getName(), suite);
    }

    public void setSuites(Map<String, MetaTestSuite> suites) {
        this.suites = WrappedCaseInsensitiveMap.create(suites);
    }

    public List<MetaTestAction> getAfter() {
        return after;
    }

    public void addAfter(MetaTestAction action) {
        after.add(action);
    }

    public void setAfter(List<MetaTestAction> after) {
        this.after = after;
    }

    public boolean hasTests() {
        return countOfTests() > 0;
    }

    public int countOfTests() {
        AtomicInteger count = new AtomicInteger();
        calcCountOfTests(this, count);
        return count.get();
    }

    public void tryCopy(MetaTestSuite from) {
        tryCopy(from, this);
    }

    public void tryCopyChild(MetaTestSuite fromChild) {
        if(fromChild.countOfTests() == 0) {
            return;
        }

        MetaTestSuite toChild = getSuite(fromChild.getName());
        if(null == toChild) {
            toChild = new MetaTestSuite();
            MetaUtils.copyNamed(fromChild, toChild);
            addSuite(toChild);
        }
        tryCopy(fromChild, toChild);
    }

    public MetaTestSuite filter(Predicate<MetaTestCase> filter) {
        if(null == filter) {
            return this;
        }
        MetaTestSuite suite = new MetaTestSuite();
        MetaUtils.copyNamed(this, suite);
        suite.before.addAll(this.before);

        tests.forEach((n,t) -> {
            if(filter.test(t)) {
                suite.addTest(t);
            }
        });

        suites.forEach((n,s) -> {
            MetaTestSuite child = s.filter(filter);
            if(child.countOfTests() > 0) {
                suite.addSuite(child);
            }
        });

        suite.after.addAll(this.after);
        return suite;
    }

    private void calcCountOfTests(MetaTestSuite suite, AtomicInteger count) {
        count.set(count.get() + suite.tests.size());

        for (MetaTestSuite child : suite.suites.values()) {
            calcCountOfTests(child, count);
        }
    }

    private void tryCopy(MetaTestSuite from, MetaTestSuite to) {
        for(MetaTestCase test : from.getTests().values()) {
            if(null == to.getTest(test.getName())) {
                to.addTest(test);
            }
        }

        for(MetaTestSuite fromChild : from.getSuites().values()) {
            if(fromChild.countOfTests() == 0) {
                continue;
            }
            MetaTestSuite toChild = to.getSuite(fromChild.getName());
            if(null == toChild) {
                toChild = new MetaTestSuite();
                MetaUtils.copyNamed(fromChild, toChild);
                to.addSuite(toChild);
            }
            tryCopy(fromChild, toChild);
        }
    }
}
