/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.model;

import jmms.core.Types;
import leap.lang.Strings;
import leap.lang.beans.BeanProperty;
import leap.lang.beans.BeanType;
import leap.lang.json.JsonIgnore;
import leap.lang.meta.MComplexType;
import leap.lang.meta.MNamedWithDescBuilder;
import leap.web.api.meta.model.*;

import java.util.regex.Pattern;

public class MetaUtils {

    public static String getTitleOrName(MetaObjNamed named) {
        return Strings.firstNotEmpty(named.getTitle(), named.getName());
    }

    public static String nameToTitle(String name) {
        if(name.indexOf('-') < 0 && name.indexOf('_')  < 0) {
            name = Strings.lowerUnderscore(name);
        }
        String[] parts = Strings.split(name, '_', '-');
        StringBuilder s = new StringBuilder();
        for(int i=0;i<parts.length;i++) {
            if(i > 0) {
                s.append(' ');
            }
            if(i == 0) {
                s.append(Strings.upperFirst(parts[i]));
            }else {
                s.append(parts[i]);
            }

        }
        return s.toString();
    }

    public static void copyNamed(MetaObjNamed from, MetaObjNamed to) {
        if(!Strings.isEmpty(from.getName())) {
            to.setName(from.getName());
        }
        copyTitled(from, to);
    }

    public static void copyNamed(MetaObjNamed from, MNamedWithDescBuilder to) {
        to.setName(from.getName());
        copyTitled(from, to);
    }

    public static void copyTitled(MetaObjTitled from, MetaObjTitled to) {
        if(!Strings.isEmpty(from.getTitle())) {
            to.setTitle(from.getTitle());
        }
        if(!Strings.isEmpty(from.getSummary())) {
            to.setSummary(from.getSummary());
        }
        if(!Strings.isEmpty(from.getDescription())) {
            to.setDescription(from.getDescription());
        }
    }

    public static void copyTitled(MetaObjTitled from, MNamedWithDescBuilder to) {
        to.setTitle(from.getTitle());
        to.setSummary(from.getSummary());
        to.setDescription(from.getDescription());
    }

    public static void copyTitled(MetaObjTitled from, MApiObjectWithDescBuilder to) {
        to.setSummary(from.getSummary());
        to.setDescription(from.getDescription());
    }

    public static void tryCopyNamed(MetaObjNamed from, MetaObjNamed to) {
        if(Strings.isEmpty(to.getName())) {
            to.setName(from.getName());
        }
        tryCopyTitled(from, to);
    }

    public static void tryCopyTitled(MetaObjTitled from, MetaObjTitled to) {
        if(Strings.isEmpty(to.getTitle())) {
            to.setTitle(from.getTitle());
        }
        if(Strings.isEmpty(to.getSummary())) {
            to.setSummary(from.getSummary());
        }
        if(Strings.isEmpty(to.getDescription())) {
            to.setDescription(from.getDescription());
        }
    }

    public static void tryCopyNamed(MNamedWithDescBuilder from, MetaObjNamed to) {
        if(Strings.isEmpty(to.getName())) {
            to.setName(from.getName());
        }

        if(Strings.isEmpty(to.getTitle())) {
            to.setTitle(from.getTitle());
        }

        if(Strings.isEmpty(to.getSummary())) {
            to.setSummary(from.getSummary());
        }

        if(Strings.isEmpty(to.getDescription())) {
            to.setDescription(from.getDescription());
        }
    }

    public static void tryCopyField(MetaField from, MetaField to) {
        tryCopy(from, to);
    }

    public static <T> void tryCopy(T from, T to) {
        BeanType bt = BeanType.of(from.getClass());

        for(BeanProperty bp : bt.getProperties()) {
            if(!bp.isField() || !bp.isReadable() || !bp.isWritable()) {
                continue;
            }
            if(bp.isAnnotationPresent(JsonIgnore.class)) {
                continue;
            }
            Object toVal = bp.getValue(to);
            Object fromVal = bp.getValue(from);
            if(null == toVal && null != fromVal) {
                bp.setValue(to, fromVal);
            }
        }
    }

    public static <T> void applySimpleExtension(T ex, T target) {
        BeanType bt = BeanType.of(ex.getClass());

        for(BeanProperty bp : bt.getProperties()) {
            if(!bp.isField() || !bp.isReadable() || !bp.isWritable()) {
                continue;
            }
            if(bp.isAnnotationPresent(JsonIgnore.class)) {
                continue;
            }
            if(!bp.getTypeInfo().isSimpleType()) {
                continue;
            }
            Object fromVal = bp.getValue(ex);
            if(null != fromVal) {
                bp.setValue(target, fromVal);
            }
        }
    }

    public static void tryCopyParameterBase(MApiParameterBaseBuilder from, MetaParameterBase to) {
        tryCopyNamed(from, to);

        if(null == to.getResolvedType()) {
            to.setResolvedType(from.getType());
        }

        if(Strings.isEmpty(to.getFormat())) {
            to.setFormat(from.getFormat());
        }

        if(from.isFile()) {
            to.setFile(true);
        }

        if(null == to.getRequired()) {
            to.setRequired(from.getRequired());
        }

        if(null == to.getDefaults()) {
            to.setDefaults(from.getDefaultValue());
        }

        if(null == to.getEnums()) {
            to.setEnums(from.getEnumValues());
        }

        if(null != from.getValidation()) {
            MApiValidationBuilder v = from.getValidation();
            if (null == to.getPattern() && !Strings.isEmpty(v.getPattern())) {
                to.setPattern(Pattern.compile(v.getPattern()));
            }
            if(null == to.getEnums() && null != v.getEnumValues()) {
                to.setEnums(v.getEnumValues());
            }
            if(null == to.getMinimum()) {
                to.setMinimum(v.getMinimum());
            }
            if(null == to.getExclusiveMinimum()) {
                to.setExclusiveMinimum(v.isExclusiveMinimum());
            }
            if(null == to.getMaximum()) {
                to.setMaximum(v.getMaximum());
            }
            if(null == to.getExclusiveMaximum()) {
                to.setExclusiveMaximum(v.isExclusiveMaximum());
            }
            if(null == to.getMinLength()) {
                to.setMinLength(v.getMinLength());
            }
            if(null == to.getMaxLength()) {
                to.setMaxLength(v.getMaxLength());
            }
            if(null == to.getUniqueItems()) {
                to.setUniqueItems(v.isUniqueItems());
            }
            if(null == to.getMinItems()) {
                to.setMinItems(v.getMinItems());
            }
            if(null == to.getMaxItems()) {
                to.setMaxItems(v.getMaxItems());
            }
            if(null == to.getMultipleOf()) {
                to.setMultipleOf(v.getMultipleOf());
            }
        }
    }

    public static void tryCopyModel(MComplexType from, MetaModel to) {
        tryCopyModel(new MApiModelBuilder(from), to);
    }

    public static void tryCopyModel(MApiModelBuilder from, MetaModel to) {
        tryCopyNamed(from, to);

        if (null != from.getJavaTypes() && !from.getJavaTypes().isEmpty()) {
            to.setMappingClass(from.getJavaTypes().iterator().next());
        }

        for (MApiPropertyBuilder ap : from.getProperties().values()) {
            MetaProperty mp = to.getProperty(ap.getName());
            if (null == mp) {
                to.addProperty(toMetaProperty(ap));
            } else {
                tryCopyProperty(ap, mp);
            }
        }
    }

    public static void tryCopyProperty(MApiPropertyBuilder from, MetaProperty to) {
        MetaUtils.tryCopyParameterBase(from, to);

        if (Strings.isEmpty(to.getType())) {
            to.setType(Types.toTypeName(from.getType()));
        }

        if (null == to.getMappingProperty()) {
            to.setMappingProperty(from.getBeanProperty());
        }

        if (null == to.getReadOnly()) {
            to.setReadOnly(from.getReadOnly());
        }
    }

    public static MetaModel toMetaModel(MComplexType ct) {
        MetaModel m = new MetaModel();

        MApiModelBuilder am = new MApiModelBuilder(ct);
        tryCopyModel(am, m);

        m.setApiModel(am);
        m.getProperties().values().forEach(p -> {
            p.setApiProperty(am.getProperties().get(p.getName()));
        });

        return m;
    }

    public static MetaProperty toMetaProperty(MApiPropertyBuilder p) {
        MetaProperty mp = new MetaProperty();
        tryCopyProperty(p, mp);
        return mp;
    }

    public static void tryCopyValidation(MetaObjValidated mv, MApiValidationBuilder v) {
        if(null == mv || null == v) {
            return;
        }

        if(Strings.isEmpty(v.getPattern())) {
            v.setPattern(null == mv.getPattern() ? null : mv.getPattern().pattern());
        }

        if(null != mv.getEnums() && mv.getEnums().length > 0) {
            v.setEnumValues(mv.getEnums());
        }

        if(null == v.getMinimum()) {
            v.setMinimum(mv.getMinimum());
        }

        if(null == v.getMaximum()) {
            v.setMaximum(mv.getMaximum());
        }

        if(null == v.getMinLength()) {
            v.setMinLength(mv.getMinLength());
        }

        if(null == v.getMaxLength()) {
            v.setMaxLength(mv.getMaxLength());
        }

        if(null == v.getMinItems()) {
            v.setMinItems(mv.getMinItems());
        }

        if(null == v.getMaxItems()) {
            v.setMaximum(mv.getMaxItems());
        }

        if(null == v.getMultipleOf()) {
            v.setMultipleOf(mv.getMultipleOf());
        }

        if(null != mv.getExclusiveMinimum()) {
            v.setExclusiveMinimum(mv.getExclusiveMinimum());
        }

        if(null != mv.getExclusiveMaximum()) {
            v.setExclusiveMaximum(mv.getExclusiveMaximum());
        }

        if(null != mv.getUniqueItems()) {
            v.setUniqueItems(mv.getUniqueItems());
        } 
    }
}
