/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.modules;

import jmms.core.ScriptModule;
import jmms.core.Api;
import jmms.core.model.MetaEntity;
import jmms.core.model.MetaSql;
import leap.core.BeanFactory;
import leap.core.value.Record;
import leap.lang.Strings;
import leap.orm.OrmContext;
import leap.orm.dao.Dao;
import leap.orm.mapping.EntityMapping;
import leap.orm.query.Query;
import leap.orm.query.QueryResult;
import leap.web.api.meta.model.MApiModel;

import javax.script.ScriptException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class DaoModule {

    //cache
    private final Map<String, EntityModule> entityModules = new ConcurrentHashMap<>();

    protected final BeanFactory          factory;
    protected final ScriptModule         from;
    protected final Api                  api;
    protected final Dao                  dao;
    protected final Map<String, MetaSql> sqls;

    public DaoModule(BeanFactory factory, ScriptModule from, Api api, Map<String, MetaSql> sqls) {
        this.factory = factory;
        this.from = from;
        this.api  = api;
        this.dao  = api.getOrmContext().getDao();
        this.sqls = sqls;
    }

    /**
     * Returns the {@link EntityModule} of the given entity..
     * @return
     */
    public EntityModule entity(String name) throws ScriptException {
        String key = Strings.lowerCase(name);
        MetaEntity meta = api.getMeta().getEntity(key);
        if(null == meta) {
            throw from.createScriptException("The entity '" + name + "' not exists");
        }

        EntityModule module = entityModules.get(key);
        if(null == module) {
            MApiModel am = api.getDyna().getMetadata().getModel(name);

            OrmContext    oc = api.getOrmContext();
            EntityMapping em = oc.getMetadata().getEntityMapping(name);

            module = new EntityModule(this, meta, am, em);
            entityModules.put(key, module);
        }
        return module;
    }

    public int exec(String sql, Object params) {
        return executeUpdate(sql, params);
    }

    public int executeUpdate(String sql, Object params) {
        if(sql.startsWith("@")) {
            String key = sql.substring(1);
            MetaSql sqlDef = sqls.get(key);
            if(null == sqlDef) {
                sqlDef = api.getMeta().getSql(key);
            }
            if(null != sqlDef) {
                //todo: cache the sql command.
                return dao.executeUpdate(sqlDef.getScript(), params);
            }else {
                return dao.executeNamedUpdate(key, params);
            }
        }else {
            return dao.executeUpdate(sql, params);
        }
    }

    public Query<Record> createQuery(String sql, Map<String, Object> params) {
        Query<Record> query;
        if(sql.startsWith("@")) {
            String key = sql.substring(1);
            MetaSql sqlDef = sqls.get(key);
            if(null == sqlDef) {
                sqlDef = api.getMeta().getSql(key);
            }
            if(null != sqlDef) {
                //todo: cache the sql command.
                query = dao.createSqlQuery(Record.class, sqlDef.getScript());
            }else {
                query = dao.createNamedQuery(Record.class, key);
            }
        }else {
            query = dao.createSqlQuery(Record.class, sql);
        }

        if(null != params) {
            query.params(params);
        }

        return query;
    }

    public QueryResult<Record> query(String sql, Map<String, Object> params) {
        return createQuery(sql, params).result();
    }

    public Record queryFirst(String sql, Map<String, Object> params) {
        return query(sql, params).first();
    }

    public Record queryFirstOrNull(String sql, Map<String, Object> params) {
        return query(sql, params).firstOrNull();
    }

    public Record querySingle(String sql, Map<String, Object> params) {
        return query(sql, params).single();
    }

    public Record querySingleOrNull(String sql, Map<String, Object> params) {
        return query(sql, params).singleOrNull();
    }

    public List<Record> queryList(String sql, Map<String, Object> params) {
        return createQuery(sql, params).list();
    }

    public String queryString(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalarOrNull().getString();
    }

    public Integer queryInteger(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalar().getInteger();
    }

    public Long queryLong(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalar().getLong();
    }

    public Boolean queryBoolean(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalarOrNull().get(Boolean.class);
    }

    public Double queryDouble(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalarOrNull().get(Double.class);
    }

    public Object queryScalar(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalarOrNull().get();
    }

    public List<Object> queryScalars(String sql, Map<String, Object> params) {
        return createQuery(sql, params).scalars().list();
    }
}