/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.modules;

import jmms.core.model.MetaEntity;
import leap.core.value.Record;
import leap.lang.Beans;
import leap.orm.command.InsertCommand;
import leap.orm.dao.Dao;
import leap.orm.mapping.EntityMapping;
import leap.web.api.dyna.DynaApi;
import leap.web.api.meta.model.MApiModel;
import leap.web.api.mvc.params.QueryOptions;
import leap.web.api.orm.ModelExecutorFactory;
import leap.web.api.orm.ModelQueryExecutor;
import leap.web.api.orm.QueryListResult;
import leap.web.api.orm.SimpleModelExecutorContext;
import leap.web.exception.NotFoundException;

import java.util.List;

public final class EntityModule {

    private final DaoModule            daoModule;
    private final DynaApi              runtimeApi;
    private final Dao                  dao;
    private final MetaEntity           meta;
    private final MApiModel            am;
    private final EntityMapping        em;
    private final ModelExecutorFactory mef;

    public EntityModule(DaoModule daoModule, MetaEntity meta, MApiModel am, EntityMapping em) {
        this.daoModule  = daoModule;
        this.runtimeApi = daoModule.api.getDyna();
        this.dao        = daoModule.dao;
        this.meta       = meta;
        this.am         = am;
        this.em         = em;
        this.mef        = daoModule.factory.getBean(ModelExecutorFactory.class);
    }

    public DaoModule getDao() {
        return daoModule;
    }

    public MetaEntity getMeta() {
        return meta;
    }

    public EntityMapping getMapping() {
        return em;
    }

    public Object create(Object record) {
        InsertCommand insert =
                dao.cmdInsert(em).from(record);

        insert.execute();

        return insert.id();
    }

    public int update(Object record) {
        return dao.update(em, record);
    }

    public int update(Object record, Object id) {
        if(null == id) {
            return dao.update(em, record);
        }else {
            return dao.update(em, id, Beans.toMap(record));
        }
    }

    public int delete(Object id) {
        return dao.delete(em, id);
    }

    public Record find(Object id) throws NotFoundException {
        Record r = dao.findOrNull(em, id);
        if(null == r) {
            throw new NotFoundException("Record '" + id + "' not exists in '" + meta.getName() + "'");
        }
        return r;
    }

    public Record findOrNull(Object id) {
        return dao.findOrNull(em, id);
    }

    public QueryListResult query() {
        return query(new QueryOptions());
    }

    public QueryListResult query(QueryOptions options) {
        return newQueryExecutor().queryList(options);
    }

    public List<Record> queryList() {
        return queryList(new QueryOptions());
    }

    public List<Record> queryList(QueryOptions options) {
        return newQueryExecutor().queryList(options).list;
    }

    public ModelQueryExecutor newQueryExecutor() {
        SimpleModelExecutorContext c = new SimpleModelExecutorContext(runtimeApi, am, dao, em);
        return mef.newQueryExecutor(c);
    }
}
