/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.MetaField;
import leap.lang.Classes;
import leap.lang.Strings;
import leap.lang.beans.BeanProperty;
import leap.lang.beans.BeanType;
import leap.lang.convert.Converts;
import leap.lang.jdbc.JdbcType;
import leap.lang.jdbc.JdbcTypes;

public class FieldDefParser extends AbstractExprParser {

    public static void parse(String s, MetaField field) {
        new FieldDefParser(s, field).parse();
    }

    private static final BeanType bt = BeanType.of(MetaField.class);

    private final MetaField field;

    private String word;

    public FieldDefParser(String expr, MetaField field) {
        super(expr);
        this.field = field;
    }

    public void parse() {
        if(chars.length() == 0) {
            return;
        }

        Boolean first = null;
        boolean reference = false;
        for(;;) {
            word = nextWord();
            if(null == word) {
                return;
            }

            if(word.equals("->")) {
                word = expectNextWord();
                reference = true;
                field.setReferenceTo(word);
                continue;
            }

            if(word.startsWith("->")) {
                reference = true;
                field.setReferenceTo(word.substring(2));
                continue;
            }

            if(word.equalsIgnoreCase("-")) {
                field.setDescription(substring(pos+1).trim());
                return;
            }

            if(word.startsWith("-")) {
                field.setDescription((word.substring(1) + substring(pos)).trim());
                return;
            }

            if(null == first) {
                first = true;
            }else {
                first = false;
            }

            //prop(value)
            int lparenIndex = word.indexOf('(');
            if(lparenIndex > 0) {
                String name  = word.substring(0, lparenIndex);
                String value = word.substring(lparenIndex + 1, word.length() - 1);

                BeanProperty bp = bt.tryGetProperty(name, true);
                if(null != bp) {
                    bp.setValue(field, value);
                    continue;
                }

                //is type name ? varchar(100)
                if(parseType(name, value, field)) {
                    continue;
                }
            }

            if(parseSpecial(word)) {
                continue;
            }

            //prop -> boolean property
            BeanProperty bp = bt.tryGetProperty(word, true);
            if(null != bp) {
                if(Classes.isBoolean(bp.getType())) {
                    int mark = pos;

                    String next = nextWord();
                    if("true".equalsIgnoreCase(next)) {
                        bp.setValue(field, true);
                        continue;
                    }

                    if("false".equalsIgnoreCase(next)) {
                        bp.setValue(field, false);
                        continue;
                    }

                    bp.setValue(field, true);
                    pos = mark;
                }else {
                    String next = expectNextWord();
                    bp.setValue(field, Converts.convert(next, bp.getType(), bp.getGenericType()));
                }
                continue;
            }

            //type name only
            JdbcType type = JdbcTypes.tryForTypeName(word.toLowerCase());
            if(null != type) {
                field.setType(type.getName());
                continue;
            }

            //reference name
            if(first && reference) {
                field.setRelationName(word);
                continue;
            }

            unexpected(word);
        }
    }

    public static void parseType(MetaField field) {
        parseType(field.getType(), field);
    }

    public static void parseType(String type, MetaField field) {
        if(Strings.isEmpty(type)) {
           return;
        }

        int lparenIndex = type.indexOf('(');
        if(lparenIndex > 0) {
            String name  = type.substring(0, lparenIndex);
            String value = type.substring(lparenIndex + 1, type.length() - 1);

            field.setType(name);
            parseType(name, value, field);
        }
    }

    private static boolean parseType(String name, String value, MetaField field) {
        //is type name ? varchar(100)
        JdbcType type = JdbcTypes.tryForTypeName(name.toLowerCase());
        if(null != type) {
            field.setType(type.getName());

            String[] numbers = Strings.split(value, ',');
            if(numbers.length == 2) {
                field.setPrecision(Integer.parseInt(numbers[0]));
                field.setScale(Integer.parseInt(numbers[1]));
                return true;
            }

            if(numbers.length == 1) {
                field.setLength(Integer.parseInt(numbers[0]));
                return true;
            }
        }
        return false;
    }

    private boolean parseSpecial(String word) {
        //column
        if("column".equalsIgnoreCase(word)) {
            field.setColumn(expectNextWord(word));
            return true;
        }

        //auto increment
        if("auto_increment".equalsIgnoreCase(word) || "increment".equalsIgnoreCase(word)) {
            field.setIncrement(true);
            if(Strings.isEmpty(field.getType())) {
                field.setType("integer");
            }
            return true;
        }

        //uuid
        if("uuid".equalsIgnoreCase(word)) {
            field.setValue("uuid");
            if(Strings.isEmpty(field.getType())) {
                field.setType("varchar");
                field.setLength(36);
            }
            return true;
        }

        if("short_uuid".equalsIgnoreCase(word)) {
            field.setValue("short_uuid");
            if(Strings.isEmpty(field.getType())) {
                field.setType("varchar");
                field.setLength(36);
            }
            return true;
        }

        if("short_id".equalsIgnoreCase(word)) {
            field.setValue("short_id");
            if(Strings.isEmpty(field.getType())) {
                field.setType("varchar");
                field.setLength(36);
            }
            return true;
        }

        //map
        if("map".equalsIgnoreCase(word) || Strings.startsWithIgnoreCase(word,"map<")) {
            field.setDataType(word);
            if(Strings.isEmpty(field.getType())) {
                field.setType("varchar");
                field.setLength(4000);
            }
            return true;
        }

        //array
        if("array".equalsIgnoreCase(word) || Strings.startsWithIgnoreCase(word,"array<")) {
            field.setDataType(word);
            if(Strings.isEmpty(field.getType())) {
                field.setType("varchar");
                field.setLength(2000);
            }
            return true;
        }

        //null -> required(false) , nullable -> required(false)
        if("null".equalsIgnoreCase(word) || "nullable".equalsIgnoreCase(word)) {
            field.setRequired(false);
            return true;
        }

        //optional
        if("optional".equalsIgnoreCase(word)) {
            field.setRequired(false);
            return true;
        }

        //logical
        if("logical".equalsIgnoreCase(word)) {
            field.setPersist(false);
            field.setLogical(true);
            return true;
        }

        //not null
        if("not".equalsIgnoreCase(word)) {
            String next = expectNextWord(word);

            if("null".equalsIgnoreCase(next)) {
                field.setRequired(true);
                return true;
            }

            BeanProperty bp = bt.tryGetProperty(next, true);
            if(null != bp && Classes.isBoolean(bp.getType())) {
                bp.setValue(field, false);
                return true;
            }

            unexpected(word);
        }

        //default value
        if("default".equalsIgnoreCase(word)) {
            String v = expectNextWord(word);
            field.setDefaults(quotedWord ? "'" + v + "'" : v);
            return true;
        }

        if("options".equalsIgnoreCase(word)) {
            field.setOptions(expectNextWord(word));
            return true;
        }

        //value
        if("value".equalsIgnoreCase(word)) {
            String v = expectNextWord(word);
            field.setValue(quotedWord ? "'" + v + "'" : v);
            return true;
        }

        //primary key
        if("primary".equalsIgnoreCase(word)) {
            expectNextWord(word, "key");
            field.setIdentity(true);
            return true;
        }

        //format
        if("format".equalsIgnoreCase(word)) {
            field.setFormat(expectNextWord(word));
            return true;
        }

        //input
        if("input".equalsIgnoreCase(word)) {
            field.setInput(expectNextWord(word));
            return true;
        }

        //meaning
        if("meaning".equalsIgnoreCase(word)) {
            field.setMeaning(expectNextWord(word));
            return true;
        }

        //description
        if("desc".equalsIgnoreCase(word) || "description".equalsIgnoreCase(word)) {
            field.setDescription(expectNextWord(word));
            return true;
        }

        //validate length
        if(Strings.startsWithIgnoreCase(word, "length[")) {
            addValidateExpr();
            return true;
        }

        //validate range
        if(Strings.startsWithIgnoreCase(word, "range[")) {
            addValidateExpr();
            return true;
        }

        //validate pattern
        if(Strings.startsWithIgnoreCase(word, "pattern[")) {
            addValidateExpr();
            return true;
        }

        //validate enum
        if(Strings.startsWithIgnoreCase(word, "enum[")) {
            addValidateExpr();
            return true;
        }

        return false;
    }

    private void addValidateExpr() {
        String expr = word;
        if(field.getValidate() == null) {
            field.setValidate(expr);
        }else {
            field.setValidate(field.getValidate() + " " + expr);
        }
    }
}
