/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.*;
import leap.lang.Exceptions;
import leap.lang.Strings;
import leap.lang.http.HTTP;
import leap.web.api.meta.model.MApiParameter;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

public class OperationParser extends AbstractLineParser {

    public static final String PROP_IMPLEMENTED = "implemented";

    public static OperationParser parse(String s) {
        return new OperationParser(s).parseComment();
    }

    private final MetaApi api;

    private String        s;
    private String        doc;
    private String        rest;
    private boolean       ignore;
    private MetaOperation op;
    private Map<String, Object> properties        = new LinkedHashMap<>();
    private Set<String>         mappingClassNames = new HashSet<>(); //for params

    private String line;

    public OperationParser(String s) {
        this(s, null, null);
    }

    public OperationParser(String s, MetaApi api, MetaOperation op) {
        this.s = s;
        this.api = api;
        this.op  = op;
    }

    public boolean isIgnore() {
        return ignore;
    }

    public String doc() {
        return doc;
    }

    public String rest() {
        return rest;
    }

    public Map<String, Object> getProperties() {
        return properties;
    }

    public <T> T getProperty(String name) {
        return (T)properties.get(name);
    }

    public Set<String> getMappingClassNames() {
        return mappingClassNames;
    }

    public MetaOperation getOperation() {
        return op;
    }

    public OperationParser parseComment() {
        int max = s.indexOf("function");
        if(max < 0) {
            max = s.length();
        }

        int end = s.lastIndexOf("*/", max);
        if(end < 0) {
            rest = s;
            return this;
        }

        int start = s.lastIndexOf("/*", end);
        if(start < 0) {
            throw new IllegalStateException("The comment must starts with '/*'");
        }

        doc  = s.substring(start + 2, end);
        rest = s.substring(end + 2);

        doParse();

        return this;
    }

    protected void doParse() {
        if(null == op) {
            op = new MetaOperation();
        }

        boolean descParsed = false;
        StringBuilder desc = new StringBuilder();
        try {
            try(StringReader sr = new StringReader(doc)) {
                BufferedReader reader = new BufferedReader(sr);

                for(;;) {
                    line = reader.readLine();
                    if(null == line) {
                        break;
                    }

                    String content = content(line);
                    if(content.isEmpty()) {
                        continue;
                    }

                    if(content.startsWith("@")) {
                        parseDirective(reader, content.substring(1));
                        descParsed = true;
                        continue;
                    }

                    if(!descParsed) {
                        if(desc.length() > 0) {
                            desc.append('\n');
                        }
                        desc.append(content);
                    }else{
                        throw new IllegalStateException("Unexpected line : " + line);
                    }
                }
            }

            if(desc.length() > 0) {
                op.setDescription(desc.toString());
            }
        }catch (IOException e) {
            throw Exceptions.uncheck(e);
        }
    }

    protected void parseDirective(BufferedReader reader, String line) {
        int i = indexOfWhiteSpaceOrEnd(line);

        String name = line.substring(0, i);
        if(name.isEmpty()) {
            throw new IllegalStateException("Unexpected line : " + line);
        }

        String expr = line.substring(i).trim();

        //@{method}, @GET
        try {
            HTTP.Method method = HTTP.Method.valueOf(name);
            if (null != method) {
                parseMethodAndPath(method, expr);
                return;
            }
        }catch (IllegalArgumentException e) {
            //do nothing.
        }

        if(name.equalsIgnoreCase("ignore") || name.equalsIgnoreCase("ignored")) {
            this.ignore = true;
            return;
        }

        //@anonymous || @allowAnonymous
        if(name.equalsIgnoreCase("anonymous") || name.equalsIgnoreCase("allowAnonymous")) {
            op.setAnonymous(Strings.isEmpty(expr) ? true : Boolean.valueOf(expr));
            return;
        }

        //@title
        if(name.equalsIgnoreCase("title")) {
            op.setTitle(expr);
            return;
        }

        //@summary
        if(name.equalsIgnoreCase("summary")) {
            op.setSummary(expr);
            return;
        }

        //@security
        if(name.equalsIgnoreCase("security")) {
            MetaSecurity security = op.getSecurity();
            if(null == security) {
                security = new MetaSecurity();
                op.setSecurity(security);
            }
            SecurityParser.parse(security, expr);
            return;
        }

        //@deprecated
        if(name.equalsIgnoreCase("deprecated")) {
           op.setDeprecated(true);
            return;
        }
        
        //@tag
        if(name.equalsIgnoreCase("tags")){
            parseTags(expr);
            return;
        }
        
        //@param {typeName} name desc
        if(name.equalsIgnoreCase("param")) {
            parseParam(expr);
            return;
        }

        //@mapping-class name
        if(name.equalsIgnoreCase("mapping-class")) {
            parseMappingClass(expr);
            return;
        }

        //@param-include name
        if(name.equalsIgnoreCase("param-include")) {
            parseParamInclude(expr);
            return;
        }

        //@return [status] {typeName} desc
        if(name.equalsIgnoreCase("return") || name.equalsIgnoreCase("returns")) {
            parseResponse(expr);
            return;
        }

        //@return-include name
        if(name.equalsIgnoreCase("return-include") || name.equalsIgnoreCase("returns-include")) {
            parseReturnInclude(expr);
            return;
        }

        //@produces
        if(name.equalsIgnoreCase("produces")) {
            for(String s : parseProducesOrConsumes(expr)) {
                op.addProduce(s);
            }
            return;
        }

        //@consumes
        if(name.equalsIgnoreCase("consumes")) {
            for(String s : parseProducesOrConsumes(expr)) {
                op.addConsume(s);
            }
            return;
        }

        //@implemented
        if(name.equals(PROP_IMPLEMENTED)) {
            properties.put(PROP_IMPLEMENTED, true);
            return;
        }

        throw new IllegalStateException("Unsupported directive at line : " + line);
    }

    /*
     * @GET
     */
    protected void parseMethodAndPath(HTTP.Method method, String expr) {
        if(null != op.getMethod()) {
            throw new IllegalStateException("Duplicated http method at line : " + line);
        }
        op.setMethod(method);

        //todo: path
    }
    
    //@tags a,b,c
    protected void parseTags(String expr) {
        if(Strings.isNotEmpty(expr)){
            String[] tags = Strings.split(expr,",");
            for(String tag : tags){
                if(Strings.isNotEmpty(tag)){
                    op.addTag(tag.trim());
                }
            }
        }
    }
    
    //@param name
    //@param name desc
    //@param name - desc
    //@param {typeName} name
    //@param {typeName} name desc
    //@param {typeName} name - desc
    //@param <location> {typeName} name desc
    protected void parseParam(String expr) {
        MetaParameter p = new MetaParameter();
        p.setRequired(true);

        if(expr.charAt(0) == '<') {
            int end = expr.indexOf('>', 1);
            if(end < 0) {
                throw new IllegalStateException("The closed '>' not found for param location, line : " + line);
            }
            String name = expr.substring(1, end).trim();

            try {
                MApiParameter.Location location = MApiParameter.Location.valueOf(name.toUpperCase());
                p.setLocation(location);
            }catch (IllegalArgumentException e) {
                throw new IllegalStateException("Invalid param location '" + name + "', line : " + line);
            }

            expr = expr.substring(end + 1).trim();
        }

        if(expr.length() > 0 && expr.charAt(0) == '{') {
            int end = expr.indexOf('}', 1);
            if(end < 0) {
                throw new IllegalStateException("The closed '}' not found for type name, line : " + line);
            }
            String typeName = expr.substring(1, end).trim();
            if(typeName.startsWith("?")) {
                p.setRequired(false);
                typeName = typeName.substring(1);
            }

            if(Strings.isEmpty(typeName)) {
                throw new IllegalStateException("The typeName can't be empty, line : " + line);
            }

            p.setType(typeName);
            expr = expr.substring(end + 1).trim();
        }

        if(expr.length() == 0) {
            if(null == p.getType()) {
                throw new IllegalStateException("Unexpected end of @param, line : " + line);
            }else{
                p.setName(Strings.lowerFirst(p.getType()));
                op.addParameter(p);
                return;
            }
        }

        int index = indexOfWhiteSpaceOrEnd(expr);
        String name = expr.substring(0, index);
        if(name.endsWith(".")) {
            name = name.substring(0, name.length()-1);
        }
        if(name.startsWith("[")) {
            if(!name.endsWith("]")) {
                throw new IllegalStateException("The closed ']' not found for param name, line : " + line);
            }
            name = name.substring(1, name.length() -1);
            p.setRequired(false);
        }

        if(Character.isLetter(name.charAt(0))) {
            p.setName(name);
            if(index < expr.length() - 1) {
                expr = expr.substring(index).trim();
                if(expr.charAt(0)== '-') {
                    expr = expr.substring(1).trim();
                }
                if(expr.length() > 0) {
                    p.setDescription(expr);
                }
            }
        }else{
            p.setName(p.getType());
            p.setDescription(expr);
        }

        op.addParameter(p);
    }

    protected void parseMappingClass(String expr) {
        String name = expr.trim();
        if(Strings.isEmpty(name)) {
            throw new IllegalStateException("The class name must not be empty for @mapping-class");
        }
        this.mappingClassNames.add(name);
    }

    protected void parseParamInclude(String expr) {
        String name = expr.trim();

        if(name.startsWith("{")) {
            name = name.substring(1, name.length() - 1).trim();
        }

        if(null == api) {
            throw new IllegalStateException("Can't parse @param-include, the api is null");
        }

        if(!api.getParameterSets().containsKey(name)) {
            throw new IllegalStateException("The param include '" + name + "' not exists!");
        }

        for(MetaParameter p : api.getParameterSets().get(name).getItems()) {
            op.addParameter(p);
        }
    }

    protected void parseReturnInclude(String expr) {
        String name = expr.trim();

        if(name.startsWith("{")) {
            name = name.substring(1, name.length() - 1).trim();
        }

        if(null == api) {
            throw new IllegalStateException("Can't parse @return-include, the api is null");
        }

        if(!api.getResponseSets().containsKey(name)) {
            throw new IllegalStateException("The return include '" + name + "' not exists!");
        }

        for(MetaResponse resp : api.getResponseSets().get(name).getItems()) {
            op.addResponse(resp);
        }
    }

    //@return description
    //@return {type} description
    //@return [status] {type} description
    protected void parseResponse(String expr) {
        MetaResponse resp = new MetaResponse();
        resp.setStatus(200);

        if(expr.length() > 0) {
            if(expr.charAt(0) == '<') {
                int end = expr.indexOf('>', 1);
                if(end < 0) {
                    throw new IllegalStateException("The closed '>' not found for status code, line : " + line);
                }
                String codeName = expr.substring(1, end);
                try {
                    resp.setStatus(Integer.parseInt(codeName));
                }catch(NumberFormatException e) {
                    throw new IllegalStateException("Invalid status code '" + codeName + "'");
                }
                expr = expr.substring(end + 1).trim();
            }

            if(expr.length() > 0 && expr.charAt(0) == '{') {
                int end = expr.indexOf('}', 1);
                if(end < 0) {
                    throw new IllegalStateException("The closed '}' not found for type name, line : " + line);
                }
                String typeName = expr.substring(1, end);
                resp.setType(typeName);
                expr = expr.substring(end + 1).trim();
            }

            if(expr.length() > 0) {
                resp.setDescription(expr);
            }
        }

        op.addResponse(resp);
    }

    protected String[] parseProducesOrConsumes(String expr) {
        return Strings.splitWhitespaces(Strings.replace(expr, ",", " "));
    }
}
