/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.MetaRelation;
import leap.lang.Enums;
import leap.lang.Strings;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class RelationDefParser extends AbstractExprParser{

    private static final Map<String, MetaRelation.Type> TYPES = new HashMap<>();
    static {
        TYPES.put("1..*", MetaRelation.Type.ONE_TO_MANY);
        TYPES.put("*..*", MetaRelation.Type.MANY_TO_MANY);
        TYPES.put("*..1", MetaRelation.Type.MANY_TO_ONE);
    }

    public static void parse(String s, MetaRelation field) {
        new RelationDefParser(s, field).parse();
    }

    private final MetaRelation relation;

    public RelationDefParser(String expr, MetaRelation relation) {
        super(expr);
        this.relation = relation;
    }

    public void parse() {
        if(chars.length() == 0) {
            return;
        }

        //parse relation type
        String word = nextWord();
        MetaRelation.Type type = TYPES.get(word);
        if(null == type) {
            try {
                type = Enums.valueOf(MetaRelation.Type.class, word.toLowerCase());
            } catch (Exception e) {}
        }
        if(null == type) {
            unexpected("Invalid relation type '" + word + "'");
        }
        relation.setType(type);

        //parse target entity
        word = expectNextWord();
        parseTargetEntity(word);

        //parse others
        parseOthers();
    }

    private void parseTargetEntity(String word) {
        int index = word.indexOf('(');
        if(index >= 0) {

            if(!word.endsWith(")")) {
                unexpected("The expr '" + word + "' must closed by ')'");
            }

            String entity = word.substring(0, index);
            relation.setTargetEntity(entity);

            String joins = word.substring(index+1, word.length() -1).trim();

            if(relation.isOneToMany()) {
                relation.setInverseFields(Strings.split(joins, ','));
                return;
            }

            if(relation.isManyToOne()) {
                String[] parts = Strings.split(joins, ',');

                List<MetaRelation.JoinField> fields = new ArrayList<>();
                for(int i=0;i<parts.length;i++) {
                    String[] joined = Strings.split(parts[i], "->");

                    if(parts.length == 1 && joined.length == 1) {
                        fields.add(new MetaRelation.JoinField(joined[0], null));
                        break;
                    }

                    if(joined.length != 2) {
                        unexpected("Invalid join fields '" + joins + "', the correct format 'field1 -> id1, field2 -> id2'");
                    }

                    fields.add(new MetaRelation.JoinField(joined[0], joined[1]));
                }
                relation.setJoinFields(fields);
                return;
            }

            if(relation.isManyToMany()) {
                relation.setJoinEntity(joins);
                return;
            }

            unexpected("Unsupported relation type '" + relation.getType() + "'");
        }else {
            if(relation.isManyToOne()) {
                unexpected("Target entity '" + word + "' must defines join fields '" + word + "({join fields})' for many-to-one relation");
            }

            if(relation.isManyToMany()) {
                unexpected("Target entity '" + word + "' must defines join entity '" + word + "({join entity})' for many-to-many relation");
            }

            relation.setTargetEntity(word);
        }
    }

    private void parseOthers() {
        String word;
        for(;;) {
            word = nextWord();
            if(null == word) {
                return;
            }

            if(word.equalsIgnoreCase("optional")) {
                relation.setOptional(true);
                continue;
            }

            if(word.equalsIgnoreCase("logical")) {
                relation.setLogical(true);
                continue;
            }

            if(word.equalsIgnoreCase("required")) {
                relation.setOptional(false);
                continue;
            }

            if(word.equalsIgnoreCase("expandable")) {
                relation.setExpandable(true);
                continue;
            }

            unexpected(word);
        }
    }

}
