/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.MetaSecurity;
import leap.lang.Arrays2;
import leap.lang.Strings;

public class SecurityParser {

    public static void parse(MetaSecurity security, String expr) {
        expr = Strings.trim(expr);
        if(Strings.isEmpty(expr)) {
            return;
        }

        String[] rules = Strings.split(expr, '|');

        if(rules.length > 2) {
            throw new IllegalStateException("Invalid security expr '" + expr + "'");
        }

        for(String rule : rules) {
            parseRule(security, rule);
        }
    }

    private static void parseRule(MetaSecurity security, String expr) {
        expr = Strings.trim(expr);
        String[] parts = Strings.split(expr, ' ', ',');

        String   name        = parts[0];
        String[] permissions = parts.length > 1 ? removeFirst(parts) : Arrays2.EMPTY_STRING_ARRAY;
        boolean  disabled    = permissions.length == 1 && permissions[0].equalsIgnoreCase("disabled");

        MetaSecurity.Rule rule = disabled ? new MetaSecurity.Rule() : new MetaSecurity.Rule(permissions);

        if(name.equalsIgnoreCase("user")) {
            security.setUser(rule);
            return;
        }

        if(name.equalsIgnoreCase("client")) {
            security.setClient(rule);
            return;
        }

        throw new IllegalStateException("Invalid security '" + name + "', must be 'user' or 'client'");
    }

    private static String[] removeFirst(String[] a0) {
        String[] a1 = new String[a0.length-1];
        for(int i=0;i<a1.length;i++) {
            a1[i] = a0[i+1];
        }
        return a1;
    }

    protected SecurityParser() {

    }
}