/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.MetaSql;
import leap.lang.Exceptions;
import leap.lang.Strings;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

public class SqlsParser extends AbstractLineParser {

    public static final String PREFIX = "//@SQL";

    public static Map<String, MetaSql> parse(String s) {
        if(!Strings.isEmpty(s)) {
            int index = s.indexOf(PREFIX);
            if(index >= 0) {
                s = s.substring(index + PREFIX.length()).trim();
                return parseWithoutPrefix(s);
            }
        }
        return Collections.emptyMap();
    }

    public static Map<String, MetaSql> parseWithoutPrefix(String s) {
        Map<String, MetaSql> sqls = new LinkedHashMap<>();

        new SqlsParser(s, sqls).parse();

        return sqls;
    }

    private String               s;
    private String               line;
    private Map<String, MetaSql> sqls;

    private SqlsParser(String s, Map<String,MetaSql> sqls) {
        this.s = s;
        this.sqls = sqls;
    }

    protected void parse() {
        if(s.startsWith("/*")) {
            s = s.substring(2).trim();
        }

        int end = s.indexOf("*/");
        if(end > 0) {
            s = s.substring(0, end).trim();
        }

        if(s.length() > 0) {
            doParse(sqls, s);
        }
    }

    /*
     * @name desc
     *
     *      script
     *
     * @name desc
     *
     *      script
     */
    protected void doParse(Map<String, MetaSql> sqls, String s) {
        try {
            try(StringReader sr = new StringReader(s)) {
                BufferedReader reader = new BufferedReader(sr);

                for(;;) {
                    line = reader.readLine();
                    if(null == line) {
                        break;
                    }

                    String content = content(line);
                    if(content.isEmpty()) {
                        continue;
                    }

                    if(content.startsWith("#")) {
                        content = content.substring(1).trim();
                        parseSqlDef(reader, content);
                        continue;
                    }

                    if(content.startsWith("--")) {
                        content = content.substring(2).trim();
                        if(content.startsWith("@")) {
                            parseSqlDef(reader, content.substring(1));
                        }
                        continue;
                    }

                    if(content.startsWith("@")) {
                        parseSqlDef(reader, content.substring(1));
                        continue;
                    }
                }
            }
        }catch (IOException e) {
            throw Exceptions.uncheck(e);
        }
    }

    private void parseSqlDef(BufferedReader reader, String start) throws IOException {
        int i = indexOfWhiteSpaceOrEnd(start);

        String name = start.substring(0, i);
        if(name.isEmpty()) {
            throw new IllegalStateException("Unexpected line : " + line);
        }

        if(sqls.containsKey(name)) {
            throw new IllegalStateException("Duplicated sql name '" + name + "'");
        }

        //String desc = line.substring(i).trim();

        //@name [-] desc
        MetaSql sql = new MetaSql();
        sql.setName(name);

        readSqlScript(reader, sql);

        sqls.put(name, sql);
    }

    private void readSqlScript(BufferedReader reader, MetaSql sql) throws IOException {
        StringBuilder buf = new StringBuilder();

        String nextStart = null;

        for(;;) {
            line = reader.readLine();
            if(null == line) {
                break;
            }

            String content = line.trim();

            if(content.startsWith("*")) {
                content = content.substring(1);
            }

            if(Strings.isBlank(content)) {
                continue;
            }

            if(startsWithSkipWhiteSpace(content, "--")) {
                content = content.trim().substring(2).trim();
                if(content.startsWith("@")) {
                    nextStart = content.substring(1).trim();
                    break;
                }else{
                    continue;
                }
            }

            if(startsWithSkipWhiteSpace(content, "@") || startsWithSkipWhiteSpace(content, "#")) {
                nextStart = content.trim().substring(1);
                break;
            }

            buf.append(content).append('\n');
        }

        String script = buf.toString().trim();
        if(script.isEmpty()) {
            throw new IllegalStateException("Empty script of sql '" + sql.getName() + "'");
        }

        sql.setScript(script);

        if(null != nextStart && !nextStart.isEmpty()) {
            parseSqlDef(reader, nextStart);
        }
    }

}
