/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.parser;

import jmms.core.model.MetaFormat;
import leap.lang.Arrays2;
import leap.lang.Strings;
import leap.lang.json.JSON;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

public class ValidateExprParser extends AbstractExprParser {

    public static void parse(String s, MetaFormat format) {
        new ValidateExprParser(s, format).parse();
    }

    private final MetaFormat format;

    private String word;

    public ValidateExprParser(String expr, MetaFormat format) {
        super(expr);
        this.format = format;
    }

    public void parse() {
        if(chars.length() == 0) {
            return;
        }

        for(;;) {
            word = nextWord();
            if(null == word) {
                break;
            }

            int index = word.indexOf('[');
            if(index > 0) {
                String type = word.substring(0, index);
                String expr = word.substring(index+1,word.length()-1);

                try {
                    if (parseValidator(type, expr)) {
                        continue;
                    }
                }catch (Exception e) {
                    error("Invalid '" + word + "', " + e.getMessage());
                }
            }

            unexpected(word);
        }
    }

    private boolean parseValidator(String type, String expr) {
        if(type.equalsIgnoreCase("length")) {
            String[] parts = Strings.split(expr, ",", true, false);
            if(parts.length != 2) {
                error("Invalid expr '" + word + "', must be length[min,max]");
            }

            if(!parts[0].isEmpty()) {
                format.setMinLength(Integer.parseInt(parts[0]));
            }

            if(!parts[1].isEmpty()) {
                format.setMaxLength(Integer.parseInt(parts[1]));
            }

            if(null != format.getMinLength() && null != format.getMaxLength()) {
                if(format.getMinLength() > format.getMaxLength()) {
                    error("The minLength " + format.getMinLength() + " > maxLength " + format.getMaxLength());
                }
            }

            return true;
        }

        if(type.equalsIgnoreCase("range")) {
            String[] parts = Strings.split(expr, ",", true, false);
            if(parts.length != 2) {
                error("Invalid expr '" + word + "', must be range[min,max]");
            }

            if(!parts[0].isEmpty()) {
                format.setMinimum(new BigDecimal(parts[0]));
            }

            if(!parts[1].isEmpty()) {
                format.setMaximum(new BigDecimal(parts[1]));
            }

            if(null != format.getMinimum() && null != format.getMaximum()) {
                if(format.getMinimum().doubleValue() > format.getMaximum().doubleValue()) {
                    error("The min " + format.getMinimum() + " > max " + format.getMaximum());
                }
            }

            return true;
        }

        if(type.equalsIgnoreCase("enum")) {
            List<String> list = new ArrayList<>();

            String[] items = Strings.split(expr, ',');
            for(int i=0;i<items.length;i++) {
                String item = items[i];
                if(item.startsWith("'")) {
                    item = item.substring(1);
                }
                if(item.endsWith("'")) {
                    item = item.substring(0, item.length() - 1);
                }
                list.add(item);
            }
            format.setEnums(list.toArray(new String[0]));
            return true;
        }

        if(type.equalsIgnoreCase("pattern")) {
            try {
                format.setPattern(Pattern.compile(expr));
            } catch (Exception e) {
                error("pattern['" + expr + "'] compile error, " + e.getMessage());
            }
            return true;
        }

        return false;
    }


}
