/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.validate;

import jmms.core.model.MetaObjFormatted;
import jmms.core.model.MetaObjValidated;
import leap.core.validation.Validation;
import leap.lang.Enumerable;
import leap.lang.Enumerables;
import leap.lang.Out;
import leap.lang.annotation.Nullable;

public class ArrayValidator extends Validator {

    protected final SimpleValidator arrayValidator;
    protected final Validator       elementValidator;

    public ArrayValidator(String name, boolean required, @Nullable MetaObjFormatted formatted, @Nullable Validator elementValidator) {
        super(name, required);

        if(formatted instanceof MetaObjValidated) {
            MetaObjValidated validated = (MetaObjValidated)formatted;
            this.arrayValidator = validated.hasArrayValidation() ? new SimpleValidator(name, validated) : null;
        }else {
            this.arrayValidator = null;
        }

        this.elementValidator = elementValidator;
    }

    @Override
    protected boolean doValidate(Validation validation, String name, Object value, Out<Object> out) {
        Enumerable e;
        try {
            e = Enumerables.of(value);
        }catch (IllegalArgumentException ex) {
            validation.addError(name, "The value must be an array or collection");
            return false;
        }

        if(null != arrayValidator && !arrayValidator.doValidateArray(validation, name, e)) {
            return false;
        }

        if(null != elementValidator) {
            int i=0;

            for(Object element : e) {
                if(!elementValidator.validate(validation, name + "[" + i + "]", element, Out.empty())) {
                    return false;
                }
                i++;
            }
        }

        return true;
    }

}