/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.validate;

import jmms.core.model.MetaApi;
import jmms.core.model.MetaModelBase;
import leap.core.validation.Validation;
import leap.lang.Arrays2;
import leap.lang.Beans;
import leap.lang.Out;
import leap.lang.Strings;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public abstract class ModelValidatorBase<T extends MetaModelBase> extends Validator {

    protected final T                    model;
    protected final String[]             requiredProperties;
    protected final String[]             readOnlyProperties;
    protected final PropertyValidation[] propertyValidations;

    protected ModelValidatorBase(String name, boolean required, MetaApi api, T model, boolean partial) {
        super(name, required);
        this.model = model;
        this.requiredProperties  = partial ? Arrays2.EMPTY_STRING_ARRAY : resolveRequiredProperties(model);
        this.readOnlyProperties  = resolveReadOnlyProperties(model);
        this.propertyValidations = resolvePropertyValidations(api, model, name);
    }

    @Override
    protected boolean validateRequired(Validation validation, String name, Object value) {
        return validation.stateNotNull(name, value);
    }

    @Override
    protected boolean doValidate(Validation validation, String name, Object value, Out<Object> out) {
        boolean isMap = value instanceof Map;
        Map map = isMap ? (Map)value : Beans.toMap(value);

        if(requiredProperties.length > 0) {
            for(String fieldName : requiredProperties) {
                if(Strings.isNullOrEmpty(map.get(fieldName))) {
                    validation.addError(qname(name, fieldName), "The property is required!");
                    return false;
                }
            }
        }

        /*else {
            //removes null or empty
            for(Object key : map.keySet().toArray()) {
                if(Strings.isNullOrEmpty(map.get(key))) {
                    map.remove(key);
                }
            }
        }
        */

        if(readOnlyProperties.length > 0) {
            for(String fieldName : readOnlyProperties) {
                Object v = map.get(fieldName);
                if(!Strings.isNullOrEmpty(v)) {
                    validation.addError(qname(name, fieldName), "The property is read only");
                    return false;
                }
            }
        }

        Out<Object> propOut = isMap ? new Out<>() : Out.empty();
        for(PropertyValidation pv : propertyValidations) {
            Object prop = map.get(pv.name);
            if(null == prop) {
                continue;
            }

            if(!pv.validator.validate(validation, prop, propOut)) {
                return false;
            }

            if(propOut.isPresent()) {
                map.put(pv.name, propOut.getAndReset());
            }
        }

        return true;
    }

    protected String[] resolveRequiredProperties(T model) {
        Set<String> set = new HashSet<>();
        try {
            model.getApiModel().getProperties().values().forEach(p -> {
                if (p.isRequired()) {
                    set.add(p.getName());
                }
            });
        }catch (Exception e) {
            System.out.println();
        }

        return set.toArray(new String[0]);
    }

    protected String[] resolveReadOnlyProperties(T model) {
        Set<String> set = new HashSet<>();

        model.getApiModel().getProperties().values().forEach(p -> {
            if(p.isReadOnly()) {
                set.add(p.getName());
            }
        });
        return set.toArray(new String[0]);
    }

    protected abstract PropertyValidation[] resolvePropertyValidations(MetaApi api, T model, String name);

    protected static final class PropertyValidation {
        final String    name;
        final Validator validator;

        PropertyValidation(String name, Validator validator) {
            this.name      = name;
            this.validator = validator;
        }

        @Override
        public String toString() {
            return name;
        }
    }
}
