/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.validate;

import jmms.core.model.MetaObjValidated;
import leap.core.validation.Validation;
import leap.lang.*;
import leap.lang.convert.Converts;
import leap.orm.mapping.FieldMapping;
import leap.orm.validation.FieldValidator;
import leap.orm.value.EntityWrapper;

import java.math.BigDecimal;

public class SimpleValidator extends Validator implements FieldValidator {

    private final MetaObjValidated validated;
    private final SimpleValidator  formatValidator;

    private final BigDecimal maximum;
    private final BigDecimal minimum;

    private final String stringEnums;

    public SimpleValidator(String name, MetaObjValidated validated) {
        this(name, validated.isRequired(), validated);
    }

    public SimpleValidator(String name, boolean required, MetaObjValidated validated) {
        super(name, required);
        this.validated = validated;
        this.formatValidator = null == validated.getResolvedFormat() ? null : new SimpleValidator(name, false, validated.getResolvedFormat());
        this.maximum = null == validated.getMaximum() ? null : new BigDecimal(validated.getMaximum().toString());
        this.minimum = null == validated.getMinimum() ? null : new BigDecimal(validated.getMinimum().toString());
        this.stringEnums = validated.hasEnumValidation() ? Strings.join(validated.getEnums(), ",") : null;
    }

    @Override
    public boolean validate(EntityWrapper entity, FieldMapping fm, Object value, Validation validation, int maxErrors) {
        return validate(validation, value, Out.empty());
    }

    @Override
    protected boolean doValidate(Validation validation, String name, Object value, Out<Object> out) {
        if(!validateString(validation, name, value)) {
            return false;
        }

        if(!validateNumber(validation, name, value)) {
            return false;
        }

        if(!validateArray(validation, name, value)) {
            return false;
        }

        if(!validateEnums(validation, name, value)) {
            return false;
        }

        if(null != formatValidator) {
            return formatValidator.validate(validation, name, value, out);
        }

        return true;
    }

    protected boolean validateString(Validation validation, String name, Object value) {
        if(validated.hasStringValidation()) {
            String s = Converts.toString(value);

            if(null != validated.getMaxLength() && s.length() > validated.getMaxLength()) {
                validation.addError(name, "The string's length must be <= " + validated.getMaxLength());
                return false;
            }

            if(null != validated.getMinLength() && s.length() < validated.getMinLength()) {
                validation.addError(name, "The string's length must be >= " + validated.getMinLength());
                return false;
            }

            if(null != validated.getEnums() && !Arrays2.contains(validated.getEnums(), s)) {
                validation.addError(name, "The value must be one of [" + Strings.join(validated.getEnums(), ',') + "]");
                return false;
            }

            if(null != validated.getPattern()) {
                if(!validated.getPattern().matcher(value.toString()).matches()){
                    validation.addError(name, "The value did not matches the pattern!");
                    return false;
                }
            }
        }

        return true;
    }

    protected boolean validateNumber(Validation validation, String name, Object value) {
        if(validated.hasNumberValidation()) {
            BigDecimal n = Converts.convert(value, BigDecimal.class);

            if(null != maximum && n.compareTo(maximum) > 0) {
                validation.addError(name, "The number must <= " + validated.getMaximum());
                return false;
            }

            if(null != minimum && n.compareTo(minimum) < 0) {
                validation.addError(name, "The number must >= " + validated.getMinimum());
                return false;
            }
        }

        return true;
    }

    protected boolean validateArray(Validation validation, String name, Object value) {
        if(validated.hasArrayValidation()) {
            return doValidateArray(validation, name, Enumerables.of(value));
        }

        return true;
    }

    protected boolean doValidateArray(Validation validation, String name, Enumerable e) {
        int size = e.size();

        if(null != validated.getMinItems() && size < validated.getMinItems()) {
            validation.addError(name, "The array's size must be >= " + validated.getMinItems());
            return false;
        }

        if(null != validated.getMaxItems() && size > validated.getMaxItems()) {
            validation.addError(name, "The array's size must <= " + validated.getMaxItems());
            return false;
        }

        return true;
    }

    protected boolean validateEnums(Validation validation, String name, Object value) {
        if(validated.hasEnumValidation()) {
            String s = Converts.toString(value);

            if(Arrays2.indexOf(validated.getEnums(), s) < 0) {
                validation.addError(name, "The value must be one of [" + stringEnums + "]");
                return false;
            }
        }
        return true;
    }
}