/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.validate;

import leap.core.validation.Validation;
import leap.lang.Out;
import leap.lang.convert.ConvertException;
import leap.lang.convert.Converts;
import leap.lang.meta.MSimpleType;
import leap.lang.meta.MSimpleTypeKind;

import java.sql.Time;
import java.sql.Timestamp;
import java.util.Date;

public class TypeValidator extends Validator {

    private final MSimpleType         type;
    private final TypeCheckAndConvert typeCheckAndConvert;
    private final Validator           combined;

    public TypeValidator(String name, boolean required, MSimpleType type) {
        this(name, required, type, null);
    }

    public TypeValidator(String name, boolean required, MSimpleType type, Validator combined) {
        super(name, required);
        this.type = type;
        this.typeCheckAndConvert = resolveTypeCheckAndConvert();
        this.combined = combined;
    }

    @Override
    protected boolean doValidate(Validation validation, String name, Object value, Out<Object> out) {
        if(null != typeCheckAndConvert) {
            if(!typeCheckAndConvert.checkAndConvert(validation, name, value, out)) {
                return false;
            }
            if(out.isPresent()) {
                value = out.get();
            }
        }
        return null != combined ? combined.validate(validation, name, value, out) : true;
    }

    protected TypeCheckAndConvert resolveTypeCheckAndConvert() {
        MSimpleTypeKind kind = type.getSimpleTypeKind();

        if(kind == MSimpleTypeKind.DATETIME) {
            return SimpleTypeCheckAndConvert.DATETIME;
        }

        if(kind == MSimpleTypeKind.DATE) {
            return SimpleTypeCheckAndConvert.DATE;
        }

        if(kind == MSimpleTypeKind.DATE) {
            return SimpleTypeCheckAndConvert.TIME;
        }

        //todo: other data types.

        return null;
    }

    protected interface TypeCheckAndConvert {
        boolean checkAndConvert(Validation validation, String name, Object value, Out<Object> out);
    }

    protected static final class SimpleTypeCheckAndConvert implements TypeCheckAndConvert {

        static SimpleTypeCheckAndConvert DATETIME = new SimpleTypeCheckAndConvert(Timestamp.class);
        static SimpleTypeCheckAndConvert DATE     = new SimpleTypeCheckAndConvert(Date.class);
        static SimpleTypeCheckAndConvert TIME     = new SimpleTypeCheckAndConvert(Time.class);

        private final Class<?> targetType;

        public SimpleTypeCheckAndConvert(Class<?> targetType) {
            this.targetType = targetType;
        }

        @Override
        public boolean checkAndConvert(Validation validation, String name, Object value, Out<Object> out) {
            try {
                out.set(Converts.convert(value, targetType));
                return true;
            }catch (ConvertException e) {
                validation.addError(name, "Can't convert to '" + targetType.getSimpleName() + "', invalid data type");
                return false;
            }
        }
    }
}