/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package jmms.core.validate;

import jmms.core.model.*;
import leap.lang.annotation.Nullable;
import leap.lang.meta.MCollectionType;
import leap.lang.meta.MComplexTypeRef;
import leap.lang.meta.MDictionaryType;
import leap.lang.meta.MType;

public class Validators {

    private static ThreadLocal<Boolean> partial = new ThreadLocal<>();

    public static void setPartial(boolean check) {
        partial.set(check);
    }

    public static void removePartial() {
        partial.remove();
    }

    public static boolean isPartial() {
        Boolean b = partial.get();
        if(null == b) {
            return false;
        }else {
            return b;
        }
    }

    /**
     * Returns the validator of the parameter or null if no validation.
     */
    public static Validator of(MetaApi api, MetaObjFormatted p) {
        return of(api, p, p.getName());
    }

    /**
     * Returns the validator of the parameter or null if no validation.
     */
    public static Validator of(MetaApi api, MetaObjFormatted p, String name) {
        return of(api, name, p.isRequired(), p.getResolvedType(), p);
    }

    public static Validator of(MetaApi api, String name, MType type) {
        return of(api, name, false, type, null);
    }

    public static Validator of(MetaApi api, String name, boolean required, MType type, @Nullable MetaObjFormatted p) {
        if(type.isSimpleType()) {
            SimpleValidator validator = null;
            if(null != p) {
                if(p instanceof MetaObjValidated && ((MetaObjValidated) p).hasValidation()) {
                    validator = new SimpleValidator(name, (MetaObjValidated)p);
                }else if(null != p.getResolvedFormat() && p.getResolvedFormat().hasValidation()) {
                    validator = new SimpleValidator(name, p.isRequired(), p.getResolvedFormat());
                }
            }

            if(null != validator && p instanceof MetaParameter) {
                return validator;
            }else {
                return new TypeValidator(name, required, type.asSimpleType(), validator);
            }
        }

        if(type.isComplexTypeOrRef()) {
            Boolean partial = null;
            if(type instanceof MComplexTypeRef) {
                partial = ((MComplexTypeRef) type).getPartial();
            }
            if(null == partial) {
                partial = isPartial();
            }

            MetaEntity entity = api.getEntity(type.getName());
            if(null != entity) {
                return new EntityValidator(name, required, api, entity, partial);
            }

            MetaModel model = api.getModel(type.getName());
            if(null != model) {
                return new ModelValidator(name, required, api, model, partial);
            }

            if(type.isComplexType()) {
                model = MetaUtils.toMetaModel(type.asComplexType());
                return new ModelValidator(name, required, api, model, partial);
            }

            throw new IllegalStateException("Invalid complex type '" + type + "' for validation");
        }

        if(type.isCollectionType()) {
            return ofArray(api, name, required, type.asCollectionType(), p);
        }

        if(type.isDictionaryType()) {
            return ofMap(api, name, required, type.asDictionaryType());
        }

        return null;
    }

    protected static Validator ofArray(MetaApi api, String name, boolean required, MCollectionType t, @Nullable MetaObjFormatted p) {
        MType elementType = t.getElementType();

        Validator elementValidator = of(api, name + ".items", elementType);
        return new ArrayValidator(name, required, p, elementValidator);
    }

    protected static Validator ofMap(MetaApi api, String name, boolean required, MDictionaryType t) {
        MType valueType = t.getValueType();

        Validator valueValidator = of(api, name + ".values", valueType);
        return new MapValidator(name, required, valueValidator);
    }
}
