/*
 * Copyright 2019 lorislab.org.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.lorislab.quarkus.jel.log.deployment;

import io.quarkus.arc.deployment.AnnotationsTransformerBuildItem;
import io.quarkus.arc.processor.AnnotationsTransformer;
import io.quarkus.deployment.annotations.BuildProducer;
import io.quarkus.deployment.annotations.BuildStep;
import io.quarkus.deployment.annotations.Record;
import io.quarkus.deployment.builditem.CombinedIndexBuildItem;
import io.quarkus.deployment.builditem.substrate.ReflectiveClassBuildItem;
import org.jboss.jandex.AnnotationInstance;
import org.jboss.jandex.AnnotationTarget;
import org.jboss.jandex.ClassInfo;
import org.jboss.jandex.DotName;
import org.lorislab.quarkus.jel.log.interceptor.LoggerService;
import org.lorislab.quarkus.jel.log.interceptor.LoggerServiceInterceptor;
import org.lorislab.quarkus.jel.log.parameters.LoggerParameter;

import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.context.RequestScoped;
import javax.inject.Singleton;
import java.util.*;
import java.util.stream.Collectors;

import static io.quarkus.deployment.annotations.ExecutionTime.STATIC_INIT;

public class Build {

    BuildConfig buildConfig;

    private static final List<String> ANNOTATIONS = Arrays.asList(
            ApplicationScoped.class.getName(),
            Singleton.class.getName(),
            RequestScoped.class.getName()
    );

    private static final List<DotName> ANNOTATION_DOT_NAMES =
            ANNOTATIONS.stream()
                    .map(DotName::createSimple)
                    .collect(Collectors.toList());

    @BuildStep
    @Record(STATIC_INIT)
    void loggerParameter(CombinedIndexBuildItem indexBuildItem, BuildProducer<ReflectiveClassBuildItem> reflectiveClass) {
        Collection<ClassInfo> cc = indexBuildItem.getIndex().getAllKnownSubclasses(DotName.createSimple(LoggerParameter.class.getName()));
        for (ClassInfo c : cc) {
            try {
                @SuppressWarnings("unchecked")
                Class<LoggerParameter> clazz = (Class<LoggerParameter>) Class.forName(c.name().toString());
                LoggerParameter lp = clazz.getConstructor().newInstance();
                LoggerServiceInterceptor.addLoggerParameter(lp);
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    @BuildStep
    @Record(STATIC_INIT)
    public AnnotationsTransformerBuildItem interceptorBinding() {
        return new AnnotationsTransformerBuildItem(new AnnotationsTransformer() {

            @Override
            public boolean appliesTo(AnnotationTarget.Kind kind) {
                return kind == AnnotationTarget.Kind.CLASS;
            }

            public void transform(TransformationContext context) {
                ClassInfo target = context.getTarget().asClass();
                Map<DotName, List<AnnotationInstance>> tmp = target.annotations();
                Optional<DotName> dot = ANNOTATION_DOT_NAMES.stream().filter(tmp::containsKey).findFirst();
                if (dot.isPresent()) {
                    String name = target.name().toString();
                    Optional<String> add = buildConfig.includes.stream().filter(name::startsWith).findFirst();
                    if (add.isPresent()) {
                        context.transform().add(LoggerService.class).done();
                    }
                }
            }
        });
    }

}
