
package org.lwapp.dropwizard.core;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.Any;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import javax.servlet.DispatcherType;
import javax.servlet.FilterRegistration;
import javax.servlet.http.HttpServlet;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.servlets.CrossOriginFilter;
import org.glassfish.jersey.server.ServerProperties;
import org.jboss.weld.environment.se.StartMain;
import org.jboss.weld.environment.se.events.ContainerInitialized;
import org.lwapp.commons.cli.AbstractCommand;
import org.lwapp.commons.cli.CliBuilder;
import org.lwapp.commons.cli.CliClientHandler;
import org.lwapp.commons.cli.CliServerFactory;
import org.lwapp.commons.cli.TelnetClientHandler;
import org.lwapp.commons.cli.Terminal;
import org.lwapp.commons.utils.ClassUtils;
import org.lwapp.configclient.AutoStartable;
import org.lwapp.dropwizard.core.config.ApplicationServerConfig;
import org.lwapp.dropwizard.core.config.CoreConfiguration;
import org.lwapp.dropwizard.core.metrics.CpuUsageGauge;
import org.lwapp.dropwizard.core.metrics.FreeMemoryGauge;
import org.lwapp.dropwizard.core.rest.ws.filter.LogTraceIdFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.SharedMetricRegistries;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.DeserializationFeature;

import io.dropwizard.Application;
import io.dropwizard.Bundle;
import io.dropwizard.ConfiguredBundle;
import io.dropwizard.cli.Command;
import io.dropwizard.cli.ConfiguredCommand;
import io.dropwizard.jersey.setup.JerseyEnvironment;
import io.dropwizard.jetty.MutableServletContextHandler;
import io.dropwizard.setup.Bootstrap;
import io.dropwizard.setup.Environment;

public abstract class AbstractApplicationMain<T extends CoreConfiguration> extends Application<T> {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractApplicationMain.class);

    @Inject
    protected ApplicationServerConfig applicationServerConfig;
    protected static final Terminal terminal = new Terminal();

    public static void main(final String... args) throws Exception {
        try {
            StartMain.main(args);
        } catch (final Exception e) {
            LOG.error("Could not able to start the application.\n" +
                    "Please make sure you have an empty beans.xml file in your META-INF folder under 'src/main/resources/META-INF' folder. "
                    + "\nThis is because we are using CDI powered by WELD ('http://weld.cdi-spec.org/')", e);
            System.exit(-1);
            throw e;
        }
    }

    @Inject
    private void initServices(@Any final Instance<AutoStartable> services) throws Exception {
        for (final AutoStartable service : services) {
            service.init();
        }
    }

    /**
     * Get array of application packages to load.
     * @return ApplicationPackagesToLoad array
     **/
    protected Set<String> getApplicationPackagesToLoad() {
        return applicationServerConfig.getCoreConfiguration().getApplicationPackagesToLoad();
    }

    /**
     * Override the method in case of mapping of new servlets.
     * @return The key value pair for mapped servlet url and servlet class.
     **/
    protected Map<String, Class<? extends HttpServlet>> getServlets() {
        return Collections.emptyMap();
    }

    /**
     * Override the method in case of adding new beans to the server.
     * @return List of Object The list of beans to be added to the server.
     **/
    protected List<Object> getBeans() {
        return Collections.emptyList();
    }

    /**
     * Override this method if you want to perform operation during application
     * startup.
     * @param bootstrap
     */
    protected void initApplication(final Bootstrap<T> bootstrap) {
    }

    /**
     * Override this method if you want to perform any operation after application server
     * started.
     */
    protected void postApplicationStartup() {
    }

    protected void configureApplicationBeforeStartup(final T configuration, final Environment environment) {
    }

    /**
     * Override this method if you want to perform any resource clean up while
     * shutting down
     */
    protected void shutdown() {
    }

    public final void start(@Observes final ContainerInitialized event) throws Exception {
        try {

            String[] args = StartMain.getParameters();
            if (args.length == 0) {
                args = new String[2];
                args[0] = terminal.readString("Please provide the command", "server");
                args[1] = terminal.readString("Please provide the application yaml configuration file", "application.properties.yml");
            }
            run(args);
        } catch (final Exception e) {
            LOG.error("Could not able to start the application.\n" +
                    "Please make sure you have an empty beans.xml file in your META-INF folder under 'src/main/resources/META-INF' folder. "
                    + "\nThis is because we are using CDI powered by WELD ('http://weld.cdi-spec.org/')", e);
            System.exit(-1);
            throw e;
        }
    }

    @Override
    public String getName() {
        return StringUtils.defaultString(applicationServerConfig.getCoreConfiguration().getApplicationName(), super.getName());
    }

    @Override
    public final void initialize(final Bootstrap<T> bootstrap) {
        initApplication(bootstrap);

        for (final Command command : getCommands()) {
            if (command != null) {
                bootstrap.addCommand(command);
            }
        }

        for (final ConfiguredCommand<? super T> bundle : getConfiguredCommand()) {
            if (bundle != null) {
                bootstrap.addCommand(bundle);
            }
        }

        for (final ConfiguredBundle<? super T> bundle : getConfiguredBundles()) {
            if (bundle != null) {
                bootstrap.addBundle(bundle);
            }
        }

        for (final Bundle bundle : getBundles()) {
            if (bundle != null) {
                bootstrap.addBundle(bundle);
            }
        }

    }

    protected List<Command> getCommands() {
        return Collections.emptyList();
    }

    protected List<ConfiguredCommand<? super T>> getConfiguredCommand() {
        return Collections.emptyList();
    }

    protected List<Bundle> getBundles() {
        return Collections.emptyList();
    }

    protected List<ConfiguredBundle<? super T>> getConfiguredBundles() {
        return Collections.emptyList();
    }

    @Override
    public final void run(final T config, final Environment environment) throws Exception {
        SharedMetricRegistries.add("matrix", environment.metrics());
        environment.getObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        environment.getObjectMapper().setSerializationInclusion(Include.NON_NULL);

        final MutableServletContextHandler applicationContext = environment.getApplicationContext();
        final JerseyEnvironment jersey = environment.jersey();
        jersey.getResourceConfig().property(ServerProperties.WADL_FEATURE_DISABLE, false);
        jersey.getResourceConfig().packages(getApplicationPackagesToLoad().toArray(new String[] {}));

        applicationContext.addFilter(LogTraceIdFilter.class, "/*", EnumSet.allOf(DispatcherType.class));

        for (final Object bean : getBeans()) {
            if (bean != null) {
                applicationContext.addBean(bean);
            }
        }

        for (final Entry<String, Class<? extends HttpServlet>> entry : getServlets().entrySet()) {
            applicationContext.addServlet(entry.getValue(), entry.getKey());
        }

        final FilterRegistration.Dynamic filter = environment.servlets().addFilter("CORS", CrossOriginFilter.class);
        filter.addMappingForUrlPatterns(EnumSet.allOf(DispatcherType.class), true, "/*");
        filter.setInitParameter("allowedOrigins", "*");
        filter.setInitParameter("allowedHeaders", "Content-Type,Authorization,X-Requested-With,Content-Length,Accept,Origin,X-Signal-Agent");
        filter.setInitParameter("allowedMethods", "GET,PUT,POST,DELETE,OPTIONS");
        filter.setInitParameter("preflightMaxAge", "5184000");
        filter.setInitParameter("allowCredentials", "true");

        environment.metrics().register(MetricRegistry.name(CpuUsageGauge.class, "cpu"), new CpuUsageGauge());
        environment.metrics().register(MetricRegistry.name(FreeMemoryGauge.class, "free_memory"), new FreeMemoryGauge());

        configureApplicationBeforeStartup(config, environment);
        startCli();
        postApplicationStartup();
    }

    private void startCli() throws Exception {
        final Integer adminPort = applicationServerConfig.getCoreConfiguration().getAdminPort();
        if (adminPort == null) {
            LOG.info("No cli is enabled.");
            return;
        }
        final CliBuilder cb = new CliBuilder();
        final Set<Class<? extends AbstractCommand>> cliClasses = ClassUtils.findSubClasses(AbstractCommand.class);
        for (final Class<? extends AbstractCommand> clazz : cliClasses) {
            LOG.info("Starting admin cli: " + clazz.getName());
            cb.addCommands(clazz.newInstance());
        }

        final CliClientHandler cli = new TelnetClientHandler(StringUtils.capitalize(getName()), cb.createCli());
        LOG.info("Starting admin cli at port: " + adminPort);
        CliServerFactory.startCli(adminPort, cli);
    }

}
