/*
 * Decompiled with CFR 0.152.
 */
package com.predic8.membrane.core.interceptor.json;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.CountingInputStream;
import com.predic8.membrane.annot.MCAttribute;
import com.predic8.membrane.annot.MCElement;
import com.predic8.membrane.core.exceptions.ProblemDetails;
import com.predic8.membrane.core.exchange.Exchange;
import com.predic8.membrane.core.http.Response;
import com.predic8.membrane.core.interceptor.AbstractInterceptor;
import com.predic8.membrane.core.interceptor.Interceptor;
import com.predic8.membrane.core.interceptor.Outcome;
import com.predic8.membrane.core.interceptor.json.JsonProtectionException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.EnumSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@MCElement(name="jsonProtection")
public class JsonProtectionInterceptor
extends AbstractInterceptor {
    private static final Logger log = LoggerFactory.getLogger(JsonProtectionInterceptor.class);
    private final ObjectMapper om = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_READING_DUP_TREE_KEY, true).configure(JsonParser.Feature.STRICT_DUPLICATE_DETECTION, true);
    private Boolean reportError;
    private int maxTokens = 10000;
    private int maxSize = 0x3200000;
    private int maxDepth = 50;
    private int maxStringLength = 262144;
    private int maxKeyLength = 256;
    private int maxObjectSize = 1000;
    private int maxArraySize = 1000;

    public JsonProtectionInterceptor() {
        this.name = "json protection";
        this.setFlow(EnumSet.of(Interceptor.Flow.REQUEST));
    }

    @Override
    public void init() {
        super.init();
        if (this.maxStringLength < this.maxKeyLength) {
            this.maxKeyLength = this.maxStringLength;
        }
    }

    private boolean shouldProvideDetails() {
        if (this.reportError != null) {
            return this.reportError;
        }
        return !this.router.isProduction();
    }

    @Override
    public Outcome handleRequest(Exchange exc) {
        if ("GET".equals(exc.getRequest().getMethod())) {
            return Outcome.CONTINUE;
        }
        try {
            this.parseJson(new CountingInputStream(exc.getRequest().getBodyAsStreamDecoded()));
        }
        catch (JsonProtectionException e) {
            log.debug(e.getMessage());
            exc.setResponse(this.createErrorResponse(e.getMessage(), e.getLine(), e.getCol()));
            return Outcome.RETURN;
        }
        catch (JsonParseException e) {
            log.debug(e.getMessage());
            exc.setResponse(this.createErrorResponse(e.getMessage(), e.getLocation().getLineNr(), e.getLocation().getColumnNr()));
            return Outcome.RETURN;
        }
        catch (Throwable e) {
            log.debug(e.getMessage());
            exc.setResponse(this.createErrorResponse(e.getMessage(), null, null));
            return Outcome.RETURN;
        }
        return Outcome.CONTINUE;
    }

    private Response createErrorResponse(String msg, Integer line, Integer col) {
        if (this.shouldProvideDetails()) {
            log.warn("JSON protection violation. Line: {}, col: {}, msg: {}", new Object[]{line, col, msg});
            ProblemDetails pd = ProblemDetails.user(false, this.getDisplayName()).statusCode(400).title("JSON Protection Violation").detail(msg);
            if (line != null) {
                pd.topLevel("line", line);
            }
            if (col != null) {
                pd.topLevel("column", col);
            }
            return pd.build();
        }
        return Response.badRequest().build();
    }

    private void parseJson(CountingInputStream cis) throws IOException, JsonProtectionException {
        JsonToken jsonToken;
        JsonParser parser = this.om.createParser((InputStream)cis);
        int tokenCount = 0;
        int depth = 0;
        ArrayList<Context> contexts = new ArrayList<Context>();
        Context currentContext = null;
        block8: while ((jsonToken = parser.nextValue()) != null) {
            if (++tokenCount > this.maxTokens) {
                throw new JsonProtectionException("Exceeded maxTokens.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
            }
            if (cis.getCount() > (long)this.maxSize) {
                throw new JsonProtectionException("Exceeded maxSize.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
            }
            if (currentContext != null) {
                currentContext.check(jsonToken, parser);
            }
            switch (jsonToken.id()) {
                case 1: {
                    if (++depth > this.maxDepth) {
                        throw new JsonProtectionException("Exceeded maxDepth.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
                    }
                    currentContext = new ObjContext();
                    contexts.add(currentContext);
                    continue block8;
                }
                case 3: {
                    if (++depth > this.maxArraySize) {
                        throw new JsonProtectionException("Exceeded maxArraySize.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
                    }
                    currentContext = new ArrContext();
                    contexts.add(currentContext);
                    continue block8;
                }
                case 2: 
                case 4: {
                    if (--depth < 0) {
                        throw new JsonProtectionException("Invalid JSON Document.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
                    }
                    contexts.removeLast();
                    currentContext = contexts.isEmpty() ? null : (Context)contexts.getLast();
                    continue block8;
                }
                case 6: {
                    if (parser.getValueAsString().length() <= this.maxStringLength) continue block8;
                    throw new JsonProtectionException("Exceeded maxStringLength.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
                }
                case 7: 
                case 8: 
                case 9: 
                case 10: 
                case 11: {
                    continue block8;
                }
                case -1: 
                case 0: 
                case 5: 
                case 12: {
                    throw new JsonProtectionException("Not handled.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
                }
            }
            throw new JsonProtectionException("Not handled (\" + jsonToken.id() + \")", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
        }
        if (cis.getCount() > (long)this.maxSize) {
            throw new JsonProtectionException("Exceeded maxSize.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
        }
    }

    public int getMaxTokens() {
        return this.maxTokens;
    }

    @MCAttribute
    public void setReportError(boolean reportError) {
        this.reportError = reportError;
    }

    public Boolean getReportError() {
        return this.reportError;
    }

    @MCAttribute
    public void setMaxTokens(int maxTokens) {
        this.maxTokens = maxTokens;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    @MCAttribute
    public void setMaxSize(int maxSize) {
        this.maxSize = maxSize;
    }

    public int getMaxDepth() {
        return this.maxDepth;
    }

    @MCAttribute
    public void setMaxDepth(int maxDepth) {
        this.maxDepth = maxDepth;
    }

    public int getMaxStringLength() {
        return this.maxStringLength;
    }

    @MCAttribute
    public void setMaxStringLength(int maxStringLength) {
        this.maxStringLength = maxStringLength;
    }

    public int getMaxKeyLength() {
        return this.maxKeyLength;
    }

    @MCAttribute
    public void setMaxKeyLength(int maxKeyLength) {
        this.maxKeyLength = maxKeyLength;
    }

    public int getMaxObjectSize() {
        return this.maxObjectSize;
    }

    @MCAttribute
    public void setMaxObjectSize(int maxObjectSize) {
        this.maxObjectSize = maxObjectSize;
    }

    public int getMaxArraySize() {
        return this.maxArraySize;
    }

    @MCAttribute
    public void setMaxArraySize(int maxArraySize) {
        this.maxArraySize = maxArraySize;
    }

    @Override
    public String getShortDescription() {
        return "Protects against several JSON attack classes.";
    }

    @Override
    public String getLongDescription() {
        return "<div>Enforces the following constraints:<br/><ul><li>HTTP request body must be well-formed JSON, if the HTTP verb is not<font style=\"font-family: monospace\">GET</font>.</li><li>Limits the maximum number of tokens to " + this.maxTokens + ". (Each string and opening bracket countsas a token: <font style=\"font-family: monospace\">{\"a\":\"b\"}</font> counts as 3 tokens)</li><li>Forbids duplicate keys. (<font style=\"font-family: monospace\">{\"a\":\"b\", \"a\":\"c\"}</font> will be rejected.)</li><li>Limits the total size in bytes of the body to " + this.maxSize + ".</li><li>Limits the maximum depth to " + this.maxDepth + ". (<font style=\"font-family: monospace\">{\"a\":[{\"b\":\"c\"}]}</font> has depth 3.)</li><li>Limits the maximum string length to " + this.maxStringLength + ". (<font style=\"font-family: monospace\">{\"a\":\"abc\"}</font> has max string length 3.)</li><li>Limits the maximum key length to " + this.maxKeyLength + ". (<font style=\"font-family: monospace\">{\"abc\":\"a\"}</font> has key length 3.)</li><li>Limits the maximum object size to " + this.maxObjectSize + ". (<font style=\"font-family: monospace\">{\"a\":\"b\",\"c\":\"d\"}</font> has object size 2.)</li><li>Limits the maximum array size to " + this.maxArraySize + ". (<font style=\"font-family: monospace\">[\"a\", \"b\"]</font> has array size 2.)</li></ul></div>";
    }

    private static abstract class Context {
        private Context() {
        }

        public abstract void check(JsonToken var1, JsonParser var2) throws IOException, JsonProtectionException;
    }

    private class ObjContext
    extends Context {
        int n;

        private ObjContext() {
        }

        @Override
        public void check(JsonToken jsonToken, JsonParser parser) throws JsonProtectionException, IOException {
            if (jsonToken.id() == 2) {
                return;
            }
            ++this.n;
            if (this.n > JsonProtectionInterceptor.this.maxObjectSize) {
                throw new JsonProtectionException("Exceeded maxObjectSize.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
            }
            if (parser.currentName().length() > JsonProtectionInterceptor.this.maxKeyLength) {
                throw new JsonProtectionException("Exceeded maxKeyLength.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
            }
        }
    }

    private class ArrContext
    extends Context {
        int n;

        private ArrContext() {
        }

        @Override
        public void check(JsonToken jsonToken, JsonParser parser) throws JsonProtectionException {
            if (jsonToken.id() == 4) {
                return;
            }
            ++this.n;
            if (this.n > JsonProtectionInterceptor.this.maxArraySize) {
                throw new JsonProtectionException("Exceeded maxArraySize.", parser.currentLocation().getLineNr(), parser.currentLocation().getColumnNr());
            }
        }
    }
}

