/*
 * Decompiled with CFR 0.152.
 */
package com.predic8.membrane.core.openapi.validators;

import com.predic8.membrane.core.openapi.model.Request;
import com.predic8.membrane.core.openapi.validators.ValidationContext;
import com.predic8.membrane.core.openapi.validators.ValidationError;
import com.predic8.membrane.core.openapi.validators.ValidationErrors;
import com.predic8.membrane.core.security.ApiKeySecurityScheme;
import com.predic8.membrane.core.security.BasicHttpSecurityScheme;
import com.predic8.membrane.core.security.JWTSecurityScheme;
import com.predic8.membrane.core.security.OAuth2SecurityScheme;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.Operation;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecurityValidator {
    private final Logger log = LoggerFactory.getLogger(SecurityValidator.class);
    private final OpenAPI api;

    public SecurityValidator(OpenAPI api) {
        this.api = api;
    }

    public ValidationErrors validateSecurity(ValidationContext ctx, Request<?> request, Operation operation) {
        ctx = ctx.statusCode(403);
        ValidationErrors errors = this.checkGlobalSecurity(ctx, request);
        errors.add(this.checkOperationSecurity(ctx, operation, request));
        return errors;
    }

    private ValidationErrors checkGlobalSecurity(ValidationContext ctx, Request<?> request) {
        if (this.api.getSecurity() == null) {
            return new ValidationErrors();
        }
        return this.validateDisjunctiveSecurityRequirement(ctx, this.api.getSecurity(), request);
    }

    private ValidationErrors checkOperationSecurity(ValidationContext ctx, Operation operation, Request<?> request) {
        if (operation.getSecurity() == null) {
            return new ValidationErrors();
        }
        return this.validateDisjunctiveSecurityRequirement(ctx, operation.getSecurity(), request);
    }

    private ValidationErrors validateDisjunctiveSecurityRequirement(ValidationContext ctx, List<SecurityRequirement> requirements, Request<?> request) {
        ValidationErrors errors = new ValidationErrors();
        for (SecurityRequirement requirement : requirements) {
            ValidationErrors errorsInt = this.checkSecurityRequirements(ctx, requirement, request);
            if (errorsInt.isEmpty()) {
                return new ValidationErrors();
            }
            errors.add(errorsInt);
        }
        return errors;
    }

    private ValidationErrors checkSecurityRequirements(ValidationContext ctx, SecurityRequirement requirement, Request<?> request) {
        return requirement.keySet().stream().map(requirementName -> this.checkSingleRequirement(ctx, requirement, request, (String)requirementName)).reduce(new ValidationErrors(), ValidationErrors::add);
    }

    private ValidationErrors checkSingleRequirement(ValidationContext ctx, SecurityRequirement requirement, Request<?> request, String schemeName) {
        this.log.debug("Checking mechanism: " + schemeName);
        ValidationErrors errors = new ValidationErrors();
        Map securitySchemes = this.api.getComponents().getSecuritySchemes();
        if (securitySchemes == null) {
            this.log.error("In OpenAPI with title '%s' there are no securitySchemes. Check the OpenAPI document!".formatted(this.getOpenAPITitle()));
            return SecurityValidator.getValidationErrorsProblemServerSide(ctx);
        }
        SecurityScheme schemeDefinition = (SecurityScheme)securitySchemes.get(schemeName);
        if (schemeDefinition == null) {
            this.log.error("In OpenAPI with title '%s' there is no securityScheme '%s'. Check the OpenAPI document!".formatted(this.getOpenAPITitle(), schemeName));
            return SecurityValidator.getValidationErrorsProblemServerSide(ctx);
        }
        if (schemeDefinition.getType() == null) {
            this.log.error("In OpenAPI with title '%s' the securityScheme '%s' has no type. Check the OpenAPI document!".formatted(this.api.getInfo().getTitle(), schemeName));
            return SecurityValidator.getValidationErrorsProblemServerSide(ctx);
        }
        ValidationErrors errorsInt = this.checkSecuritySchemeType(ctx, request, schemeDefinition, errors);
        assert (errorsInt != null);
        if (!errorsInt.isEmpty()) {
            return errorsInt;
        }
        errors.add(this.checkScopes(ctx, requirement, request, schemeName));
        return errors;
    }

    @NotNull
    private static ValidationErrors getValidationErrorsProblemServerSide(ValidationContext ctx) {
        return ValidationErrors.create(ctx, "There is a problem with the OpenAPI configuration at the server side.");
    }

    private String getOpenAPITitle() {
        return this.api.getInfo().getTitle();
    }

    @Nullable
    private ValidationErrors checkSecuritySchemeType(ValidationContext ctx, Request<?> request, SecurityScheme scheme, ValidationErrors errors) {
        return switch (scheme.getType()) {
            default -> throw new MatchException(null, null);
            case SecurityScheme.Type.HTTP -> SecurityValidator.checkHttp(ctx, request, scheme);
            case SecurityScheme.Type.APIKEY -> errors.add(this.checkApiKey(ctx, request, scheme));
            case SecurityScheme.Type.OAUTH2, SecurityScheme.Type.OPENIDCONNECT -> this.checkOAuth2OrOpenIdConnectScheme(ctx, request);
            case SecurityScheme.Type.MUTUALTLS -> throw new RuntimeException("Security scheme mutualTLS is not implemented yet.");
        };
    }

    private ValidationErrors checkScopes(ValidationContext ctx, SecurityRequirement requirement, Request<?> request, String schemeName) {
        ValidationErrors errors = new ValidationErrors();
        for (String scope : (List)requirement.get((Object)schemeName)) {
            this.log.debug("Checking scope: " + scope);
            AtomicBoolean hasScope = new AtomicBoolean();
            request.getSecuritySchemes().forEach(scheme -> {
                if (scheme.hasScope(scope)) {
                    hasScope.set(true);
                }
            });
            if (hasScope.get()) continue;
            this.log.info("Caller of {} {} ist not in scope {} required by OpenAPI definition.", new Object[]{ctx.getMethod(), ctx.getPath(), scope});
            errors.add(ctx, "Caller ist not in scope %s".formatted(scope));
        }
        return errors;
    }

    private static ValidationErrors checkHttp(ValidationContext ctx, Request<?> request, SecurityScheme schemeDefinition) {
        ValidationErrors errors = new ValidationErrors();
        switch (schemeDefinition.getScheme().toLowerCase()) {
            case "basic": {
                if (request.hasScheme(BasicHttpSecurityScheme.BASIC())) {
                    return errors;
                }
                errors.add(ctx.statusCode(401), "Caller ist not authenticated with HTTP and %s.".formatted(BasicHttpSecurityScheme.BASIC()));
                return errors;
            }
            case "bearer": {
                if (request.hasScheme(BasicHttpSecurityScheme.BEARER())) {
                    return errors;
                }
                errors.add(ctx.statusCode(401), "Caller ist not authenticated with HTTP and %s.".formatted(BasicHttpSecurityScheme.BEARER()));
                return errors;
            }
        }
        errors.add(ctx.statusCode(401), "Scheme %s is not supported".formatted(schemeDefinition.getScheme()));
        return errors;
    }

    private ValidationErrors checkOAuth2OrOpenIdConnectScheme(ValidationContext ctx, Request<?> request) {
        if (this.securitySchemeIsNotPresent(request, OAuth2SecurityScheme.class) && this.securitySchemeIsNotPresent(request, JWTSecurityScheme.class)) {
            return ValidationErrors.create(ctx.statusCode(401), "OAuth2 or JWT authentication is required.");
        }
        return ValidationErrors.empty();
    }

    private ValidationErrors checkApiKey(ValidationContext ctx, Request<?> request, SecurityScheme securityScheme) {
        ValidationErrors errors = new ValidationErrors();
        AtomicBoolean schemeIsInRequest = new AtomicBoolean();
        List<ValidationError> e = this.getSecuritySchemes(request, ApiKeySecurityScheme.class).map(scheme1 -> {
            if (scheme1 instanceof ApiKeySecurityScheme) {
                ApiKeySecurityScheme apiKeySecurityScheme = (ApiKeySecurityScheme)scheme1;
                schemeIsInRequest.set(true);
                if (securityScheme.getName() != null && !securityScheme.getName().equalsIgnoreCase(apiKeySecurityScheme.parameterName)) {
                    return Optional.of(new ValidationError(ctx, "Name of api-key is %s but should be %s".formatted(apiKeySecurityScheme.parameterName, securityScheme.getName())));
                }
                if (securityScheme.getIn() != null && !securityScheme.getIn().toString().equalsIgnoreCase(apiKeySecurityScheme.in.toString())) {
                    return Optional.of(new ValidationError(ctx, "Api-key is in %s but should be in %s".formatted(new Object[]{apiKeySecurityScheme.in, securityScheme.getIn()})));
                }
            }
            return Optional.empty();
        }).flatMap(Optional::stream).toList();
        if (!schemeIsInRequest.get()) {
            errors.add(ctx.statusCode(401), "Authentication by API key is required.");
        }
        errors.add(e);
        return errors;
    }

    private boolean securitySchemeIsNotPresent(Request<?> request, Class<? extends com.predic8.membrane.core.security.SecurityScheme> clazz) {
        return this.getSecuritySchemes(request, clazz).findFirst().isEmpty();
    }

    private Stream<com.predic8.membrane.core.security.SecurityScheme> getSecuritySchemes(Request<?> request, Class<? extends com.predic8.membrane.core.security.SecurityScheme> clazz) {
        if (request.getSecuritySchemes() == null) {
            return Stream.empty();
        }
        return request.getSecuritySchemes().stream().filter(scheme -> scheme.getClass().equals(clazz));
    }
}

