/*
 * Decompiled with CFR 0.152.
 */
package com.predic8.membrane.core.transport.ssl.acme;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.predic8.membrane.core.transport.ssl.AcmeSSLContext;
import com.predic8.membrane.core.transport.ssl.PEMSupport;
import com.predic8.membrane.core.transport.ssl.SSLContext;
import com.predic8.membrane.core.transport.ssl.acme.AcmeClient;
import com.predic8.membrane.core.transport.ssl.acme.AcmeErrorLog;
import com.predic8.membrane.core.transport.ssl.acme.AcmeException;
import com.predic8.membrane.core.transport.ssl.acme.AcmeKeyPair;
import com.predic8.membrane.core.transport.ssl.acme.AcmeSynchronizedStorageEngine;
import com.predic8.membrane.core.transport.ssl.acme.Authorization;
import com.predic8.membrane.core.transport.ssl.acme.Challenge;
import com.predic8.membrane.core.transport.ssl.acme.FatalAcmeException;
import com.predic8.membrane.core.transport.ssl.acme.OrderAndLocation;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AcmeRenewal {
    private static final Logger LOG = LoggerFactory.getLogger(AcmeRenewal.class);
    private static final long ERROR_WAIT_MILLISECONDS = 900000L;
    private static final long LEASE_DURATION_MILLISECONDS = 300000L;
    private static final long LEASE_RENEW_MILLISECONDS = 240000L;
    private final AcmeSynchronizedStorageEngine asse;
    private final String[] hosts;
    private final AcmeClient client;
    private final ObjectMapper om;

    public AcmeRenewal(AcmeClient client, String[] hosts) {
        this.client = client;
        this.asse = client.getAsse();
        this.hosts = hosts;
        this.om = new ObjectMapper().registerModule((Module)new JodaModule());
    }

    public void doWork() {
        if (!this.requiresWork()) {
            return;
        }
        this.withMasterLease(() -> {
            try {
                this.tryGetCertificate();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                e.printStackTrace();
                try {
                    Thread.sleep(900000L);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
            }
        });
    }

    private void tryGetCertificate() throws Exception {
        this.client.loadDirectory();
        this.verifyAccountContact();
        if (this.getAccountURL() == null) {
            this.client.ensureAccountKeyExists();
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): storing account URL");
            }
            this.setAccountURL(this.client.createAccount());
        }
        if (this.isOALExpiredOrError()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): archiving OAL");
            }
            this.client.getAsse().archiveOAL(this.hosts);
        }
        if (this.getOAL() == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): creating OAL");
            }
            this.setOAL(this.client.createOrder(this.getAccountURL(), Arrays.asList(this.hosts)));
        }
        AtomicReference<OrderAndLocation> oal = new AtomicReference<OrderAndLocation>(this.getOAL());
        try {
            this.makeOrderValid(oal);
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): downloading certificate");
            }
            String certs = this.client.downloadCertificate(this.getAccountURL(), oal.get().getOrder().getCertificate());
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): promoting key+cert to production");
            }
            this.asse.setKeyPair(this.hosts, this.client.getOALKey(this.hosts));
            this.asse.setCertChain(this.hosts, certs);
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): retiring OAL");
            }
            this.asse.archiveOAL(this.hosts);
        }
        catch (Exception e) {
            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw));
            this.client.setOALError(this.hosts, new AcmeErrorLog(sw.toString(), e instanceof FatalAcmeException, new DateTime()));
            throw e;
        }
    }

    private void makeOrderValid(AtomicReference<OrderAndLocation> oal) throws Exception {
        oal.set(this.client.getOrder(this.getAccountURL(), oal.get().getLocation()));
        if (LOG.isDebugEnabled()) {
            LOG.debug("acme (" + this.id() + "): order is " + oal.get().getOrder().getStatus());
        }
        if ("pending".equals(oal.get().getOrder().getStatus())) {
            this.fulfillChallenges(oal.get());
            oal.set(this.client.getOrder(this.getAccountURL(), oal.get().getLocation()));
            this.waitFor("order to become non-'PENDING'", () -> !"pending".equals(((OrderAndLocation)oal.get()).getOrder().getStatus()), () -> oal.set(this.client.getOrder(this.getAccountURL(), ((OrderAndLocation)oal.get()).getLocation())));
            if (!"ready".equals(oal.get().getOrder().getStatus())) {
                throw new FatalAcmeException("order status " + this.om.writeValueAsString(oal));
            }
        }
        if ("ready".equals(oal.get().getOrder().getStatus())) {
            if (this.client.getOALKey(this.hosts) == null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("acme (" + this.id() + "): generating certificate key");
                }
                AcmeKeyPair key = this.client.generateCertificateKey();
                this.client.setOALKey(this.hosts, key);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): finalizing order");
            }
            try {
                this.client.finalizeOrder(this.getAccountURL(), oal.get().getOrder().getFinalize(), this.client.generateCSR(this.hosts, this.client.getOALKey(this.hosts).getPrivateKey()));
            }
            catch (AcmeException e) {
                if (e.getMessage().contains("urn:ietf:params:acme:error:orderNotReady Order's status (\"valid\") is not acceptable for finalization")) {
                    oal.set(this.client.getOrder(this.getAccountURL(), oal.get().getLocation()));
                    return;
                }
                throw e;
            }
        }
        this.waitFor("order to become 'VALID'", () -> !"ready".equals(((OrderAndLocation)oal.get()).getOrder().getStatus()) && !"processing".equals(((OrderAndLocation)oal.get()).getOrder().getStatus()), () -> oal.set(this.client.getOrder(this.getAccountURL(), ((OrderAndLocation)oal.get()).getLocation())));
        if (!"valid".equals(oal.get().getOrder().getStatus())) {
            throw new FatalAcmeException("order status " + this.om.writeValueAsString(oal));
        }
    }

    private String id() {
        return this.hosts[0] + (this.hosts.length > 1 ? ",..." : "");
    }

    private void fulfillChallenges(OrderAndLocation oal) throws Exception {
        for (String authorization : oal.getOrder().getAuthorizations()) {
            AtomicReference<Authorization> auth = new AtomicReference<Authorization>(this.client.getAuth(this.getAccountURL(), authorization));
            AtomicReference<Challenge> challenge = new AtomicReference<Challenge>(this.getChallenge(auth.get()));
            if (LOG.isDebugEnabled()) {
                LOG.debug("acme (" + this.id() + "): authorization is " + auth.get().getStatus() + ", challenge is " + challenge.get().getStatus());
            }
            if ("pending".equals(challenge.get().getStatus())) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("acme (" + this.id() + "): provisioning challenge " + auth.get().getIdentifier().getValue());
                }
                String challengeUrl = this.client.provision(auth.get());
                if (LOG.isDebugEnabled()) {
                    LOG.debug("acme (" + this.id() + "): triggering challenge check " + auth.get().getIdentifier().getValue());
                }
                this.client.readyForChallenge(this.getAccountURL(), challengeUrl);
            }
            this.waitFor("challenge and authorization to become non-'PENDING'", () -> !"pending".equals(((Challenge)challenge.get()).getStatus()) || !"pending".equals(((Authorization)auth.get()).getStatus()), () -> {
                auth.set(this.client.getAuth(this.getAccountURL(), authorization));
                challenge.set(this.getChallenge((Authorization)auth.get()));
            });
            if (!"valid".equals(challenge.get().getStatus())) {
                throw new FatalAcmeException(challenge.get().getStatus() + " during " + this.om.writeValueAsString(auth));
            }
            if ("valid".equals(auth.get().getStatus())) continue;
            throw new FatalAcmeException(auth.get().getStatus() + " during " + this.om.writeValueAsString(auth));
        }
    }

    private void waitFor(String what, Supplier<Boolean> condition, Runnable job) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("acme (" + this.id() + "): waiting for " + what);
        }
        long now = System.currentTimeMillis();
        long wait = 1000L;
        while (!condition.get().booleanValue()) {
            Thread.sleep(wait);
            if (wait < 20000L) {
                wait *= 2L;
            }
            job.run();
            if (System.currentTimeMillis() - now <= 300000L) continue;
            throw new RuntimeException("Timeout (5min) while waiting for " + what + ".");
        }
    }

    private Challenge getChallenge(Authorization auth) throws JsonProcessingException, FatalAcmeException {
        Optional<Challenge> challenge = auth.getChallenges().stream().filter(c -> this.client.getChallengeType().equals(c.getType())).findAny();
        if (challenge.isEmpty()) {
            throw new FatalAcmeException("Could not find challenge of type " + this.client.getChallengeType() + ": " + this.om.writeValueAsString((Object)auth));
        }
        return challenge.get();
    }

    private void verifyAccountContact() {
        String contacts = String.join((CharSequence)",", this.client.getContacts());
        if (this.asse.getAccountContacts() == null) {
            this.asse.setAccountContacts(contacts);
        } else if (!contacts.equals(this.asse.getAccountContacts())) {
            throw new RuntimeException("It looks like you pointed an ACME client configured with '" + contacts + "' as contact to a storage where a key for '" + this.asse.getAccountContacts() + "' is present.");
        }
    }

    private OrderAndLocation getOAL() throws JsonProcessingException {
        String oal = this.asse.getOAL(this.hosts);
        if (oal == null) {
            return null;
        }
        return (OrderAndLocation)this.om.readValue(oal, OrderAndLocation.class);
    }

    private void setOAL(OrderAndLocation oal) throws JsonProcessingException {
        this.asse.setOAL(this.hosts, this.om.writeValueAsString((Object)oal));
    }

    private String getAccountURL() {
        return this.asse.getAccountURL();
    }

    private void setAccountURL(String url) {
        this.asse.setAccountURL(url);
    }

    private boolean requiresWork() {
        String certificates = this.client.getCertificates(this.hosts);
        if (certificates == null) {
            return true;
        }
        try {
            ArrayList<Certificate> certs = new ArrayList<Certificate>(PEMSupport.getInstance().parseCertificates(certificates));
            if (certs.size() == 0) {
                return true;
            }
            long validFrom = SSLContext.getValidFrom(certs);
            long validUntil = SSLContext.getMinimumValidity(certs);
            return System.currentTimeMillis() > AcmeSSLContext.renewAt(validFrom, validUntil);
        }
        catch (IOException e) {
            LOG.warn("Error parsing ACME certificate " + Arrays.toString(this.hosts), (Throwable)e);
            return true;
        }
    }

    private boolean isOALExpiredOrError() throws JsonProcessingException {
        OrderAndLocation oal = this.getOAL();
        if (oal != null && oal.getOrder().getExpires().isAfterNow()) {
            return true;
        }
        AcmeErrorLog error = this.client.getOALError(this.hosts);
        if (error != null) {
            long wait = error.getTime().getMillis() + 900000L - System.currentTimeMillis();
            if (wait > 0L) {
                try {
                    LOG.warn("Waiting " + wait / 1000L + " seconds after ACME order error...");
                    Thread.sleep(wait);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
            return true;
        }
        return false;
    }

    private boolean withMasterLease(Runnable runnable) {
        if (!this.client.getAsse().acquireLease(300000L)) {
            return false;
        }
        AtomicReference error = new AtomicReference();
        Thread t = new Thread(() -> {
            try {
                runnable.run();
            }
            catch (InterruptedException interruptedException) {
            }
            catch (Throwable e) {
                error.set(e);
            }
        });
        t.start();
        do {
            try {
                t.join(240000L);
                if (!t.isAlive()) {
                    this.client.getAsse().releaseLease();
                    if (error.get() != null) {
                        throw new RuntimeException((Throwable)error.get());
                    }
                    return true;
                }
            }
            catch (InterruptedException e) {
                t.interrupt();
                Thread.currentThread().interrupt();
            }
        } while (this.client.getAsse().prolongLease(300000L));
        t.interrupt();
        return true;
    }

    @FunctionalInterface
    public static interface Runnable {
        public void run() throws Exception;
    }

    @FunctionalInterface
    public static interface Supplier<T> {
        public T get() throws Exception;
    }
}

