/*
 * TeleStax, Open Source Cloud Communications
 * Copyright 2011-2014, Telestax Inc and individual contributors
 * by the @authors tag.
 *
 * This program is free software: you can redistribute it and/or modify
 * under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 */

package org.mobicents.servlet.sip.message;

import java.io.Serializable;

import javax.servlet.ServletException;
import javax.servlet.http.HttpSession;
import javax.servlet.sip.Address;
import javax.servlet.sip.AuthInfo;
import javax.servlet.sip.Parameterable;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.URI;

import org.apache.log4j.Logger;
import org.mobicents.javax.servlet.sip.SipFactoryExt;
import org.mobicents.javax.servlet.sip.SipSessionsUtilExt;
import org.mobicents.servlet.sip.core.SipContext;
import org.mobicents.servlet.sip.core.session.ConvergedSession;
import org.mobicents.servlet.sip.core.session.MobicentsSipApplicationSession;
import org.mobicents.servlet.sip.core.session.MobicentsSipSession;

/**
 * Facade object which masks the internal <code>SipFactoryImpl</code>
 * object from the web application.
 *
 * @author Jean Deruelle
 *
 */
public class SipFactoryFacade implements SipFactoryExt, Serializable {
	
	private static final long serialVersionUID = 1L;
	private static final Logger logger = Logger.getLogger(SipFactoryFacade.class
			.getName());
	
	private SipFactoryImpl sipFactoryImpl;
	private transient SipContext sipContext;
	private static transient ThreadLocal<HttpSession> threadLocalHttpSession = new ThreadLocal<HttpSession>();;
	
	public SipFactoryFacade(SipFactoryImpl sipFactoryImpl, SipContext sipContext) {
		this.sipFactoryImpl = sipFactoryImpl;
		this.sipContext = sipContext;
	}
	
	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createAddress(java.lang.String)
	 */
	public Address createAddress(String sipAddress) throws ServletParseException {
		return sipFactoryImpl.createAddress(sipAddress);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createAddress(javax.servlet.sip.URI)
	 */
	public Address createAddress(URI uri) {
		return sipFactoryImpl.createAddress(uri);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createAddress(javax.servlet.sip.URI, java.lang.String)
	 */
	public Address createAddress(URI uri, String displayName) {
		return sipFactoryImpl.createAddress(uri, displayName);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createApplicationSession()
	 */
	public SipApplicationSession createApplicationSession() {		
		return createApplicationSession(true);
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.mobicents.javax.servlet.sip.SipFactoryExt#createApplicationSession(boolean)
	 */
	public SipApplicationSession createApplicationSession(boolean managed) {
		MobicentsSipApplicationSession sipApplicationSessionImpl = null; 
		HttpSession httpSession = threadLocalHttpSession.get();
		// make sure we don't create a new sip app session if the http session has already one associated
		if(httpSession != null) {
			ConvergedSession convergedSession = (ConvergedSession) httpSession;
			sipApplicationSessionImpl = convergedSession.getApplicationSession(false);
		}
		if(sipApplicationSessionImpl == null) {
			sipApplicationSessionImpl =
				(MobicentsSipApplicationSession)sipFactoryImpl.createApplicationSessionByAppName(sipContext.getApplicationName(), managed);
			associateHttpSession(sipApplicationSessionImpl);
		}
		return sipApplicationSessionImpl;
	}

	/**
	 * A servlet wants to create a sip application
	 * session, we can retrieve its http session in the thread local data 
	 * if it has a sip http session (sip servlet only won't have any http sessions)  
	 * and associate it with the sip app session.
	 * @param sipApplicationSessionImpl the app session to assocaiate the http session with
	 */
	private void associateHttpSession(
			MobicentsSipApplicationSession sipApplicationSessionImpl) {
		
		HttpSession httpSession = threadLocalHttpSession.get();
		if(httpSession != null) {
			sipApplicationSessionImpl.addHttpSession(httpSession);
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createParameterable(java.lang.String)
	 */
	public Parameterable createParameterable(String s) throws ServletParseException {
		return sipFactoryImpl.createParameterable(s);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createRequest(javax.servlet.sip.SipApplicationSession, java.lang.String, javax.servlet.sip.Address, javax.servlet.sip.Address)
	 */
	public SipServletRequest createRequest(SipApplicationSession appSession,
			String method, Address from, Address to) {
		SipServletRequest sipServletRequest = sipFactoryImpl.createRequest(appSession, method, from, to, ((MobicentsSipApplicationSession)appSession).getCurrentRequestHandler(), null, null);
		return sipServletRequest;
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createRequest(javax.servlet.sip.SipApplicationSession, java.lang.String, java.lang.String, java.lang.String)
	 */
	public SipServletRequest createRequest(SipApplicationSession appSession,
			String method, String from, String to) throws ServletParseException {
		SipServletRequest sipServletRequest = sipFactoryImpl.createRequest(appSession, method, from, to, ((MobicentsSipApplicationSession)appSession).getCurrentRequestHandler());
		return sipServletRequest;
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createRequest(javax.servlet.sip.SipApplicationSession, java.lang.String, javax.servlet.sip.URI, javax.servlet.sip.URI)
	 */
	public SipServletRequest createRequest(SipApplicationSession appSession,
			String method, URI from, URI to) {
		SipServletRequest sipServletRequest = sipFactoryImpl.createRequest(appSession, method, from, to, ((MobicentsSipApplicationSession)appSession).getCurrentRequestHandler());
		return sipServletRequest;
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createRequest(javax.servlet.sip.SipServletRequest, boolean)
	 */
	public SipServletRequest createRequest(SipServletRequest origRequest,
			boolean sameCallId) {		
		SipServletRequest sipServletRequest = sipFactoryImpl.createRequest(origRequest, sameCallId);
		checkHandler(sipServletRequest);		
		return sipServletRequest;
	}

	/**
	 * set the handler for this request if none already exists.
	 * The handler will be the main servlet of the sip context 
	 * @param request the session of this request will have its handler set.
	 */
	private void checkHandler(SipServletRequest request) {
		MobicentsSipSession sipSessionImpl = (MobicentsSipSession)request.getSession();
		if(sipSessionImpl.getHandler() == null) {
			try {
				sipSessionImpl.setHandler(sipContext.getServletHandler());
//				((SipApplicationSessionImpl)sipSessionImpl.getApplicationSession()).setSipContext(sipContext);
			} catch (ServletException se) {
				//should never happen
				logger.error("Impossible to set the default handler on the newly created request "+ request.toString(),se);
			} 
		}
	}
	
	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createSipURI(java.lang.String, java.lang.String)
	 */
	public SipURI createSipURI(String user, String host) {
		return sipFactoryImpl.createSipURI(user, host);
	}

	/* (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createURI(java.lang.String)
	 */
	public URI createURI(String uri) throws ServletParseException {
		return sipFactoryImpl.createURI(uri);
	}
	
	/**
	 * Store the http session in the sip Factory's thread local 
	 * @param httpSession the http session to store for later retrieval
	 */
	public void storeHttpSession(HttpSession httpSession) {
		threadLocalHttpSession.set(httpSession);
	}
	
	/**
	 * Remove the http session in the sip Factory's thread local  
	 */
	public void removeHttpSession() {
		threadLocalHttpSession.set(null);
		threadLocalHttpSession.remove();
	}
	
	/**
	 * Retrieve the http session previously stored in the sip Factory's thread local
	 * @return the http session previously stored in the sip Factory's thread local
	 */
	public HttpSession retrieveHttpSession() {
		return threadLocalHttpSession.get();
	}

	/*
	 * (non-Javadoc)
	 * @see javax.servlet.sip.SipFactory#createApplicationSessionByKey(java.lang.String)
	 */
	public SipApplicationSession createApplicationSessionByKey(
			String sipApplicationKey) {		
		return createApplicationSessionByKey(sipApplicationKey, true);
	}
	
	
	public SipApplicationSession createApplicationSessionByKey(
			String sipApplicationKey, boolean managed) {
		MobicentsSipApplicationSession sipApplicationSessionImpl = null;
		// make sure we don't create a new sip app session if the http session has already one associated
		HttpSession httpSession = threadLocalHttpSession.get();
		if(httpSession != null) {
			ConvergedSession convergedSession = (ConvergedSession) httpSession;
			sipApplicationSessionImpl = convergedSession.getApplicationSession(false);
		}
		if(sipApplicationSessionImpl == null) {
			sipApplicationSessionImpl = (MobicentsSipApplicationSession) ((SipSessionsUtilExt)sipContext.getSipSessionsUtil()).getApplicationSessionByKey(sipApplicationKey, true, managed);
			associateHttpSession(sipApplicationSessionImpl);
		}
		return sipApplicationSessionImpl;
	}

	/**
	 * {@inheritDoc}
	 */
	public AuthInfo createAuthInfo() {		
		return sipFactoryImpl.createAuthInfo();
	}

	/**
	 * {@inheritDoc}
	 */
	public SipApplicationSession createApplicationSessionByAppName(String arg0) {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.mobicents.javax.servlet.sip.SipFactoryExt#isRouteOrphanRequests()
	 */
	public boolean isRouteOrphanRequests() {
		return sipFactoryImpl.isRouteOrphanRequests();
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.mobicents.javax.servlet.sip.SipFactoryExt#setRouteOrphanRequest(boolean)
	 */
	public void setRouteOrphanRequests(boolean routeOrphanRequets) {
		sipFactoryImpl.setRouteOrphanRequests(routeOrphanRequets);
	}

    @Override
    public SipServletRequest createRequestWithCallID(SipApplicationSession appSession, String method, Address from, Address to, String callID) {
        SipServletRequest sipServletRequest = sipFactoryImpl.createRequestWithCallID(appSession, method, from, to, callID);
        return sipServletRequest;
    }

    @Override
    public SipServletRequest createRequestWithCallID(SipApplicationSession appSession, String method, String from, String to, String callID) throws ServletParseException {
        SipServletRequest sipServletRequest = sipFactoryImpl.createRequestWithCallID(appSession, method, from, to, callID);
        return sipServletRequest;
    }

    @Override
    public SipServletRequest createRequestWithCallID(SipApplicationSession appSession, String method, URI from, URI to, String callID) {
            SipServletRequest sipServletRequest = sipFactoryImpl.createRequestWithCallID(appSession, method, from, to, callID);
            return sipServletRequest;
    }

}
