package org.mollyware
package settings

import sbt._
import Keys._

// SCALA
import scalariform.formatter.preferences._
import com.typesafe.sbt.SbtScalariform
import com.typesafe.sbt.SbtScalariform.ScalariformKeys
import spray.revolver.RevolverPlugin._
import com.orrsella.sbtsublime.SublimePlugin
import sbtassembly.Plugin._
import AssemblyKeys._

// SBT-WEB
import com.typesafe.sbt.web.Import._
import com.typesafe.sbt.web.Import.WebKeys._
import net.ground5hark.sbt.concat.Import._
import com.typesafe.sbt.jshint.Import._
import com.slidingautonomy.sbt.filter.Import._
import com.typesafe.sbt.less.Import._
import com.typesafe.sbt.mocha.Import._

object Settings {
  import PluginSettings._

  lazy val defaultSettings = 
    formatSettings ++
    scalac ++
    javac ++ 
    jarSettings ++
    sbtSublimeSettings ++
    graphSettings ++
    commonSettings ++
    repos ++
    prompt

  lazy val prompt = Seq(
      shellPrompt <<= (name) { (name) => _ => name + "> " }
    )

  lazy val microServiceSettings = 
    Revolver.settings ++ 
    assemblyMicroService 

  import WebPluginSettings._

  lazy val webUiSettings = 
    concatSettings ++
    filterSettings ++
    jshintSettings ++
    webDistSettings ++
    pipeline
}

object WebPluginSettings {

  val concatSettings = Seq(
    Concat.parentDir := ""
  )

  lazy val webDist = taskKey[File]("Task that packages files in web staging directory to zip.")
  lazy val webDistFile = taskKey[File]("Path to write zip file to. Default is target/dist/<name>-<version>.zip.")
  
  val webDistSettings = Seq(
      webDistFile := webTarget.value / "dist" / s"${name.value}-${version.value}.zip",
      webDist := {
        val dir = (stagingDirectory).value
        val out: File = webDistFile.value
        IO.zip(Path.allSubpaths(dir), out)
        out
      },
      webDist <<= webDist.dependsOn(stage)
    )  

  val filterSettings = Seq(
    // We must filter out all subfolders before filtering parent folders.
    includeFilter in filter := "*.js" || "*.css" || "*.css.map" || "*.less" || "controllers" || "directives" || "filters" || "services" || "js" || "bootstrap-less" || "mixins" || "less" || "vendor",
    // Only allow concatenated files to slip through
    excludeFilter in filter := "app.js" || "vendor.js" || "app.css"
  )

  val jshintSettings = Seq(
    // JS Hint shoud not run on vendor
    excludeFilter in (Assets, JshintKeys.jshint) := new FileFilter{ def accept(f: File) = ".*/vendor/.*".r.pattern.matcher(f.getAbsolutePath).matches }
  )

  val pipeline = Seq(
    pipelineStages := Seq(concat, filter)
  )

}

object PluginSettings {
  val commonSettings = Seq(
    scalaVersion := "2.10.4",
    organization := "mollyware",
    version := "0.0.1-SNAPSHOT"
  )

  lazy val graphSettings = net.virtualvoid.sbt.graph.Plugin.graphSettings

// compile options
  lazy val scalac = Seq( 
    scalacOptions ++= Seq( 
      "-encoding", 
      "UTF-8", 
      "-deprecation",
      "-feature",
      "-language:postfixOps",  
      "-language:implicitConversions",
      "-language:higherKinds",
      "-language:existentials",
      "-language:postfixOps",
      "-Ywarn-adapted-args",
      "-Ywarn-dead-code",
      "-Ywarn-numeric-widen",
      "-Ywarn-inaccessible",
      "-Yinline-warnings",
      "-unchecked"
    ),
    scalacOptions <++= (version) map { v =>
      if (v.endsWith("SNAPSHOT")) Nil else Seq("-optimize")
    },
    incOptions := incOptions.value.withNameHashing(true)
  )

  lazy val javac = Seq( 
    javacOptions ++= Seq( 
      "-Xlint:unchecked", 
      "-Xlint:deprecation",
      "-Xlint",
      "-Djava.awt.headless=true", 
      "-Xmx1024M", 
      "-XX:MaxPermSize=512m"
    )
  )

  lazy val sbtSublimeSettings = Seq( 
      SublimePlugin.sublimeExternalSourceDirectoryName := "ext-lib-src" 
    )

  lazy val jarSettings = Seq( exportJars := true )

  // Produces a jar without dependencies and scala language jar included
  lazy val assemblyMicroService = assemblySettings ++ Seq( 
      assembleArtifact in packageScala := true,
      assembleArtifact in packageDependency := true
      //assemblyOption in packageDependency ~= { _.copy(includeScala = false) }
    )

  lazy val repos = Seq( 
    resolvers ++= 
      Seq(
        "Spray repo" at "http://repo.spray.io",
        "Sonatype Releases Repo" at "http://oss.sonatype.org/content/repositories/releases/"
      )
    )

  lazy val formatSettings = SbtScalariform.scalariformSettings ++ Seq(
      ScalariformKeys.preferences in Compile := formattingPreferences,
      ScalariformKeys.preferences in Test    := formattingPreferences
    )

  lazy val formattingPreferences = FormattingPreferences()
    .setPreference( RewriteArrowSymbols, true )
    .setPreference( AlignParameters, true )
    .setPreference( AlignSingleLineCaseStatements, true )
    .setPreference( SpaceInsideParentheses, true )
    .setPreference( DoubleIndentClassDeclaration, true )
    .setPreference( PreserveDanglingCloseParenthesis, true )
}