/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.examples.socnet;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.Random;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.neo4j.examples.socnet.Person;
import org.neo4j.examples.socnet.PersonRepository;
import org.neo4j.examples.socnet.StatusUpdate;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.index.Index;
import org.neo4j.helpers.collection.IteratorUtil;
import org.neo4j.test.TestGraphDatabaseFactory;

public class SocnetTest {
    private static final Random r = new Random(System.currentTimeMillis());
    private static final int nrOfPersons = 20;
    private GraphDatabaseService graphDb;
    private PersonRepository personRepository;

    @Before
    public void setup() throws Exception {
        this.graphDb = new TestGraphDatabaseFactory().newImpermanentDatabase();
        try (Transaction tx = this.graphDb.beginTx();){
            Index index = this.graphDb.index().forNodes("nodes");
            this.personRepository = new PersonRepository(this.graphDb, index);
            this.createPersons();
            this.setupFriendsBetweenPeople(10);
            tx.success();
        }
    }

    @After
    public void teardown() {
        this.graphDb.shutdown();
    }

    @Test
    public void addStatusAndRetrieveIt() throws Exception {
        Person person;
        try (Transaction tx = this.graphDb.beginTx();){
            person = this.getRandomPerson();
            person.addStatus("Testing!");
            tx.success();
        }
        tx = this.graphDb.beginTx();
        var3_2 = null;
        try {
            StatusUpdate update = (StatusUpdate)person.getStatus().iterator().next();
            Assert.assertThat((Object)update, (Matcher)CoreMatchers.notNullValue());
            Assert.assertThat((Object)update.getStatusText(), (Matcher)CoreMatchers.equalTo((Object)"Testing!"));
            Assert.assertThat((Object)update.getPerson(), (Matcher)CoreMatchers.equalTo((Object)person));
        }
        catch (Throwable throwable) {
            var3_2 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var3_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var3_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    @Test
    public void multipleStatusesComeOutInTheRightOrder() throws Exception {
        ArrayList<String> statuses = new ArrayList<String>();
        statuses.add("Test1");
        statuses.add("Test2");
        statuses.add("Test3");
        try (Transaction tx = this.graphDb.beginTx();){
            Person person = this.getRandomPerson();
            for (String status : statuses) {
                person.addStatus(status);
            }
            int i = statuses.size();
            for (StatusUpdate update : person.getStatus()) {
                Assert.assertThat((Object)update.getStatusText(), (Matcher)CoreMatchers.equalTo(statuses.get(--i)));
            }
        }
    }

    @Test
    public void removingOneFriendIsHandledCleanly() {
        int noOfFriends;
        Person person2;
        Person person1;
        try (Transaction tx = this.graphDb.beginTx();){
            person1 = this.personRepository.getPersonByName("person#1");
            person2 = this.personRepository.getPersonByName("person#2");
            person1.addFriend(person2);
            noOfFriends = person1.getNrOfFriends();
            tx.success();
        }
        tx = this.graphDb.beginTx();
        var5_2 = null;
        try {
            person1.removeFriend(person2);
            tx.success();
        }
        catch (Throwable x2) {
            var5_2 = x2;
            throw x2;
        }
        finally {
            if (tx != null) {
                if (var5_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var5_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
        tx = this.graphDb.beginTx();
        var5_2 = null;
        try {
            int noOfFriendsAfterChange = person1.getNrOfFriends();
            Assert.assertThat((Object)noOfFriends, (Matcher)CoreMatchers.equalTo((Object)(noOfFriendsAfterChange + 1)));
        }
        catch (Throwable throwable) {
            var5_2 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var5_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var5_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    @Test
    public void retrieveStatusUpdatesInDateOrder() throws Exception {
        int numberOfStatuses;
        Person person;
        try (Transaction tx = this.graphDb.beginTx();){
            person = this.getRandomPersonWithFriends();
            numberOfStatuses = 20;
            for (int i = 0; i < numberOfStatuses; ++i) {
                Person friend = this.getRandomFriendOf(person);
                friend.addStatus("Dum-deli-dum...");
            }
            tx.success();
        }
        try (Transaction tx = this.graphDb.beginTx();){
            ArrayList<StatusUpdate> updates = this.fromIterableToArrayList(person.friendStatuses());
            Assert.assertThat((Object)updates.size(), (Matcher)CoreMatchers.equalTo((Object)numberOfStatuses));
            this.assertUpdatesAreSortedByDate(updates);
        }
    }

    @Test
    public void friendsOfFriendsWorks() throws Exception {
        try (Transaction tx = this.graphDb.beginTx();){
            Person person = this.getRandomPerson();
            Person friend = this.getRandomFriendOf(person);
            for (Person friendOfFriend : friend.getFriends()) {
                if (friendOfFriend.equals((Object)person)) continue;
                Assert.assertThat((Object)person.getFriendsOfFriends(), (Matcher)CoreMatchers.hasItems((Object[])new Person[]{friendOfFriend}));
            }
        }
    }

    @Test
    public void shouldReturnTheCorrectPersonFromAnyStatusUpdate() throws Exception {
        try (Transaction tx = this.graphDb.beginTx();){
            Person person = this.getRandomPerson();
            person.addStatus("Foo");
            person.addStatus("Bar");
            person.addStatus("Baz");
            for (StatusUpdate status : person.getStatus()) {
                Assert.assertThat((Object)status.getPerson(), (Matcher)CoreMatchers.equalTo((Object)person));
            }
        }
    }

    @Test
    public void getPathBetweenFriends() throws Exception {
        Person endMan;
        Person middleMan2;
        Person middleMan1;
        Person start;
        this.deleteSocialGraph();
        try (Transaction tx = this.graphDb.beginTx();){
            start = this.personRepository.createPerson("start");
            middleMan1 = this.personRepository.createPerson("middle1");
            middleMan2 = this.personRepository.createPerson("middle2");
            endMan = this.personRepository.createPerson("endMan");
            start.addFriend(middleMan1);
            middleMan1.addFriend(middleMan2);
            middleMan2.addFriend(endMan);
            tx.success();
        }
        tx = this.graphDb.beginTx();
        var6_2 = null;
        try {
            Iterable path = start.getShortestPathTo(endMan, 4);
            this.assertPathIs(path, start, middleMan1, middleMan2, endMan);
        }
        catch (Throwable throwable) {
            var6_2 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var6_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var6_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    @Test
    public void singleFriendRecommendation() throws Exception {
        Person e;
        Person a;
        this.deleteSocialGraph();
        try (Transaction tx = this.graphDb.beginTx();){
            a = this.personRepository.createPerson("a");
            Person b = this.personRepository.createPerson("b");
            Person c = this.personRepository.createPerson("c");
            Person d = this.personRepository.createPerson("d");
            e = this.personRepository.createPerson("e");
            a.addFriend(b);
            a.addFriend(c);
            a.addFriend(d);
            e.addFriend(b);
            e.addFriend(c);
            e.addFriend(d);
            tx.success();
        }
        tx = this.graphDb.beginTx();
        var4_2 = null;
        try {
            Person recommendation = (Person)IteratorUtil.single(a.getFriendRecommendation(1).iterator());
            Assert.assertThat((Object)recommendation, (Matcher)CoreMatchers.equalTo((Object)e));
        }
        catch (Throwable throwable) {
            var4_2 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var4_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var4_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    @Test
    public void weightedFriendRecommendation() throws Exception {
        Person f;
        Person e;
        Person a;
        this.deleteSocialGraph();
        try (Transaction tx = this.graphDb.beginTx();){
            a = this.personRepository.createPerson("a");
            Person b = this.personRepository.createPerson("b");
            Person c = this.personRepository.createPerson("c");
            Person d = this.personRepository.createPerson("d");
            e = this.personRepository.createPerson("e");
            f = this.personRepository.createPerson("f");
            a.addFriend(b);
            a.addFriend(c);
            a.addFriend(d);
            e.addFriend(b);
            f.addFriend(b);
            f.addFriend(c);
            f.addFriend(d);
            tx.success();
        }
        tx = this.graphDb.beginTx();
        var5_2 = null;
        try {
            ArrayList recommendations = this.fromIterableToArrayList(a.getFriendRecommendation(2).iterator());
            Assert.assertThat(recommendations.get(0), (Matcher)CoreMatchers.equalTo((Object)f));
            Assert.assertThat(recommendations.get(1), (Matcher)CoreMatchers.equalTo((Object)e));
        }
        catch (Throwable throwable) {
            var5_2 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var5_2 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        var5_2.addSuppressed(x2);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    private <T> ArrayList<T> fromIterableToArrayList(Iterator<T> iterable) {
        ArrayList collection = new ArrayList();
        IteratorUtil.addToCollection(iterable, collection);
        return collection;
    }

    private void assertPathIs(Iterable<Person> path, Person ... expectedPath) {
        ArrayList pathArray = new ArrayList();
        IteratorUtil.addToCollection(path, pathArray);
        Assert.assertThat((Object)pathArray.size(), (Matcher)CoreMatchers.equalTo((Object)expectedPath.length));
        for (int i = 0; i < expectedPath.length; ++i) {
            Assert.assertThat(pathArray.get(i), (Matcher)CoreMatchers.equalTo((Object)expectedPath[i]));
        }
    }

    private void setupFriendsBetweenPeople(int maxNrOfFriendsEach) {
        for (Person person : this.personRepository.getAllPersons()) {
            int nrOfFriends = r.nextInt(maxNrOfFriendsEach) + 1;
            for (int j = 0; j < nrOfFriends; ++j) {
                person.addFriend(this.getRandomPerson());
            }
        }
    }

    private Person getRandomPerson() {
        return this.personRepository.getPersonByName("person#" + r.nextInt(20));
    }

    private void deleteSocialGraph() {
        try (Transaction tx = this.graphDb.beginTx();){
            for (Person person : this.personRepository.getAllPersons()) {
                this.personRepository.deletePerson(person);
            }
        }
    }

    private Person getRandomFriendOf(Person p) {
        ArrayList friends = new ArrayList();
        IteratorUtil.addToCollection(p.getFriends().iterator(), friends);
        return (Person)friends.get(r.nextInt(friends.size()));
    }

    private Person getRandomPersonWithFriends() {
        Person p;
        while ((p = this.getRandomPerson()).getNrOfFriends() == 0) {
        }
        return p;
    }

    private void createPersons() throws Exception {
        for (int i = 0; i < 20; ++i) {
            this.personRepository.createPerson("person#" + i);
        }
    }

    private void assertUpdatesAreSortedByDate(ArrayList<StatusUpdate> statusUpdates) {
        Date date = new Date(0L);
        for (StatusUpdate update : statusUpdates) {
            Assert.assertTrue((date.getTime() < update.getDate().getTime() ? 1 : 0) != 0);
        }
    }
}

