/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.gds.kcore;

import org.neo4j.gds.WriteNodePropertiesComputationResultConsumer;
import org.neo4j.gds.core.write.ImmutableNodeProperty;
import org.neo4j.gds.executor.AlgorithmSpec;
import org.neo4j.gds.executor.ComputationResult;
import org.neo4j.gds.executor.ComputationResultConsumer;
import org.neo4j.gds.executor.ExecutionContext;
import org.neo4j.gds.executor.GdsCallable;
import org.neo4j.gds.executor.NewConfigFunction;
import org.neo4j.gds.procedures.community.kcore.KCoreDecompositionWriteResult;
import org.neo4j.gds.result.AbstractResultBuilder;

import java.util.List;
import java.util.stream.Stream;

import static org.neo4j.gds.executor.ExecutionMode.WRITE_NODE_PROPERTY;
import static org.neo4j.gds.kcore.KCoreCompanion.nodePropertyValues;
import static org.neo4j.gds.kcore.KCoreDecomposition.KCORE_DESCRIPTION;

@GdsCallable(name = "gds.kcore.write", description = KCORE_DESCRIPTION, executionMode = WRITE_NODE_PROPERTY)
public class KCoreDecompositionWriteSpec implements AlgorithmSpec<KCoreDecomposition, KCoreDecompositionResult, KCoreDecompositionWriteConfig, Stream<KCoreDecompositionWriteResult>, KCoreDecompositionAlgorithmFactory<KCoreDecompositionWriteConfig>> {
    @Override
    public String name() {
        return "KCoreWrite";
    }

    @Override
    public KCoreDecompositionAlgorithmFactory<KCoreDecompositionWriteConfig> algorithmFactory(ExecutionContext executionContext) {
        return new KCoreDecompositionAlgorithmFactory<>();
    }

    @Override
    public NewConfigFunction<KCoreDecompositionWriteConfig> newConfigFunction() {
        return (__, userInput) -> KCoreDecompositionWriteConfig.of(userInput);
    }

    @Override
    public ComputationResultConsumer<KCoreDecomposition, KCoreDecompositionResult, KCoreDecompositionWriteConfig, Stream<KCoreDecompositionWriteResult>> computationResultConsumer() {
        return new WriteNodePropertiesComputationResultConsumer<>(
            this::resultBuilder,
            computationResult -> List.of(ImmutableNodeProperty.of(
                computationResult.config().writeProperty(),
                nodePropertyValues(computationResult.result()))),
            name()
        );
    }

    private AbstractResultBuilder<KCoreDecompositionWriteResult> resultBuilder(
        ComputationResult<KCoreDecomposition, KCoreDecompositionResult, KCoreDecompositionWriteConfig> computationResult,
        ExecutionContext executionContext
    ) {
        var builder = new KCoreDecompositionWriteResult.Builder();
        computationResult.result().ifPresent(result -> builder.withDegeneracy(result.degeneracy()));

        return builder;
    }
}
