/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.backup;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Path;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.neo4j.backup.BackupService;
import org.neo4j.backup.ConsistencyCheck;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.IncorrectUsage;
import org.neo4j.commandline.admin.OutsideWorld;
import org.neo4j.commandline.arguments.Arguments;
import org.neo4j.commandline.arguments.MandatoryNamedArg;
import org.neo4j.commandline.arguments.NamedArgument;
import org.neo4j.commandline.arguments.OptionalBooleanArg;
import org.neo4j.commandline.arguments.OptionalNamedArg;
import org.neo4j.commandline.arguments.common.MandatoryCanonicalPath;
import org.neo4j.commandline.arguments.common.OptionalCanonicalPath;
import org.neo4j.consistency.ConsistencyCheckService;
import org.neo4j.consistency.ConsistencyCheckSettings;
import org.neo4j.consistency.checking.full.CheckConsistencyConfig;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.helpers.Args;
import org.neo4j.helpers.HostnamePort;
import org.neo4j.helpers.TimeUtil;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.helpers.collection.Pair;
import org.neo4j.helpers.progress.ProgressMonitorFactory;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.util.Converters;
import org.neo4j.logging.FormattedLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.server.configuration.ConfigLoader;

public class OnlineBackupCommand
implements AdminCommand {
    private static final Arguments arguments = new Arguments().withArgument((NamedArgument)new MandatoryCanonicalPath("backup-dir", "backup-path", "Directory to place backup in.")).withArgument((NamedArgument)new MandatoryNamedArg("name", "graph.db-backup", "Name of backup. If a backup with this name already exists an incremental backup will be attempted.")).withArgument((NamedArgument)new OptionalNamedArg("from", "address", "localhost:6362", "Host and port of Neo4j.")).withArgument((NamedArgument)new OptionalBooleanArg("fallback-to-full", true, "If an incremental backup fails backup will move the old backup to <name>.err.<N> and fallback to a full backup instead.")).withArgument((NamedArgument)new OptionalNamedArg("timeout", "timeout", "20m", "Timeout in the form <time>[ms|s|m|h], where the default unit is seconds.")).withArgument((NamedArgument)new OptionalBooleanArg("check-consistency", true, "If a consistency check should be made.")).withArgument((NamedArgument)new OptionalCanonicalPath("cc-report-dir", "directory", ".", "Directory where consistency report will be written.")).withArgument((NamedArgument)new OptionalCanonicalPath("additional-config", "config-file-path", "", "Configuration file to supply additional configuration in. This argument is DEPRECATED.")).withArgument((NamedArgument)new OptionalBooleanArg("cc-graph", true, "Perform consistency checks between nodes, relationships, properties, types and tokens.")).withArgument((NamedArgument)new OptionalBooleanArg("cc-indexes", true, "Perform consistency checks on indexes.")).withArgument((NamedArgument)new OptionalBooleanArg("cc-label-scan-store", true, "Perform consistency checks on the label scan store.")).withArgument((NamedArgument)new OptionalBooleanArg("cc-property-owners", false, "Perform additional consistency checks on property ownership. This check is *very* expensive in time and memory."));
    static final int STATUS_CC_ERROR = 2;
    static final int STATUS_CC_INCONSISTENT = 3;
    static final int MAX_OLD_BACKUPS = 1000;
    private final BackupService backupService;
    private final Path homeDir;
    private final Path configDir;
    private ConsistencyCheckService consistencyCheckService;
    private final OutsideWorld outsideWorld;

    public static Arguments arguments() {
        return arguments;
    }

    public OnlineBackupCommand(BackupService backupService, Path homeDir, Path configDir, ConsistencyCheckService consistencyCheckService, OutsideWorld outsideWorld) {
        this.backupService = backupService;
        this.homeDir = homeDir;
        this.configDir = configDir;
        this.consistencyCheckService = consistencyCheckService;
        this.outsideWorld = outsideWorld;
    }

    public void execute(String[] args) throws IncorrectUsage, CommandFailed {
        boolean checkPropertyOwners;
        boolean checkLabelScanStore;
        boolean checkIndexes;
        boolean checkGraph;
        Path reportDir;
        Optional additionalConfig;
        long timeout;
        boolean doConsistencyCheck;
        boolean fallbackToFull;
        String name;
        Path folder;
        HostnamePort address;
        try {
            address = (HostnamePort)Converters.toHostnamePort((HostnamePort)new HostnamePort("localhost", 6362)).apply(arguments.parse(args).get("from"));
            folder = arguments.getMandatoryPath("backup-dir");
            name = arguments.get("name");
            fallbackToFull = arguments.getBoolean("fallback-to-full");
            doConsistencyCheck = arguments.getBoolean("check-consistency");
            timeout = (Long)arguments.get("timeout", TimeUtil.parseTimeMillis);
            additionalConfig = arguments.getOptionalPath("additional-config");
            reportDir = (Path)arguments.getOptionalPath("cc-report-dir").orElseThrow(() -> new IllegalArgumentException("cc-report-dir must be a path"));
        }
        catch (IllegalArgumentException e) {
            throw new IncorrectUsage(e.getMessage());
        }
        if (!this.outsideWorld.fileSystem().isDirectory(folder.toFile())) {
            throw new CommandFailed(String.format("Directory '%s' does not exist.", folder));
        }
        if (!this.outsideWorld.fileSystem().isDirectory(reportDir.toFile())) {
            throw new CommandFailed(String.format("Directory '%s' does not exist.", reportDir));
        }
        File destination = folder.resolve(name).toFile();
        Config config = this.loadConfig(additionalConfig);
        boolean done = false;
        try {
            checkGraph = arguments.has("cc-graph") ? arguments.getBoolean("cc-graph") : ((Boolean)ConsistencyCheckSettings.consistency_check_graph.from((Configuration)config)).booleanValue();
            checkIndexes = arguments.has("cc-indexes") ? arguments.getBoolean("cc-indexes") : ((Boolean)ConsistencyCheckSettings.consistency_check_indexes.from((Configuration)config)).booleanValue();
            checkLabelScanStore = arguments.has("cc-label-scan-store") ? arguments.getBoolean("cc-label-scan-store") : ((Boolean)ConsistencyCheckSettings.consistency_check_label_scan_store.from((Configuration)config)).booleanValue();
            checkPropertyOwners = arguments.has("cc-property-owners") ? arguments.getBoolean("cc-property-owners") : ((Boolean)ConsistencyCheckSettings.consistency_check_property_owners.from((Configuration)config)).booleanValue();
        }
        catch (IllegalArgumentException e) {
            throw new IncorrectUsage(e.getMessage());
        }
        File[] listFiles = this.outsideWorld.fileSystem().listFiles(destination);
        if (listFiles != null && listFiles.length > 0) {
            this.outsideWorld.stdOutLine("Destination is not empty, doing incremental backup...");
            try {
                this.backupService.doIncrementalBackup(address.getHost(), address.getPort(), destination, ConsistencyCheck.NONE, timeout, config);
                done = true;
            }
            catch (Exception e) {
                if (fallbackToFull) {
                    this.outsideWorld.stdErrLine("Incremental backup failed: " + e.getMessage());
                    String renamed = this.renameExistingBackup(folder, name);
                    this.outsideWorld.stdErrLine(String.format("Old backup renamed to '%s'.", renamed));
                }
                throw new CommandFailed("Backup failed: " + e.getMessage(), (Throwable)e);
            }
        }
        if (!done) {
            this.outsideWorld.stdOutLine("Doing full backup...");
            try {
                this.backupService.doFullBackup(address.getHost(), address.getPort(), destination, ConsistencyCheck.NONE, config, timeout, false);
            }
            catch (Exception e) {
                throw new CommandFailed("Backup failed: " + e.getMessage(), (Throwable)e);
            }
        }
        if (doConsistencyCheck) {
            try {
                this.outsideWorld.stdOutLine("Doing consistency check...");
                ConsistencyCheckService.Result ccResult = this.consistencyCheckService.runFullConsistencyCheck(destination, config, ProgressMonitorFactory.textual((OutputStream)this.outsideWorld.errorStream()), (LogProvider)FormattedLogProvider.toOutputStream((OutputStream)this.outsideWorld.outStream()), this.outsideWorld.fileSystem(), false, reportDir.toFile(), new CheckConsistencyConfig(checkGraph, checkIndexes, checkLabelScanStore, checkPropertyOwners));
                if (!ccResult.isSuccessful()) {
                    throw new CommandFailed(String.format("Inconsistencies found. See '%s' for details.", ccResult.reportFile()), 3);
                }
            }
            catch (Throwable e) {
                if (e instanceof CommandFailed) {
                    throw (CommandFailed)e;
                }
                throw new CommandFailed("Failed to do consistency check on backup: " + e.getMessage(), e, 2);
            }
        }
        this.outsideWorld.stdOutLine("Backup complete.");
    }

    private String renameExistingBackup(Path folder, String oldName) throws CommandFailed {
        for (int i = 1; i < 1000; ++i) {
            String newName = oldName + ".err." + i;
            if (this.outsideWorld.fileSystem().fileExists(folder.resolve(newName).toFile())) {
                continue;
            }
            try {
                this.outsideWorld.fileSystem().renameFile(folder.resolve(oldName).toFile(), folder.resolve(newName).toFile(), new CopyOption[0]);
                return newName;
            }
            catch (IOException e) {
                throw new CommandFailed("Failed to move old backup out of the way: " + e.getMessage(), (Throwable)e);
            }
        }
        throw new CommandFailed("Failed to move old backup out of the way: too many old backups.");
    }

    private long parseTimeout(String[] args) {
        return Args.parse((String[])args).getDuration("timeout", TimeUnit.MINUTES.toMillis(20L));
    }

    private Config loadConfig(Optional<Path> additionalConfig) throws CommandFailed {
        return this.withAdditionalConfig(additionalConfig, ConfigLoader.loadConfigWithConnectorsDisabled(Optional.of(this.homeDir.toFile()), Optional.of(this.configDir.resolve("neo4j.conf").toFile()), (Pair[])new Pair[0]));
    }

    private Config withAdditionalConfig(Optional<Path> additionalConfig, Config config) throws CommandFailed {
        if (additionalConfig.isPresent()) {
            try {
                return config.with(MapUtil.load((File)additionalConfig.get().toFile()));
            }
            catch (IOException e) {
                throw new CommandFailed("Could not read additional config from " + additionalConfig.get(), (Throwable)e);
            }
        }
        return config;
    }
}

