/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DynamicTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestFactory;
import org.mockito.Mockito;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.connectors.BoltConnector;
import org.neo4j.configuration.connectors.HttpConnector;
import org.neo4j.configuration.connectors.HttpsConnector;
import org.neo4j.configuration.helpers.SocketAddress;
import org.neo4j.configuration.ssl.SslPolicyConfig;
import org.neo4j.configuration.ssl.SslPolicyScope;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.Log;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class SettingMigratorsTest {
    @Inject
    private TestDirectory testDirectory;

    SettingMigratorsTest() {
    }

    @Test
    void shouldRemoveAllowKeyGenerationFrom35ConfigFormat() throws Throwable {
        this.shouldRemoveAllowKeyGeneration("dbms.ssl.policy.default.allow_key_generation", "true");
        this.shouldRemoveAllowKeyGeneration("dbms.ssl.policy.default.allow_key_generation", "false");
    }

    @Test
    void shouldRemoveAllowKeyGeneration() throws Throwable {
        this.shouldRemoveAllowKeyGeneration("dbms.ssl.policy.default.allow_key_generation", "true");
        this.shouldRemoveAllowKeyGeneration("dbms.ssl.policy.default.allow_key_generation", "false");
    }

    @TestFactory
    Collection<DynamicTest> shouldMigrateSslPolicySettingToActualPolicyGroupName() {
        ArrayList<DynamicTest> tests = new ArrayList<DynamicTest>();
        Map<String, SslPolicyScope> sources = Map.of("bolt.ssl_policy", SslPolicyScope.BOLT, "https.ssl_policy", SslPolicyScope.HTTPS, "dbms.backup.ssl_policy", SslPolicyScope.BACKUP, "causal_clustering.ssl_policy", SslPolicyScope.CLUSTER);
        sources.forEach((setting, source) -> tests.add(DynamicTest.dynamicTest((String)String.format("Test migration of SslPolicy for source %s", source.name()), () -> SettingMigratorsTest.testMigrateSslPolicy(setting, SslPolicyConfig.forScope((SslPolicyScope)source)))));
        return tests;
    }

    @Test
    void shouldWarnWhenUsingLegacySslPolicySettings() {
        Map<String, String> legacySettings = Map.of("dbms.directories.certificates", "/cert/dir/", "unsupported.dbms.security.tls_certificate_file", "public.crt", "unsupported.dbms.security.tls_key_file", "private.key");
        Config config = Config.newBuilder().setRaw(legacySettings).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        for (String setting : legacySettings.keySet()) {
            logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s. Legacy ssl policy is no longer supported.", new Object[]{setting})});
        }
    }

    @Test
    void testDefaultDatabaseMigrator() throws IOException {
        File confFile = this.testDirectory.createFile("test.conf", new String[0]);
        Files.write(confFile.toPath(), List.of("dbms.active_database=foo"), new OpenOption[0]);
        Config config = Config.newBuilder().fromFile(confFile).build();
        Log log = (Log)Mockito.mock(Log.class);
        config.setLogger(log);
        Assertions.assertEquals((Object)"foo", (Object)config.get(GraphDatabaseSettings.default_database));
        ((Log)Mockito.verify((Object)log)).warn("Use of deprecated setting %s. It is replaced by %s", new Object[]{"dbms.active_database", GraphDatabaseSettings.default_database.name()});
        config = Config.newBuilder().fromFile(confFile).set(GraphDatabaseSettings.default_database, (Object)"bar").build();
        log = (Log)Mockito.mock(Log.class);
        config.setLogger(log);
        Assertions.assertEquals((Object)"bar", (Object)config.get(GraphDatabaseSettings.default_database));
        ((Log)Mockito.verify((Object)log)).warn("Use of deprecated setting %s. It is replaced by %s", new Object[]{"dbms.active_database", GraphDatabaseSettings.default_database.name()});
    }

    @Test
    void testConnectorOldFormatMigration() throws IOException {
        File confFile = this.testDirectory.createFile("test.conf", new String[0]);
        Files.write(confFile.toPath(), Arrays.asList("dbms.connector.bolt.enabled=true", "dbms.connector.bolt.type=BOLT", "dbms.connector.http.enabled=true", "dbms.connector.https.enabled=true", "dbms.connector.bolt2.type=bolt", "dbms.connector.bolt2.listen_address=:1234"), new OpenOption[0]);
        Config config = Config.newBuilder().fromFile(confFile).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        Assertions.assertTrue((boolean)((Boolean)config.get(BoltConnector.enabled)));
        Assertions.assertTrue((boolean)((Boolean)config.get(HttpConnector.enabled)));
        Assertions.assertTrue((boolean)((Boolean)config.get(HttpsConnector.enabled)));
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s. Type is no longer required", new Object[]{"dbms.connector.bolt.type"})});
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s. No longer supports multiple connectors. Setting discarded.", new Object[]{"dbms.connector.bolt2.type"})});
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s. No longer supports multiple connectors. Setting discarded.", new Object[]{"dbms.connector.bolt2.listen_address"})});
    }

    @Test
    void testKillQueryVerbose() throws IOException {
        File confFile = this.testDirectory.createFile("test.conf", new String[0]);
        Files.write(confFile.toPath(), List.of("dbms.procedures.kill_query_verbose=false"), new OpenOption[0]);
        Config config = Config.newBuilder().fromFile(confFile).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Setting %s is removed. It's no longer possible to disable verbose kill query logging.", new Object[]{"dbms.procedures.kill_query_verbose"})});
    }

    @Test
    void testDefaultSchemaProvider() throws IOException {
        Map<String, String> migrationMap = Map.of("lucene-1.0", "lucene+native-3.0", "lucene+native-1.0", "lucene+native-3.0", "lucene+native-2.0", "native-btree-1.0", "native-btree-1.0", "native-btree-1.0");
        for (String oldSchemaProvider : migrationMap.keySet()) {
            File confFile = this.testDirectory.createFile("test.conf", new String[0]);
            Files.write(confFile.toPath(), List.of("dbms.index.default_schema_provider=" + oldSchemaProvider), new OpenOption[0]);
            Config config = Config.newBuilder().fromFile(confFile).build();
            AssertableLogProvider logProvider = new AssertableLogProvider();
            config.setLogger(logProvider.getLog(Config.class));
            Object expectedWarning = "Use of deprecated setting dbms.index.default_schema_provider.";
            if (!"native-btree-1.0".equals(oldSchemaProvider)) {
                expectedWarning = (String)expectedWarning + " Value migrated from " + oldSchemaProvider + " to " + migrationMap.get(oldSchemaProvider) + ".";
            }
            logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn((String)expectedWarning)});
        }
    }

    @TestFactory
    Collection<DynamicTest> testConnectorAddressMigration() {
        ArrayList<DynamicTest> tests = new ArrayList<DynamicTest>();
        tests.add(DynamicTest.dynamicTest((String)"Test bolt connector address migration", () -> SettingMigratorsTest.testAddrMigration((Setting<SocketAddress>)BoltConnector.listen_address, (Setting<SocketAddress>)BoltConnector.advertised_address)));
        tests.add(DynamicTest.dynamicTest((String)"Test http connector address migration", () -> SettingMigratorsTest.testAddrMigration((Setting<SocketAddress>)HttpConnector.listen_address, (Setting<SocketAddress>)HttpConnector.advertised_address)));
        tests.add(DynamicTest.dynamicTest((String)"Test https connector address migration", () -> SettingMigratorsTest.testAddrMigration((Setting<SocketAddress>)HttpsConnector.listen_address, (Setting<SocketAddress>)HttpsConnector.advertised_address)));
        return tests;
    }

    @TestFactory
    Collection<DynamicTest> testQueryLogMigration() {
        ArrayList<DynamicTest> tests = new ArrayList<DynamicTest>();
        tests.add(DynamicTest.dynamicTest((String)"Test query log migration, disabled", () -> SettingMigratorsTest.testQueryLogMigration(false, GraphDatabaseSettings.LogQueryLevel.OFF)));
        tests.add(DynamicTest.dynamicTest((String)"Test query log migration, enabled", () -> SettingMigratorsTest.testQueryLogMigration(true, GraphDatabaseSettings.LogQueryLevel.INFO)));
        return tests;
    }

    private static void testQueryLogMigration(Boolean oldValue, GraphDatabaseSettings.LogQueryLevel newValue) {
        Setting setting = GraphDatabaseSettings.log_queries;
        Config config = Config.newBuilder().setRaw(Map.of(setting.name(), oldValue.toString())).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        Assertions.assertEquals((Object)newValue, (Object)config.get(setting));
        String msg = "Use of deprecated setting value %s=%s. It is replaced by %s=%s";
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{setting.name(), oldValue.toString(), setting.name(), newValue.name()})});
    }

    private static void testAddrMigration(Setting<SocketAddress> listenAddr, Setting<SocketAddress> advertisedAddr) {
        Config config1 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), "foo:111")).build();
        Config config2 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), ":222")).build();
        Config config3 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), ":333", advertisedAddr.name(), "bar")).build();
        Config config4 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), "foo:444", advertisedAddr.name(), ":555")).build();
        Config config5 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), "foo", advertisedAddr.name(), "bar")).build();
        Config config6 = Config.newBuilder().setRaw(Map.of(listenAddr.name(), "foo:666", advertisedAddr.name(), "bar:777")).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config1.setLogger(logProvider.getLog(Config.class));
        config2.setLogger(logProvider.getLog(Config.class));
        config3.setLogger(logProvider.getLog(Config.class));
        config4.setLogger(logProvider.getLog(Config.class));
        config5.setLogger(logProvider.getLog(Config.class));
        config6.setLogger(logProvider.getLog(Config.class));
        Assertions.assertEquals((Object)new SocketAddress("localhost", 111), (Object)config1.get(advertisedAddr));
        Assertions.assertEquals((Object)new SocketAddress("localhost", 222), (Object)config2.get(advertisedAddr));
        Assertions.assertEquals((Object)new SocketAddress("bar", 333), (Object)config3.get(advertisedAddr));
        Assertions.assertEquals((Object)new SocketAddress("localhost", 555), (Object)config4.get(advertisedAddr));
        Assertions.assertEquals((Object)new SocketAddress("bar", ((SocketAddress)advertisedAddr.defaultValue()).getPort()), (Object)config5.get(advertisedAddr));
        Assertions.assertEquals((Object)new SocketAddress("bar", 777), (Object)config6.get(advertisedAddr));
        String msg = "Use of deprecated setting port propagation. port %s is migrated from %s to %s.";
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{111, listenAddr.name(), advertisedAddr.name()})});
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{222, listenAddr.name(), advertisedAddr.name()})});
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{333, listenAddr.name(), advertisedAddr.name()})});
        logProvider.assertNone(AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{444, listenAddr.name(), advertisedAddr.name()}));
        logProvider.assertNone(AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{555, listenAddr.name(), advertisedAddr.name()}));
        logProvider.assertNone(AssertableLogProvider.inLog(Config.class).warn(msg, new Object[]{666, listenAddr.name(), advertisedAddr.name()}));
    }

    private static void testMigrateSslPolicy(String oldGroupnameSetting, SslPolicyConfig policyConfig) {
        String oldFormatSetting = "dbms.ssl.policy.foo.trust_all";
        Config config = Config.newBuilder().setRaw(Map.of(oldGroupnameSetting, "foo", oldFormatSetting, "true")).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        Assertions.assertTrue((boolean)((Boolean)config.get(policyConfig.trust_all)));
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s.", new Object[]{oldGroupnameSetting})});
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Use of deprecated setting %s. It is replaced by %s", new Object[]{oldFormatSetting, policyConfig.trust_all.name()})});
    }

    private void shouldRemoveAllowKeyGeneration(String toRemove, String value) {
        Config config = Config.newBuilder().setRaw(Map.of(toRemove, value)).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        Assertions.assertThrows(IllegalArgumentException.class, () -> config.getSetting(toRemove));
        logProvider.assertAtLeastOnce(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(Config.class).warn("Setting %s is removed. A valid key and certificate are required to be present in the key and certificate path configured in this ssl policy.", new Object[]{toRemove})});
    }
}

