/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclEntryType;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.mutable.MutableInt;
import org.assertj.core.api.Assertions;
import org.eclipse.collections.impl.factory.Sets;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Mockito;
import org.neo4j.configuration.BootloaderSettings;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.ConfigUtils;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.GroupSetting;
import org.neo4j.configuration.GroupSettingHelper;
import org.neo4j.configuration.Internal;
import org.neo4j.configuration.SettingChangeListener;
import org.neo4j.configuration.SettingConstraint;
import org.neo4j.configuration.SettingConstraints;
import org.neo4j.configuration.SettingImpl;
import org.neo4j.configuration.SettingMigrator;
import org.neo4j.configuration.SettingValueParser;
import org.neo4j.configuration.SettingValueParsers;
import org.neo4j.configuration.SettingsDeclaration;
import org.neo4j.configuration.connectors.BoltConnector;
import org.neo4j.configuration.connectors.HttpConnector;
import org.neo4j.configuration.connectors.HttpsConnector;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.io.ByteUnit;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.LogAssertions;
import org.neo4j.test.extension.DisabledForRoot;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.util.FeatureToggles;

@TestDirectoryExtension
class ConfigTest {
    private static final Set<PosixFilePermission> permittedFilePermissionsForCommandExpansion = Set.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_READ);
    private static final Set<PosixFilePermission> forbiddenFilePermissionsForCommandExpansion = Set.of(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE);
    @Inject
    private TestDirectory testDirectory;

    ConfigTest() {
    }

    @Test
    void testLoadSettingsToConfig() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"hello", (Object)config.get(TestSettings.stringSetting));
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (Integer)((Integer)config.get(TestSettings.intSetting)));
        org.junit.jupiter.api.Assertions.assertEquals(List.of(Integer.valueOf(1)), (Object)config.get(TestSettings.intListSetting));
        org.junit.jupiter.api.Assertions.assertNull((Object)config.get(TestSettings.boolSetting));
    }

    @Test
    void failToBuildConfigForSettingInWrongNamespace() {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().addSettingsClass(WrongNamespaceSettings.class).build());
        LogAssertions.assertThat((Throwable)e).hasMessageContaining("Setting: 'planet.express.open' name does not reside in any of the supported setting namespaces which are: dbms., db., browser., server., internal.");
    }

    @Test
    void buildConfigForSettingInWrongNamespaceWhenStrictDisabled() {
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> Config.newBuilder().addSettingsClass(WrongNamespaceSettings.class).set(GraphDatabaseSettings.strict_config_validation, (Object)false).build());
    }

    @Test
    void failToBuildConfigForInternalSettingInWrongNamespace() {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().addSettingsClass(InternalWrongNamespaceSettings.class).build());
        LogAssertions.assertThat((Throwable)e).hasMessageContaining("Setting: 'server.setting.not_really.internal' is internal but does not reside in the correct internal settings namespace.");
    }

    @Test
    void failToBuildConfigForPublicSettingInInternalNamespace() {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().addSettingsClass(PublicWrongNamespaceSettings.class).build());
        LogAssertions.assertThat((Throwable)e).hasMessageContaining("Setting: 'setting.not_really.internal' is not internal but using internal settings namespace.");
    }

    @Test
    void failToBuildConfigForPublicSettingInLegacyUnsupportedNamespace() {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().addSettingsClass(LegacyUnsupportedNamespaceSettings.class).build());
        LogAssertions.assertThat((Throwable)e).hasMessageContaining(" Setting: 'setting.unsupported_or_not_really' is not internal but using internal settings namespace.");
    }

    @Test
    void testFetchAbsentSetting() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        Setting absentSetting = SettingImpl.newBuilder((String)"test.absent.bool", (SettingValueParser)SettingValueParsers.BOOL, null).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.get(absentSetting));
    }

    @Test
    void testUpdateValue() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.intSetting, (Object)3).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)3, (Integer)((Integer)config.get(TestSettings.intSetting)));
        config.setDynamic(TestSettings.intSetting, (Object)2, this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (Integer)((Integer)config.get(TestSettings.intSetting)));
        config.setDynamic(TestSettings.intSetting, null, this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (Integer)((Integer)config.get(TestSettings.intSetting)));
    }

    @Test
    void testSetConstrainedValue() {
        Config.Builder builder = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.constrainedIntSetting, (Object)4);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
        builder.set(TestSettings.constrainedIntSetting, (Object)2);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
    }

    @Test
    void testUpdateConstrainedValue() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.setDynamic(TestSettings.constrainedIntSetting, (Object)4, this.getClass().getSimpleName()));
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (Integer)((Integer)config.get(TestSettings.constrainedIntSetting)));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> config.setDynamic(TestSettings.constrainedIntSetting, (Object)2, this.getClass().getSimpleName()));
    }

    @Test
    void testOverrideAbsentSetting() {
        Map<String, String> settings = Map.of("test.absent.bool", "false");
        Config.Builder builder = Config.newBuilder().set(GraphDatabaseSettings.strict_config_validation, (Object)true).addSettingsClass(TestSettings.class).setRaw(settings);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
    }

    @Test
    void testOverrideDefault() {
        Map<Setting<Boolean>, Boolean> overriddenDefaults = Map.of(TestSettings.stringSetting, "foo", TestSettings.intSetting, 11, TestSettings.boolSetting, true);
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).setDefaults(overriddenDefaults).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)config.get(TestSettings.stringSetting));
        org.junit.jupiter.api.Assertions.assertEquals((int)11, (Integer)((Integer)config.get(TestSettings.intSetting)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)config.get(TestSettings.boolSetting));
    }

    @Test
    void testUpdateStatic() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.setDynamic(TestSettings.stringSetting, (Object)"not allowed", this.getClass().getSimpleName()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"hello", (Object)config.get(TestSettings.stringSetting));
        config.set(TestSettings.stringSetting, (Object)"allowed internally");
        org.junit.jupiter.api.Assertions.assertEquals((Object)"allowed internally", (Object)config.get(TestSettings.stringSetting));
    }

    @Test
    void testUpdateImmutable() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.setDynamic(TestSettings.boolSetting, (Object)true, this.getClass().getSimpleName()));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.set(TestSettings.boolSetting, (Object)true));
    }

    @Test
    void testObserver() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        MutableInt observedOld = new MutableInt(0);
        MutableInt observedNew = new MutableInt(0);
        SettingChangeListener listener = (oldValue, newValue) -> {
            observedOld.setValue((Number)oldValue);
            observedNew.setValue((Number)newValue);
        };
        config.addListener(TestSettings.intSetting, listener);
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (Integer)observedOld.getValue());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (Integer)observedNew.getValue());
        config.setDynamic(TestSettings.intSetting, (Object)2, this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (Integer)observedOld.getValue());
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (Integer)observedNew.getValue());
        config.setDynamic(TestSettings.intSetting, (Object)7, this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (Integer)observedOld.getValue());
        org.junit.jupiter.api.Assertions.assertEquals((int)7, (Integer)observedNew.getValue());
        config.removeListener(TestSettings.intSetting, listener);
        config.setDynamic(TestSettings.intSetting, (Object)9, this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (Integer)observedOld.getValue());
        org.junit.jupiter.api.Assertions.assertEquals((int)7, (Integer)observedNew.getValue());
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.addListener(TestSettings.boolSetting, (oV, nV) -> {}));
    }

    @Test
    void testGroup() {
        TestConnectionGroupSetting g1 = TestConnectionGroupSetting.group("1");
        TestConnectionGroupSetting g2 = TestConnectionGroupSetting.group("2");
        Config config = Config.newBuilder().addGroupSettingClass(TestConnectionGroupSetting.class).set(g1.port, (Object)1111).set(g1.hostname, (Object)"0.0.0.0").set(g1.secure, (Object)false).set(g2.port, (Object)2222).set(g2.hostname, (Object)"127.0.0.1").build();
        org.junit.jupiter.api.Assertions.assertEquals((int)1111, (Integer)((Integer)config.get(g1.port)));
        org.junit.jupiter.api.Assertions.assertEquals((int)2222, (Integer)((Integer)config.get(g2.port)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)false, (Object)config.get(g1.secure));
        org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)config.get(g2.secure));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.get(TestConnectionGroupSetting.group((String)"not_specified_id").port));
    }

    @Test
    void testDynamicGroup() {
        TestDynamicGroupSetting g1 = TestDynamicGroupSetting.group("1");
        TestDynamicGroupSetting g2 = TestDynamicGroupSetting.group("2");
        Config config = Config.newBuilder().addGroupSettingClass(TestDynamicGroupSetting.class).set(g1.value, (Object)"value1").set(g2.value, (Object)"value2").build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value1", (Object)config.get(g1.value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config.get(g2.value));
        config.setDynamic(g1.value, (Object)"new1", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new1", (Object)config.get(g1.value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config.get(g2.value));
        config.setDynamic(g2.value, (Object)"new2", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new2", (Object)config.get(g2.value));
        Map groups = config.getGroups(TestDynamicGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups.size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new1", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"1")).value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new2", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"2")).value));
    }

    @Test
    void testDynamicGroupWithConstraint() {
        TestDynamicGroupSetting g1 = TestDynamicGroupSetting.group("1");
        TestDynamicGroupSetting g2 = TestDynamicGroupSetting.group("2");
        Config config = Config.newBuilder().addGroupSettingClass(TestDynamicGroupSetting.class).set(g1.constrainedValue, (Object)"avalue1").set(g2.value, (Object)"value").build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"avalue1", (Object)config.get(g1.constrainedValue));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aDefaultValue", (Object)config.get(g2.constrainedValue));
        config.setDynamic(g1.constrainedValue, (Object)"aNewValue", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aNewValue", (Object)config.get(g1.constrainedValue));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aDefaultValue", (Object)config.get(g2.constrainedValue));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> config.setDynamic(g2.constrainedValue, (Object)"new2", this.getClass().getSimpleName()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aDefaultValue", (Object)config.get(g2.constrainedValue));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aNewValue", (Object)config.get(g1.constrainedValue));
        Map groups = config.getGroups(TestDynamicGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups.size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aNewValue", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"1")).constrainedValue));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"aDefaultValue", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"2")).constrainedValue));
    }

    @Test
    void testDynamicGroupFromConfigs() {
        TestDynamicGroupSetting g1 = TestDynamicGroupSetting.group("1");
        TestDynamicGroupSetting g2 = TestDynamicGroupSetting.group("2");
        Config config1 = Config.newBuilder().addGroupSettingClass(TestDynamicGroupSetting.class).set(TestDynamicGroupSetting.group((String)"1").value, (Object)"value1").set(TestDynamicGroupSetting.group((String)"2").value, (Object)"value2").build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value1", (Object)config1.get(g1.value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config1.get(g2.value));
        Config config2 = Config.newBuilder().addGroupSettingClass(TestDynamicGroupSetting.class).set(TestDynamicGroupSetting.group((String)"1").value, (Object)"value1").set(TestDynamicGroupSetting.group((String)"2").value, (Object)"value2").build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value1", (Object)config2.get(g1.value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config2.get(g2.value));
        config1.setDynamic(TestDynamicGroupSetting.group((String)"1").value, (Object)"new1", this.getClass().getSimpleName());
        config1.setDynamic(TestDynamicGroupSetting.group((String)"2").value, (Object)"new2", this.getClass().getSimpleName());
        Map groups1 = config1.getGroups(TestDynamicGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups1.size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new1", (Object)config1.get(((TestDynamicGroupSetting)groups1.get((Object)"1")).value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new2", (Object)config1.get(((TestDynamicGroupSetting)groups1.get((Object)"2")).value));
        Map groups2 = config2.getGroups(TestDynamicGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups2.size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value1", (Object)config2.get(((TestDynamicGroupSetting)groups2.get((Object)"1")).value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config2.get(((TestDynamicGroupSetting)groups2.get((Object)"2")).value));
    }

    @Test
    void testDynamicGroupObserver() {
        TestDynamicGroupSetting g1 = TestDynamicGroupSetting.group("1");
        TestDynamicGroupSetting g2 = TestDynamicGroupSetting.group("2");
        Config config = Config.newBuilder().addGroupSettingClass(TestDynamicGroupSetting.class).set(g1.value, (Object)"value1").set(g2.value, (Object)"value2").build();
        config.addListener(g1.value, (oldValue, newValue) -> {
            org.junit.jupiter.api.Assertions.assertEquals((Object)oldValue, (Object)"value1");
            org.junit.jupiter.api.Assertions.assertEquals((Object)newValue, (Object)"new1");
        });
        config.addListener(g2.value, (oldValue, newValue) -> {
            org.junit.jupiter.api.Assertions.assertEquals((Object)oldValue, (Object)"value2");
            org.junit.jupiter.api.Assertions.assertEquals((Object)newValue, (Object)"new2");
        });
        config.setDynamic(g1.value, (Object)"new1", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new1", (Object)config.get(g1.value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"value2", (Object)config.get(g2.value));
        config.setDynamic(g2.value, (Object)"new2", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new2", (Object)config.get(g2.value));
        Map groups = config.getGroups(TestDynamicGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)groups.size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new1", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"1")).value));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"new2", (Object)config.get(((TestDynamicGroupSetting)groups.get((Object)"2")).value));
    }

    @Test
    void testGroupInheritance() {
        ChildGroup group = new ChildGroup("1");
        Config config = Config.newBuilder().addGroupSettingClass(ChildGroup.class).set(group.childSetting, (Object)"child").build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"child", (Object)config.get(group.childSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"parent", (Object)config.get(group.parentSetting));
    }

    @Test
    void testDynamicGroupInheritance() {
        ChildDynamicGroup group1 = new ChildDynamicGroup("1");
        ChildDynamicGroup group2 = new ChildDynamicGroup("2");
        Config config = Config.newBuilder().addGroupSettingClass(ChildDynamicGroup.class).set(group1.childSetting, (Object)"child").set(group2.childSetting, (Object)"child").build();
        config.setDynamic(group1.parentSetting, (Object)"newParent", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"newParent", (Object)config.get(group1.parentSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"parent", (Object)config.get(group2.parentSetting));
        config.setDynamic(group1.childSetting, (Object)"newChild", this.getClass().getSimpleName());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"newChild", (Object)config.get(group1.childSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"child", (Object)config.get(group2.childSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"newChild", (Object)config.get(((ChildDynamicGroup)config.getGroups(ChildDynamicGroup.class).get((Object)"1")).childSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"newParent", (Object)config.get(((ChildDynamicGroup)config.getGroups(ChildDynamicGroup.class).get((Object)"1")).parentSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"child", (Object)config.get(((ChildDynamicGroup)config.getGroups(ChildDynamicGroup.class).get((Object)"2")).childSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"parent", (Object)config.get(((ChildDynamicGroup)config.getGroups(ChildDynamicGroup.class).get((Object)"2")).parentSetting));
    }

    @Test
    void testMalformedGroupSetting() {
        Map<String, String> settings = Map.of("dbms.test.connection.http..foo.bar", "1111");
        Config.Builder builder = Config.newBuilder().set(GraphDatabaseSettings.strict_config_validation, (Object)true).addGroupSettingClass(TestConnectionGroupSetting.class).setRaw(settings);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
    }

    @Test
    void testGetGroups() {
        Config config = Config.newBuilder().addGroupSettingClass(TestConnectionGroupSetting.class).set(TestConnectionGroupSetting.group((String)"default").port, (Object)7474).set(TestConnectionGroupSetting.group((String)"1").port, (Object)1111).set(TestConnectionGroupSetting.group((String)"1").hostname, (Object)"0.0.0.0").set(TestConnectionGroupSetting.group((String)"1").secure, (Object)false).set(TestConnectionGroupSetting.group((String)"2").port, (Object)2222).set(TestConnectionGroupSetting.group((String)"2").hostname, (Object)"127.0.0.1").build();
        Map groups = config.getGroups(TestConnectionGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals(Set.of("default", "1", "2"), groups.keySet());
        org.junit.jupiter.api.Assertions.assertEquals((int)7474, (Integer)((Integer)config.get(((TestConnectionGroupSetting)groups.get((Object)"default")).port)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)config.get(((TestConnectionGroupSetting)groups.get((Object)"2")).secure));
    }

    @Test
    void testFromConfig() {
        Config fromConfig = Config.newBuilder().addSettingsClass(TestSettings.class).setDefault(TestSettings.boolSetting, (Object)false).set(TestSettings.intSetting, (Object)3).build();
        Config config1 = Config.newBuilder().fromConfig(fromConfig).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)3, (Integer)((Integer)config1.get(TestSettings.intSetting)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"hello", (Object)config1.get(TestSettings.stringSetting));
        Config config2 = Config.newBuilder().fromConfig(fromConfig).set(TestSettings.intSetting, (Object)5).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)5, (Integer)((Integer)config2.get(TestSettings.intSetting)));
        Config config3 = Config.newBuilder().addSettingsClass(TestSettings.class).fromConfig(fromConfig).set(TestSettings.intSetting, (Object)7).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)7, (Integer)((Integer)config3.get(TestSettings.intSetting)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)false, (Object)config3.get(TestSettings.boolSetting));
    }

    @Test
    void shouldThrowIfMultipleFromConfig() {
        Config fromConfig = Config.newBuilder().addSettingsClass(TestSettings.class).setDefault(TestSettings.boolSetting, (Object)false).set(TestSettings.intSetting, (Object)3).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().fromConfig(fromConfig).fromConfig(fromConfig).build());
    }

    @Test
    void testGroupFromConfig() {
        Config fromConfig = Config.newBuilder().addGroupSettingClass(TestConnectionGroupSetting.class).set(TestConnectionGroupSetting.group((String)"default").port, (Object)7474).set(TestConnectionGroupSetting.group((String)"1").port, (Object)1111).set(TestConnectionGroupSetting.group((String)"1").hostname, (Object)"0.0.0.0").set(TestConnectionGroupSetting.group((String)"1").secure, (Object)false).build();
        Config config1 = Config.newBuilder().fromConfig(fromConfig).build();
        Map groups1 = config1.getGroups(TestConnectionGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals(Set.of("default", "1"), groups1.keySet());
        org.junit.jupiter.api.Assertions.assertEquals((int)7474, (Integer)((Integer)config1.get(((TestConnectionGroupSetting)groups1.get((Object)"default")).port)));
        Config config2 = Config.newBuilder().fromConfig(fromConfig).addGroupSettingClass(TestConnectionGroupSetting.class).set(TestConnectionGroupSetting.group((String)"1").port, (Object)3333).set(TestConnectionGroupSetting.group((String)"2").port, (Object)2222).set(TestConnectionGroupSetting.group((String)"2").hostname, (Object)"127.0.0.1").build();
        Map groups2 = config2.getGroups(TestConnectionGroupSetting.class);
        org.junit.jupiter.api.Assertions.assertEquals(Set.of("default", "1", "2"), groups2.keySet());
        org.junit.jupiter.api.Assertions.assertEquals((int)7474, (Integer)((Integer)config2.get(((TestConnectionGroupSetting)groups2.get((Object)"default")).port)));
        org.junit.jupiter.api.Assertions.assertEquals((int)3333, (Integer)((Integer)config2.get(((TestConnectionGroupSetting)groups2.get((Object)"1")).port)));
        org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)config2.get(((TestConnectionGroupSetting)groups2.get((Object)"default")).secure));
        org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)config2.get(((TestConnectionGroupSetting)groups2.get((Object)"2")).secure));
    }

    @Test
    void testResolveDefaultSettingDependency() {
        Config.Builder builder = Config.newBuilder().addSettingsClass(DependencySettings.class);
        Config config = builder.build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)config.get(DependencySettings.baseString), (Object)config.get(DependencySettings.dependingString));
        String value = "default overrides dependency";
        builder.setDefault(DependencySettings.dependingString, (Object)value);
        Config config2 = builder.build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)value, (Object)config2.get(DependencySettings.dependingString));
        value = "value overrides dependency";
        builder.set(DependencySettings.dependingString, (Object)value);
        config2 = builder.build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)value, (Object)config2.get(DependencySettings.dependingString));
    }

    @Test
    void testResolvePathSettingDependency() {
        Config config = Config.newBuilder().addSettingsClass(DependencySettings.class).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.basePath));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/mid/", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.midPath));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/mid/end/file", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.endPath));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/another/path/file", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.absolute));
        config.set(DependencySettings.endPath, (Object)Path.of("/path/another_file", new String[0]));
        config.set(DependencySettings.absolute, (Object)Path.of("path/another_file", new String[0]));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/path/another_file", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.endPath));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/mid/path/another_file", new String[0]).toAbsolutePath(), (Object)config.get(DependencySettings.absolute));
    }

    @Test
    void testResolveBrokenSettingDependency() {
        Config.Builder builder = Config.newBuilder().addSettingsClass(BrokenDependencySettings.class);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
    }

    @Test
    void testSingleSettingGroup() {
        Map<String, String> fromSettings = Map.of("db.test.single_setting.default", "default value", "db.test.single_setting.foo", "foo", "db.test.single_setting.bar", "bar");
        Config config = Config.newBuilder().addGroupSettingClass(SingleSettingGroup.class).setRaw(fromSettings).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)3, (int)config.getGroups(SingleSettingGroup.class).size());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"default value", (Object)config.get(SingleSettingGroup.group((String)"default").singleSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)config.get(SingleSettingGroup.group((String)"foo").singleSetting));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"bar", (Object)config.get(SingleSettingGroup.group((String)"bar").singleSetting));
    }

    @Test
    void shouldLogIfConfigFileCouldNotBeFound() {
        InternalLog log = (InternalLog)Mockito.mock(InternalLog.class);
        Path confFile = this.testDirectory.file("test.conf");
        Config config = Config.emptyBuilder().fromFileNoThrow(confFile).build();
        config.setLogger(log);
        ((InternalLog)Mockito.verify((Object)log)).warn("Config file [%s] does not exist.", new Object[]{confFile});
    }

    @Test
    @DisabledForRoot
    void shouldLogIfConfigFileCouldNotBeRead() throws IOException {
        AssertableLogProvider logProvider = new AssertableLogProvider(true);
        InternalLog log = logProvider.getLog(Config.class);
        Path confFile = this.testDirectory.file("test.conf");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)confFile.toFile().createNewFile());
        Assumptions.assumeTrue((boolean)confFile.toFile().setReadable(false));
        Config config = Config.emptyBuilder().fromFileNoThrow(confFile).build();
        config.setLogger(log);
        LogAssertions.assertThat((AssertableLogProvider)logProvider).containsMessages(new String[]{"Unable to load config file [%s]"});
    }

    @Test
    void canReadConfigFile() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        Files.write(confFile, Collections.singletonList(GraphDatabaseSettings.initial_default_database.name() + "=foo"), new OpenOption[0]);
        Config config1 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFile(confFile)).build());
        Config config2 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFileNoThrow(confFile)).build());
        Stream.of(config1, config2).forEach(c -> org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)c.get(GraphDatabaseSettings.initial_default_database)));
    }

    @Test
    void canOverrideDefaultCharset() throws IOException {
        String unicodeString = "\u00e5\u00e4\u00f6\u1234";
        String latin1String = "\u00c3\u00a5\u00c3\u00a4\u00c3\u00b6\u00e1\u0088\u00b4";
        Path confFile = this.testDirectory.file("test.conf");
        Files.write(confFile, Collections.singletonList(GraphDatabaseSettings.procedure_allowlist.name() + "=\u00e5\u00e4\u00f6\u1234"), new OpenOption[0]);
        Config config1 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFile(confFile)).build());
        LogAssertions.assertThat((List)((List)config1.get(GraphDatabaseSettings.procedure_allowlist))).containsExactly((Object[])new String[]{"\u00c3\u00a5\u00c3\u00a4\u00c3\u00b6\u00e1\u0088\u00b4"});
        Config config2 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().setFileCharset(StandardCharsets.UTF_8).fromFile(confFile)).build());
        LogAssertions.assertThat((List)((List)config2.get(GraphDatabaseSettings.procedure_allowlist))).containsExactly((Object[])new String[]{"\u00e5\u00e4\u00f6\u1234"});
    }

    @Test
    void canReadEscapedCharsInPathsUnescapedFromFile() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        Files.writeString(confFile, (CharSequence)(GraphDatabaseSettings.data_directory.name() + "=\\test\folder"), new OpenOption[0]);
        Config conf = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFile(confFile)).build());
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/test/folder", new String[0]).toAbsolutePath(), (Object)((Path)conf.get(GraphDatabaseSettings.data_directory)).toAbsolutePath());
    }

    @Test
    void canReadConfigDir() throws IOException {
        Path confDir = this.testDirectory.directory("test.conf");
        Path defaultDatabase = confDir.resolve(GraphDatabaseSettings.initial_default_database.name());
        Files.write(defaultDatabase, "foo".getBytes(), new OpenOption[0]);
        Config config1 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFile(confDir)).build());
        Config config2 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFileNoThrow(confDir)).build());
        Stream.of(config1, config2).forEach(c -> org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)c.get(GraphDatabaseSettings.initial_default_database)));
    }

    @Test
    void ignoreSubdirsInConfigDir() throws IOException {
        Path confDir = this.testDirectory.directory("test.conf");
        Path subDir = Files.createDirectory(confDir.resolve("more"), new FileAttribute[0]);
        Path defaultDatabase = subDir.resolve(GraphDatabaseSettings.initial_default_database.name());
        Files.write(defaultDatabase, "foo".getBytes(), new OpenOption[0]);
        Config config1 = Config.newBuilder().fromFile(confDir).build();
        Config config2 = Config.newBuilder().fromFileNoThrow(confDir).build();
        Stream.of(config1, config2).forEach(c -> {
            AssertableLogProvider logProvider = new AssertableLogProvider();
            c.setLogger(logProvider.getLog(Config.class));
            LogAssertions.assertThat((AssertableLogProvider)logProvider).forLevel(AssertableLogProvider.Level.WARN).containsMessages(new String[]{"Ignoring subdirectory in config directory [" + subDir + "]."});
            LogAssertions.assertThat((AssertableLogProvider)logProvider).forLevel(AssertableLogProvider.Level.ERROR).doesNotHaveAnyLogs();
            LogAssertions.assertThat((String)((String)c.get(GraphDatabaseSettings.initial_default_database))).isNotEqualTo((Object)"foo");
        });
    }

    @Test
    void canReadK8sStyleConfigDir() throws IOException {
        Path confDir = this.createK8sStyleConfigDir(Set.of());
        Config config = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFile(confDir)).build());
        Config config2 = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)Config.newBuilder().fromFileNoThrow(confDir)).build());
        Stream.of(config, config2).forEach(c -> {
            org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)c.get(GraphDatabaseSettings.initial_default_database));
            org.junit.jupiter.api.Assertions.assertEquals((Object)true, (Object)c.get(GraphDatabaseSettings.auth_enabled));
            org.junit.jupiter.api.Assertions.assertEquals((int)4, (Integer)((Integer)c.get(GraphDatabaseSettings.auth_max_failed_attempts)));
        });
    }

    private Path createK8sStyleConfigDir(Set<PosixFilePermission> posixFilePermissions) throws IOException {
        Path targetDir = this.testDirectory.directory("links");
        Path dotFile = Files.createFile(targetDir.resolve("..data"), new FileAttribute[0]);
        Path dotDir = Files.createDirectory(targetDir.resolve("..metadata"), new FileAttribute[0]);
        Path defaultDatabase = targetDir.resolve(GraphDatabaseSettings.initial_default_database.name());
        Files.createFile(defaultDatabase, new FileAttribute[0]);
        Files.write(defaultDatabase, "foo".getBytes(), new OpenOption[0]);
        Path authEnabled = targetDir.resolve(GraphDatabaseSettings.auth_enabled.name());
        Files.createFile(authEnabled, new FileAttribute[0]);
        Files.write(authEnabled, "true".getBytes(), new OpenOption[0]);
        Path confDir = this.testDirectory.directory("neo4j.conf");
        Files.createSymbolicLink(confDir.resolve(dotFile.getFileName()), dotFile, new FileAttribute[0]);
        Files.createSymbolicLink(confDir.resolve(dotDir.getFileName()), dotDir, new FileAttribute[0]);
        Files.createSymbolicLink(confDir.resolve(defaultDatabase.getFileName()), defaultDatabase, new FileAttribute[0]);
        Files.createLink(confDir.resolve(authEnabled.getFileName()), authEnabled);
        Files.createFile(confDir.resolve(".DS_STORE"), new FileAttribute[0]);
        Files.createDirectory(confDir.resolve("..version"), new FileAttribute[0]);
        Path authMaxFailedAttempts = confDir.resolve(GraphDatabaseSettings.auth_max_failed_attempts.name());
        Files.createFile(authMaxFailedAttempts, new FileAttribute[0]);
        Files.write(authMaxFailedAttempts, "4".getBytes(), new OpenOption[0]);
        if (!SystemUtils.IS_OS_WINDOWS && !posixFilePermissions.isEmpty()) {
            ConfigTest.setPosixFilePermissions(defaultDatabase, posixFilePermissions);
            ConfigTest.setPosixFilePermissions(authEnabled, posixFilePermissions);
            ConfigTest.setPosixFilePermissions(authMaxFailedAttempts, posixFilePermissions);
        }
        return confDir;
    }

    private static Config buildWithoutErrorsOrWarnings(Supplier<Config> buildConfig) {
        AssertableLogProvider lp = new AssertableLogProvider();
        Config config = buildConfig.get();
        config.setLogger(lp.getLog(Config.class));
        LogAssertions.assertThat((AssertableLogProvider)lp).forLevel(AssertableLogProvider.Level.WARN).doesNotHaveAnyLogs();
        LogAssertions.assertThat((AssertableLogProvider)lp).forLevel(AssertableLogProvider.Level.ERROR).doesNotHaveAnyLogs();
        return config;
    }

    @Test
    void mustThrowIfConfigFileCouldNotBeFound() {
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> {
            Path confFile = this.testDirectory.file("test.conf");
            Config.emptyBuilder().fromFile(confFile).build();
        });
    }

    @Test
    @DisabledForRoot
    void mustThrowIfConfigFileCouldNotBeRead() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)confFile.toFile().createNewFile());
        Assumptions.assumeTrue((boolean)confFile.toFile().setReadable(false));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> Config.emptyBuilder().fromFile(confFile).build());
    }

    @Test
    void mustWarnIfFileContainsDuplicateSettings() throws Exception {
        InternalLog log = (InternalLog)Mockito.mock(InternalLog.class);
        Path confFile = this.testDirectory.createFile("test.conf");
        Files.write(confFile, Arrays.asList(BootloaderSettings.initial_heap_size.name() + "=5g", BootloaderSettings.initial_heap_size.name() + "=4g", BootloaderSettings.initial_heap_size.name() + "=3g", BootloaderSettings.max_heap_size.name() + "=10g", BootloaderSettings.max_heap_size.name() + "=11g"), new OpenOption[0]);
        Config.Builder builder = Config.newBuilder().set(GraphDatabaseSettings.strict_config_validation, (Object)false).fromFile(confFile).setDefault(BootloaderSettings.initial_heap_size, (Object)ByteUnit.gibiBytes((long)1L)).setDefault(BootloaderSettings.initial_heap_size, (Object)ByteUnit.gibiBytes((long)2L));
        Config config = builder.build();
        config.setLogger(log);
        ((InternalLog)Mockito.verify((Object)log)).warn("The '%s' setting is overridden. Setting value changed from '%s' to '%s'.", new Object[]{BootloaderSettings.initial_heap_size.name(), "5g", "4g"});
        ((InternalLog)Mockito.verify((Object)log)).warn("The '%s' setting is overridden. Setting value changed from '%s' to '%s'.", new Object[]{BootloaderSettings.initial_heap_size.name(), "4g", "3g"});
        ((InternalLog)Mockito.verify((Object)log)).warn("The '%s' setting is overridden. Setting value changed from '%s' to '%s'.", new Object[]{BootloaderSettings.max_heap_size.name(), "10g", "11g"});
        builder.set(GraphDatabaseSettings.strict_config_validation, (Object)true);
        Assertions.assertThatThrownBy(() -> ((Config.Builder)builder).build()).hasMessageContaining("declared multiple times");
    }

    @Test
    void testDisableAllConnectors() {
        Config config = Config.newBuilder().set(BoltConnector.enabled, (Object)true).set(HttpConnector.enabled, (Object)true).set(HttpsConnector.enabled, (Object)true).build();
        ConfigUtils.disableAllConnectors((Config)config);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)config.get(BoltConnector.enabled)));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)config.get(HttpConnector.enabled)));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)config.get(HttpsConnector.enabled)));
    }

    @Test
    void testAmendIfNotSet() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).build();
        config.setIfNotSet(TestSettings.intSetting, (Object)77);
        org.junit.jupiter.api.Assertions.assertEquals((int)77, (Integer)((Integer)config.get(TestSettings.intSetting)));
        Config configWithSetting = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.intSetting, (Object)66).build();
        configWithSetting.setIfNotSet(TestSettings.intSetting, (Object)77);
        org.junit.jupiter.api.Assertions.assertEquals((int)66, (Integer)((Integer)configWithSetting.get(TestSettings.intSetting)));
    }

    @Test
    void testIsExplicitlySet() {
        Config config = Config.emptyBuilder().addSettingsClass(TestSettings.class).build();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)config.isExplicitlySet(TestSettings.intSetting));
        config.set(TestSettings.intSetting, (Object)77);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)config.isExplicitlySet(TestSettings.intSetting));
        Config configWithSetting = Config.emptyBuilder().addSettingsClass(TestSettings.class).set(TestSettings.intSetting, (Object)66).build();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)configWithSetting.isExplicitlySet(TestSettings.intSetting));
        configWithSetting.set(TestSettings.intSetting, null);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)configWithSetting.isExplicitlySet(TestSettings.intSetting));
    }

    @Test
    void testStrictValidationForGarbage() throws IOException {
        Path confFile = this.testDirectory.createFile("test.conf");
        Files.write(confFile, Collections.singletonList("some_unrecognized_garbage=true"), new OpenOption[0]);
        Config.Builder builder = Config.newBuilder().fromFile(confFile);
        builder.set(GraphDatabaseSettings.strict_config_validation, (Object)true);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
        builder.set(GraphDatabaseSettings.strict_config_validation, (Object)false);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
    }

    @Test
    void testStrictValidationForDuplicates() throws IOException {
        Path confFile = this.testDirectory.createFile("test.conf");
        Files.write(confFile, List.of(GraphDatabaseSettings.initial_default_database.name() + "=foo", GraphDatabaseSettings.initial_default_database.name() + "=bar"), new OpenOption[0]);
        Config.Builder builder = Config.newBuilder().fromFile(confFile);
        builder.set(GraphDatabaseSettings.strict_config_validation, (Object)true);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
        builder.set(GraphDatabaseSettings.strict_config_validation, (Object)false);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
    }

    @Test
    void testIncorrectType() {
        Map<Setting<Integer>, String> cfgMap = Map.of(TestSettings.intSetting, "not an int");
        Config.Builder builder = Config.newBuilder().addSettingsClass(TestSettings.class).set(cfgMap);
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"Error evaluating value for setting 'db.test.setting.integer'. Setting 'db.test.setting.integer' can not have value 'not an int'. Should be of type 'Integer', but is 'String'", (Object)exception.getMessage());
    }

    @Test
    void testDoesNotLogChangedJvmArgs() throws IOException {
        Path confFile = this.testDirectory.createFile("test.conf");
        Files.write(confFile, List.of("server.jvm.additional=-XX:+UseG1GC", "server.jvm.additional=-XX:+AlwaysPreTouch", "server.jvm.additional=-XX:+UnlockExperimentalVMOptions", "server.jvm.additional=-XX:+TrustFinalNonStaticFields"), new OpenOption[0]);
        Config config = Config.newBuilder().fromFile(confFile).build();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        config.setLogger(logProvider.getLog(Config.class));
        LogAssertions.assertThat((AssertableLogProvider)logProvider).doesNotHaveAnyLogs();
    }

    @Test
    void shouldCorrectlyValidateDependenciesInConstraints() {
        Config.Builder builder = Config.emptyBuilder().addSettingsClass(ConstraintDependency.class);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
        builder.set(ConstraintDependency.setting1, (Object)5);
        builder.set(ConstraintDependency.setting2, (Object)3);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
        builder.set(ConstraintDependency.setting2, (Object)4);
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"maximum allowed value is 3"});
        builder.set(ConstraintDependency.setting1, (Object)2);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> ((Config.Builder)builder).build());
    }

    @Test
    void shouldFindCircularDependenciesInConstraints() {
        Config.Builder builder = Config.emptyBuilder().addSettingsClass(CircularConstraints.class);
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"circular dependency"});
    }

    @Test
    void shouldNotAllowDependenciesOnDynamicSettings() {
        Config.Builder builder = Config.emptyBuilder().addSettingsClass(DynamicConstraintDependency.class);
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"Can not depend on dynamic setting"});
    }

    @Test
    void shouldNotEvaluateCommandsByDefault() {
        ConfigTest.assumeUnixOrWindows();
        Config.Builder builder = Config.newBuilder().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.intSetting.name(), "$(foo bar)"));
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"is a command, but config is not explicitly told to expand it"});
    }

    @Test
    void shouldReportCommandWithSyntaxError() {
        ConfigTest.assumeUnixOrWindows();
        Config.Builder builder = Config.newBuilder().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.intSetting.name(), "$(foo bar"));
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"Error evaluating value for setting 'db.test.setting.integer'"});
    }

    @Test
    void shouldReportUsefulErrorOnInvalidCommand() {
        ConfigTest.assumeUnixOrWindows();
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.intSetting.name(), "$(foo bar)"));
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"Cannot run program \"foo\""});
    }

    @Test
    void shouldCorrectlyEvaluateCommandAndLogIt() {
        ConfigTest.assumeUnixOrWindows();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        String command = SystemUtils.IS_OS_WINDOWS ? "cmd.exe /c set /a" : "expr";
        Config config = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.intSetting.name(), String.format("$(%s 10 - 2)", command))).build();
        config.setLogger(logProvider.getLog(Config.class));
        org.junit.jupiter.api.Assertions.assertEquals((int)8, (Integer)((Integer)config.get(TestSettings.intSetting)));
        LogAssertions.assertThat((AssertableLogProvider)logProvider).containsMessages(new String[]{"Command expansion is explicitly enabled for configuration", "Executing external script to retrieve value of setting " + TestSettings.intSetting.name()});
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void testThatFileAttributePermissionsDoNotWork() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        Set<PosixFilePermission> permissions = PosixFilePermissions.fromString("rw---x-w-");
        Files.createFile(confFile, PosixFilePermissions.asFileAttribute(permissions));
        org.junit.jupiter.api.Assertions.assertThrows(AssertionError.class, () -> LogAssertions.assertThat(Files.getPosixFilePermissions(confFile, new LinkOption[0])).containsExactlyInAnyOrderElementsOf((Iterable)permissions));
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void testThatFilesPosixFilePermissionsDoWork() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        Set<PosixFilePermission> permissions = PosixFilePermissions.fromString("rw---x-w-");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.setPosixFilePermissions(confFile, permissions);
        LogAssertions.assertThat(Files.getPosixFilePermissions(confFile, new LinkOption[0])).containsExactlyInAnyOrderElementsOf(permissions);
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void shouldCorrectlyEvaluateCommandFromFile() throws IOException {
        ConfigTest.assumeUnixOrWindows();
        Path confFile = this.testDirectory.file("test.conf");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.write(confFile, List.of(String.format("%s=$(expr 3 + 3)", TestSettings.intSetting.name())), new OpenOption[0]);
        ConfigTest.setPosixFilePermissions(confFile, permittedFilePermissionsForCommandExpansion);
        Config config = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confFile).build();
        org.junit.jupiter.api.Assertions.assertEquals((int)6, (Integer)((Integer)config.get(TestSettings.intSetting)));
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void shouldHandleQuotesCorrectlyInCommandExpansion() throws IOException {
        Path confFile = this.testDirectory.file("test.conf");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.write(confFile, List.of(String.format("%s=$(bash -c \"echo '1'\")", TestSettings.stringSetting.name())), new OpenOption[0]);
        ConfigTest.setPosixFilePermissions(confFile, permittedFilePermissionsForCommandExpansion);
        Config config = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confFile).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"1", (Object)config.get(TestSettings.stringSetting));
    }

    @Test
    void shouldNotEvaluateWithIncorrectFilePermission() throws IOException {
        ConfigTest.assumeUnixOrWindows();
        Path confFile = this.testDirectory.file("test.conf");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.write(confFile, List.of(TestSettings.intSetting.name() + "=$(foo bar)"), new OpenOption[0]);
        if (SystemUtils.IS_OS_WINDOWS) {
            AclFileAttributeView attrs = Files.getFileAttributeView(confFile, AclFileAttributeView.class, new LinkOption[0]);
            attrs.setAcl(List.of(AclEntry.newBuilder().setType(AclEntryType.ALLOW).setPrincipal(attrs.getOwner()).setPermissions(AclEntryPermission.READ_DATA, AclEntryPermission.WRITE_DATA, AclEntryPermission.READ_ATTRIBUTES, AclEntryPermission.WRITE_ATTRIBUTES, AclEntryPermission.READ_NAMED_ATTRS, AclEntryPermission.WRITE_NAMED_ATTRS, AclEntryPermission.APPEND_DATA, AclEntryPermission.READ_ACL, AclEntryPermission.SYNCHRONIZE, AclEntryPermission.EXECUTE).build()));
        } else {
            ConfigTest.setPosixFilePermissions(confFile, PosixFilePermissions.fromString("rw-----w-"));
        }
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confFile);
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        String expectedErrorMessage = SystemUtils.IS_OS_WINDOWS ? "does not have the correct ACL for owner" : "does not have the correct file permissions";
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{expectedErrorMessage});
    }

    private static void setPosixFilePermissions(Path confFile, Set<PosixFilePermission> filePermissions) throws IOException {
        Files.setPosixFilePermissions(confFile, filePermissions);
        LogAssertions.assertThat(Files.getPosixFilePermissions(confFile, new LinkOption[0])).containsExactlyInAnyOrderElementsOf(filePermissions);
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void shouldNotEvaluateK8sConfDirWithIncorrectFilePermission() throws IOException {
        Path confDir = this.createK8sStyleConfigDir(PosixFilePermissions.fromString("rw-----w-"));
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confDir);
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        String expectedErrorMessage = "does not have the correct file permissions";
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{expectedErrorMessage});
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void shouldEvaluateK8sConfDirWithCorrectFilePermission() throws IOException {
        Set<PosixFilePermission> permittedPermissions = permittedFilePermissionsForCommandExpansion;
        Path confDir = this.createK8sStyleConfigDir(permittedPermissions);
        Path testSetting = Files.createFile(confDir.resolve(TestSettings.intSetting.name()), new FileAttribute[0]);
        Files.write(testSetting, "$(expr 3 + 3)".getBytes(), new OpenOption[0]);
        Files.setPosixFilePermissions(testSetting, permittedPermissions);
        Config.Builder configBuilder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confDir);
        Config config = ConfigTest.buildWithoutErrorsOrWarnings(() -> ((Config.Builder)configBuilder).build());
        org.junit.jupiter.api.Assertions.assertEquals((int)6, (Integer)((Integer)config.get(TestSettings.intSetting)));
    }

    @Test
    void shouldTimeoutOnSlowCommands() {
        ConfigTest.assumeUnixOrWindows();
        String command = SystemUtils.IS_OS_WINDOWS ? "ping -n 3 localhost" : "sleep 3";
        FeatureToggles.set(Config.class, (String)"CommandEvaluationTimeout", (Object)1);
        Config.Builder builder = Config.newBuilder().set(GraphDatabaseInternalSettings.config_command_evaluation_timeout, (Object)Duration.ofSeconds(1L)).allowCommandExpansion().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.intSetting.name(), String.format("$(%s)", command)));
        String msg = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)msg).contains(new CharSequence[]{"Timed out executing command"});
    }

    @Test
    void shouldNotEvaluateCommandsOnDynamicChanges() {
        ConfigTest.assumeUnixOrWindows();
        String command1 = String.format("$(%s 2 + 2)", SystemUtils.IS_OS_WINDOWS ? "cmd.exe /c set /a" : "expr");
        String command2 = String.format("$(%s 10 - 3)", SystemUtils.IS_OS_WINDOWS ? "cmd.exe /c set /a" : "expr");
        Config config = Config.emptyBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).setRaw(Map.of(TestSettings.dynamicStringSetting.name(), command1)).build();
        LogAssertions.assertThat((String)((String)config.get(TestSettings.dynamicStringSetting))).isEqualTo("4");
        config.setDynamic(TestSettings.dynamicStringSetting, (Object)command2, "test");
        LogAssertions.assertThat((String)((String)config.get(TestSettings.dynamicStringSetting))).isNotEqualTo((Object)"7");
        LogAssertions.assertThat((String)((String)config.get(TestSettings.dynamicStringSetting))).isEqualTo(command2);
    }

    @Test
    void shouldHaveSettingSourceLocation() {
        TestConnectionGroupSetting group = TestConnectionGroupSetting.group("default");
        Config config = Config.emptyBuilder().addGroupSettingClass(TestConnectionGroupSetting.class).addSettingsClass(TestSettings.class).set(group.port, (Object)7474).build();
        LogAssertions.assertThat((String)((SettingImpl)config.getSetting(TestSettings.intSetting.name())).sourceLocation()).isEqualTo("org.neo4j.configuration.ConfigTest.TestSettings.intSetting");
        LogAssertions.assertThat((String)((SettingImpl)config.getSetting(group.port.name())).sourceLocation()).isEqualTo("org.neo4j.configuration.ConfigTest.TestConnectionGroupSetting.port");
    }

    @Test
    void shouldConcatenateMultipleJvmAdditionals() {
        Config config = Config.newBuilder().setRaw(Map.of(BootloaderSettings.additional_jvm.name(), "-Dfoo")).setRaw(Map.of(BootloaderSettings.additional_jvm.name(), "-Dbar")).setRaw(Map.of(BootloaderSettings.additional_jvm.name(), "-Dbaz")).build();
        LogAssertions.assertThat((String)((String)config.get(BootloaderSettings.additional_jvm))).isEqualTo(String.format("%s%n%s%n%s", "-Dfoo", "-Dbar", "-Dbaz"));
    }

    private static Stream<Arguments> forbiddenFilePermissions() {
        return forbiddenFilePermissionsForCommandExpansion.stream().map(p -> Arguments.of((Object[])new Object[]{Set.of(p)}));
    }

    @Test
    void testForbiddenFilePermissionsContainsAllNotPermittedPermissions() {
        Set invalidFilePermissions = ConfigTest.forbiddenFilePermissions().flatMap(a -> ((Set)a.get()[0]).stream()).collect(Collectors.toSet());
        LogAssertions.assertThat((Collection)Sets.union(invalidFilePermissions, permittedFilePermissionsForCommandExpansion)).containsExactlyInAnyOrderElementsOf(Arrays.asList(PosixFilePermission.values()));
        LogAssertions.assertThat(invalidFilePermissions).hasSize(6);
        LogAssertions.assertThat(invalidFilePermissions).contains((Object[])new PosixFilePermission[]{PosixFilePermission.OTHERS_WRITE});
    }

    @DisabledOnOs(value={OS.WINDOWS})
    @ParameterizedTest(name="{0}")
    @MethodSource(value={"forbiddenFilePermissions"})
    void testForbiddenFilePermissionsShouldBeInvalidOnTheirOwn(Set<PosixFilePermission> forbidden) throws IOException {
        Set<PosixFilePermission> readable = Set.of(PosixFilePermission.OWNER_READ);
        Path confFile = this.testDirectory.file("test.conf");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.write(confFile, List.of(String.format("%s=$(expr 3 + 3)", TestSettings.intSetting.name())), new OpenOption[0]);
        ConfigTest.setPosixFilePermissions(confFile, (Set<PosixFilePermission>)Sets.union(readable, forbidden));
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confFile);
        String errorMessage = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)errorMessage).contains(new CharSequence[]{"does not have the correct file permissions to evaluate commands"});
    }

    @DisabledOnOs(value={OS.WINDOWS})
    @ParameterizedTest(name="{0}")
    @MethodSource(value={"forbiddenFilePermissions"})
    void testForbiddenFilePermissionsShouldBeInvalidWhenCombinedWithPermittedPermissions(Set<PosixFilePermission> forbidden) throws IOException {
        Set<PosixFilePermission> permittedPermissions = permittedFilePermissionsForCommandExpansion;
        Path confFile = this.testDirectory.file("test.conf");
        Files.createFile(confFile, new FileAttribute[0]);
        Files.write(confFile, List.of(String.format("%s=$(expr 3 + 3)", TestSettings.intSetting.name())), new OpenOption[0]);
        ConfigTest.setPosixFilePermissions(confFile, (Set<PosixFilePermission>)Sets.union(permittedPermissions, forbidden));
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confFile);
        String errorMessage = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)errorMessage).contains(new CharSequence[]{"does not have the correct file permissions to evaluate commands"});
    }

    @DisabledOnOs(value={OS.WINDOWS})
    @ParameterizedTest(name="{0}")
    @MethodSource(value={"forbiddenFilePermissions"})
    void testForbiddenFilePermissionsShouldBeInvalidOnTheirOwnForK8sConfDir(Set<PosixFilePermission> forbidden) throws IOException {
        Set<PosixFilePermission> readable = Set.of(PosixFilePermission.OWNER_READ);
        Path confDir = this.createK8sStyleConfigDir((Set<PosixFilePermission>)Sets.union(readable, forbidden));
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confDir);
        String errorMessage = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)errorMessage).contains(new CharSequence[]{"does not have the correct file permissions to evaluate commands"});
    }

    @DisabledOnOs(value={OS.WINDOWS})
    @ParameterizedTest(name="{0}")
    @MethodSource(value={"forbiddenFilePermissions"})
    void testForbiddenFilePermissionsShouldBeInvalidWhenCombinedWithPermittedPermissionsForK8sConfDir(Set<PosixFilePermission> forbidden) throws IOException {
        Set<PosixFilePermission> permittedPermissions = permittedFilePermissionsForCommandExpansion;
        Path confDir = this.createK8sStyleConfigDir((Set<PosixFilePermission>)Sets.union(permittedPermissions, forbidden));
        Config.Builder builder = Config.newBuilder().allowCommandExpansion().addSettingsClass(TestSettings.class).fromFile(confDir);
        String errorMessage = ((IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((Config.Builder)builder).build())).getMessage();
        LogAssertions.assertThat((String)errorMessage).contains(new CharSequence[]{"does not have the correct file permissions to evaluate commands"});
    }

    @Test
    void settingsLoadedInPredefinedOrder() {
        Config config = Config.newBuilder().addSettingsClass(BSettings.class).addSettingsClass(ASettings.class).build();
        Object actual = config.settings.get("dbms.test.setting.marker");
        LogAssertions.assertThat((String)actual.toString()).contains(new CharSequence[]{"bValue"});
    }

    @Test
    void migratorsAppliedInPredefinedOrder() {
        Config config = Config.newBuilder().addMigrator((SettingMigrator)new BMigrator()).addMigrator((SettingMigrator)new AMigrator()).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)Duration.ofSeconds(777L), (Object)config.get(GraphDatabaseSettings.transaction_timeout));
    }

    @Test
    void shouldKnowDefaultValue() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.intSetting, (Object)77).build();
        LogAssertions.assertThat((Integer)((Integer)config.getDefault(TestSettings.intSetting))).isEqualTo(TestSettings.intSetting.defaultValue());
        config = Config.newBuilder().addSettingsClass(TestSettings.class).setDefault(TestSettings.intSetting, (Object)50).build();
        LogAssertions.assertThat((Integer)((Integer)config.getDefault(TestSettings.intSetting))).isEqualTo(50);
    }

    @Test
    void shouldRememberStartupValue() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.intSetting, (Object)77).build();
        config.set(TestSettings.intSetting, (Object)50);
        LogAssertions.assertThat((Integer)((Integer)config.getStartupValue(TestSettings.intSetting))).isEqualTo(77);
        LogAssertions.assertThat((Integer)((Integer)config.getDefault(TestSettings.intSetting))).isEqualTo(1);
    }

    @Test
    void shouldKnowValueSource() {
        Config config = Config.newBuilder().addSettingsClass(TestSettings.class).set(TestSettings.boolSetting, (Object)false).build();
        config.setDynamic(TestSettings.intSetting, (Object)50, "Test");
        config.setDynamicByUser(TestSettings.dynamicStringSetting, (Object)"foo", "Test");
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.stringSetting)).isEqualTo((Object)Config.ValueSource.DEFAULT);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.boolSetting)).isEqualTo((Object)Config.ValueSource.INITIAL);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.intSetting)).isEqualTo((Object)Config.ValueSource.SYSTEM);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.dynamicStringSetting)).isEqualTo((Object)Config.ValueSource.USER);
        config = Config.newBuilder().fromConfig(config).build();
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.stringSetting)).isEqualTo((Object)Config.ValueSource.DEFAULT);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.boolSetting)).isEqualTo((Object)Config.ValueSource.INITIAL);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.intSetting)).isEqualTo((Object)Config.ValueSource.SYSTEM);
        LogAssertions.assertThat((Comparable)config.getValueSource(TestSettings.dynamicStringSetting)).isEqualTo((Object)Config.ValueSource.USER);
    }

    private static void assumeUnixOrWindows() {
        Assumptions.assumeTrue((SystemUtils.IS_OS_WINDOWS || SystemUtils.IS_OS_UNIX ? 1 : 0) != 0, (String)"Require system to be either Unix or Windows based.");
    }

    private static final class TestSettings
    implements SettingsDeclaration {
        static final Setting<String> stringSetting = SettingImpl.newBuilder((String)"db.test.setting.string", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").build();
        static final Setting<String> dynamicStringSetting = SettingImpl.newBuilder((String)"db.test.setting.dynamicstring", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").dynamic().build();
        static final Setting<Integer> intSetting = SettingImpl.newBuilder((String)"db.test.setting.integer", (SettingValueParser)SettingValueParsers.INT, (Object)1).dynamic().build();
        static final Setting<Integer> constrainedIntSetting = SettingImpl.newBuilder((String)"db.test.setting.constrained-integer", (SettingValueParser)SettingValueParsers.INT, (Object)1).addConstraint(SettingConstraints.max((Comparable)Integer.valueOf(3))).dynamic().build();
        static final Setting<List<Integer>> intListSetting = SettingImpl.newBuilder((String)"db.test.setting.integerlist", (SettingValueParser)SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.INT), List.of(Integer.valueOf(1))).build();
        static final Setting<Boolean> boolSetting = SettingImpl.newBuilder((String)"db.test.setting.bool", (SettingValueParser)SettingValueParsers.BOOL, null).immutable().build();

        private TestSettings() {
        }
    }

    public static class TestConnectionGroupSetting
    implements GroupSetting {
        private final String id;
        public final Setting<Integer> port;
        public final Setting<String> hostname;
        public final Setting<Boolean> secure;

        public static TestConnectionGroupSetting group(String name) {
            return new TestConnectionGroupSetting(name);
        }

        public String name() {
            return this.id;
        }

        public String getPrefix() {
            return "server.test.connection.http";
        }

        TestConnectionGroupSetting(String id) {
            this.id = id;
            this.port = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)id, (String)"port", (SettingValueParser)SettingValueParsers.INT, (Object)1).build();
            this.hostname = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)id, (String)"hostname", (SettingValueParser)SettingValueParsers.STRING, (Object)"0.0.0.0").build();
            this.secure = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)id, (String)"secure", (SettingValueParser)SettingValueParsers.BOOL, (Object)true).build();
        }
    }

    public static class TestDynamicGroupSetting
    implements GroupSetting {
        private final String id;
        public final Setting<String> value;
        public final Setting<String> constrainedValue;

        public static TestDynamicGroupSetting group(String name) {
            return new TestDynamicGroupSetting(name);
        }

        public String name() {
            return this.id;
        }

        public String getPrefix() {
            return "dbms.test.dynamic";
        }

        TestDynamicGroupSetting(String id) {
            this.id = id;
            this.value = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)id, (String)"value", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").dynamic().build();
            this.constrainedValue = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)id, (String)"constrainedValue", (SettingValueParser)SettingValueParsers.STRING, (Object)"aDefaultValue").addConstraint(SettingConstraints.matches((String)"a.*")).dynamic().build();
        }
    }

    static class ChildGroup
    extends ParentGroup {
        final Setting<String> childSetting;

        private ChildGroup(String name) {
            super(name);
            this.childSetting = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)name, (String)"child", (SettingValueParser)SettingValueParsers.STRING, null).build();
        }

        public String getPrefix() {
            return "db.test.inheritance";
        }
    }

    static class ChildDynamicGroup
    extends ParentDynamicGroup {
        final Setting<String> childSetting;

        private ChildDynamicGroup(String name) {
            super(name);
            this.childSetting = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)name, (String)"child", (SettingValueParser)SettingValueParsers.STRING, null).dynamic().build();
        }

        public String getPrefix() {
            return "server.test.dynamic.inheritance";
        }
    }

    private static final class DependencySettings
    implements SettingsDeclaration {
        static final Setting<Path> basePath = SettingImpl.newBuilder((String)"db.test.base.path", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("/base/", new String[0]).toAbsolutePath()).immutable().build();
        static final Setting<Path> midPath = SettingImpl.newBuilder((String)"db.test.mid.path", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("mid/", new String[0])).setDependency(basePath).immutable().build();
        static final Setting<Path> endPath = SettingImpl.newBuilder((String)"db.test.end.path", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("end/file", new String[0])).setDependency(midPath).build();
        static final Setting<Path> absolute = SettingImpl.newBuilder((String)"db.test.absolute.path", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("/another/path/file", new String[0]).toAbsolutePath()).setDependency(midPath).build();
        static final Setting<String> baseString = SettingImpl.newBuilder((String)"db.test.default.dependency.base", (SettingValueParser)SettingValueParsers.STRING, (Object)"base").immutable().build();
        static final Setting<String> dependingString = SettingImpl.newBuilder((String)"db.test.default.dependency.dep", (SettingValueParser)SettingValueParsers.STRING, null).setDependency(baseString).build();

        private DependencySettings() {
        }
    }

    private static final class BrokenDependencySettings
    implements SettingsDeclaration {
        static final Setting<Path> broken = SettingImpl.newBuilder((String)"test.base.path", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("/base/", new String[0])).setDependency(SettingImpl.newBuilder((String)"test.not.present.dependency", (SettingValueParser)SettingValueParsers.PATH, (Object)Path.of("/broken/", new String[0])).immutable().build()).immutable().build();

        private BrokenDependencySettings() {
        }
    }

    private static final class SingleSettingGroup
    implements GroupSetting {
        final Setting<String> singleSetting;
        private final String name;

        static SingleSettingGroup group(String name) {
            return new SingleSettingGroup(name);
        }

        private SingleSettingGroup(String name) {
            this.name = name;
            this.singleSetting = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)name, (SettingValueParser)SettingValueParsers.STRING, null).build();
        }

        public String name() {
            return this.name;
        }

        public String getPrefix() {
            return "db.test.single_setting";
        }
    }

    private static final class ConstraintDependency
    implements SettingsDeclaration {
        static final Setting<Integer> setting1 = SettingImpl.newBuilder((String)"dbms.test.setting.1", (SettingValueParser)SettingValueParsers.INT, (Object)1).build();
        static final Setting<Integer> setting2 = SettingImpl.newBuilder((String)"dbms.test.setting.2", (SettingValueParser)SettingValueParsers.INT, (Object)1).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(3)), (SettingConstraint)SettingConstraints.unconstrained(), setting1, (SettingConstraint)SettingConstraints.is((Object)5))).build();

        private ConstraintDependency() {
        }
    }

    private static final class CircularConstraints
    implements SettingsDeclaration {
        private static final SettingConstraint<String> circular = new SettingConstraint<String>(){

            public void validate(String value, Configuration config) {
                config.get(setting2);
            }

            public String getDescription() {
                return "circular test dependency";
            }
        };
        static final Setting<String> setting1 = SettingImpl.newBuilder((String)"db.test.setting.1", (SettingValueParser)SettingValueParsers.STRING, (Object)"aloha").addConstraint(circular).build();
        static final Setting<Integer> setting2 = SettingImpl.newBuilder((String)"db.test.setting.2", (SettingValueParser)SettingValueParsers.INT, (Object)1).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(3)), (SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(5)), setting1, (SettingConstraint)SettingConstraints.is((Object)"aloha"))).build();

        private CircularConstraints() {
        }
    }

    private static final class DynamicConstraintDependency
    implements SettingsDeclaration {
        static final Setting<Integer> setting1 = SettingImpl.newBuilder((String)"browser.test.setting.1", (SettingValueParser)SettingValueParsers.INT, (Object)1).dynamic().build();
        static final Setting<Integer> setting2 = SettingImpl.newBuilder((String)"browser.test.setting.2", (SettingValueParser)SettingValueParsers.INT, (Object)1).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(3)), (SettingConstraint)SettingConstraints.unconstrained(), setting1, (SettingConstraint)SettingConstraints.is((Object)5))).build();

        private DynamicConstraintDependency() {
        }
    }

    private static final class BSettings
    implements SettingsDeclaration {
        static final Setting<String> stringSetting = SettingImpl.newBuilder((String)"dbms.test.setting.marker", (SettingValueParser)SettingValueParsers.STRING, (Object)"bValue").build();

        private BSettings() {
        }
    }

    private static final class ASettings
    implements SettingsDeclaration {
        static final Setting<String> stringSetting = SettingImpl.newBuilder((String)"dbms.test.setting.marker", (SettingValueParser)SettingValueParsers.STRING, (Object)"aValue").build();

        private ASettings() {
        }
    }

    private static final class BMigrator
    implements SettingMigrator {
        private BMigrator() {
        }

        public void migrate(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            values.put(GraphDatabaseSettings.transaction_timeout.name(), "777s");
        }
    }

    private static final class AMigrator
    implements SettingMigrator {
        private AMigrator() {
        }

        public void migrate(Map<String, String> values, Map<String, String> defaultValues, InternalLog log) {
            values.put(GraphDatabaseSettings.transaction_timeout.name(), "111s");
        }
    }

    private static final class LegacyUnsupportedNamespaceSettings
    implements SettingsDeclaration {
        static final Setting<String> wrongPublicSetting = SettingImpl.newBuilder((String)"setting.unsupported_or_not_really", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").build();

        private LegacyUnsupportedNamespaceSettings() {
        }
    }

    private static final class PublicWrongNamespaceSettings
    implements SettingsDeclaration {
        static final Setting<String> wrongPublicSetting = SettingImpl.newBuilder((String)"setting.not_really.internal", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").build();

        private PublicWrongNamespaceSettings() {
        }
    }

    private static final class InternalWrongNamespaceSettings
    implements SettingsDeclaration {
        @Internal
        static final Setting<String> wrongInternalSetting = SettingImpl.newBuilder((String)"server.setting.not_really.internal", (SettingValueParser)SettingValueParsers.STRING, (Object)"hello").build();

        private InternalWrongNamespaceSettings() {
        }
    }

    private static final class WrongNamespaceSettings
    implements SettingsDeclaration {
        static final Setting<Boolean> wrongSetting = SettingImpl.newBuilder((String)"planet.express.open", (SettingValueParser)SettingValueParsers.BOOL, (Object)false).build();

        private WrongNamespaceSettings() {
        }
    }

    static abstract class ParentDynamicGroup
    implements GroupSetting {
        final Setting<String> parentSetting;
        private final String name;

        ParentDynamicGroup(String name) {
            this.name = name;
            this.parentSetting = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)name, (String)"parent", (SettingValueParser)SettingValueParsers.STRING, (Object)"parent").dynamic().build();
        }

        public String name() {
            return this.name;
        }
    }

    static abstract class ParentGroup
    implements GroupSetting {
        final Setting<String> parentSetting;
        private final String name;

        ParentGroup(String name) {
            this.name = name;
            this.parentSetting = GroupSettingHelper.getBuilder((String)this.getPrefix(), (String)name, (String)"parent", (SettingValueParser)SettingValueParsers.STRING, (Object)"parent").build();
        }

        public String name() {
            return this.name;
        }
    }
}

