/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.SettingValueParsers;
import org.neo4j.configuration.connectors.BoltConnector;
import org.neo4j.configuration.connectors.HttpConnector;
import org.neo4j.configuration.connectors.HttpsConnector;
import org.neo4j.configuration.helpers.SocketAddress;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.io.ByteUnit;
import org.neo4j.logging.LogAssertions;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;

@TestDirectoryExtension
class GraphDatabaseSettingsTest {
    @Inject
    private TestDirectory directory;

    GraphDatabaseSettingsTest() {
    }

    @Test
    void mustHaveNullDefaultPageCacheMemorySizeInBytes() {
        Long bytes = (Long)Config.defaults().get(GraphDatabaseSettings.pagecache_memory);
        LogAssertions.assertThat((Long)bytes).isNull();
    }

    @Test
    void pageCacheSettingMustAcceptArbitraryUserSpecifiedValue() throws IOException {
        this.assertPageCacheMemorySettingIsParsedAsBytes("245760");
        this.assertPageCacheMemorySettingIsParsedAsBytes("2244g");
        this.assertPageCacheMemorySettingIsParsedAsBytes("8m");
    }

    private void assertPageCacheMemorySettingIsParsedAsBytes(String value) throws IOException {
        Path cfg = this.directory.file("cfg");
        Files.writeString(cfg, (CharSequence)(GraphDatabaseSettings.pagecache_memory.name() + "=" + value), StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        Config config = Config.newBuilder().fromFile(cfg).build();
        LogAssertions.assertThat((Long)((Long)config.get(GraphDatabaseSettings.pagecache_memory))).isEqualTo(ByteUnit.parse((String)value));
    }

    @Test
    void noDuplicateSettingsAreAllowed() throws Exception {
        HashMap<String, String> fields = new HashMap<String, String>();
        for (Field field : GraphDatabaseSettings.class.getDeclaredFields()) {
            if (field.getType() != Setting.class) continue;
            Setting setting = (Setting)field.get(null);
            Assertions.assertFalse((boolean)fields.containsKey(setting.name()), (String)String.format("'%s' in %s has already been defined in %s", setting.name(), field.getName(), fields.get(setting.name())));
            fields.put(setting.name(), field.getName());
        }
    }

    @Test
    void shouldEnableBoltByDefault() {
        Config config = Config.newBuilder().setDefaults(GraphDatabaseSettings.SERVER_DEFAULTS).build();
        SocketAddress listenSocketAddress = (SocketAddress)config.get(BoltConnector.listen_address);
        Assertions.assertEquals((Object)new SocketAddress("localhost", 7687), (Object)listenSocketAddress);
    }

    @Test
    void shouldBeAbleToOverrideBoltListenAddressesWithJustOneParameter() {
        Config config = Config.newBuilder().set(BoltConnector.enabled, (Object)true).set(BoltConnector.listen_address, (Object)new SocketAddress(8000)).build();
        Assertions.assertEquals((Object)new SocketAddress("localhost", 8000), (Object)config.get(BoltConnector.listen_address));
    }

    @Test
    void shouldDeriveBoltListenAddressFromDefaultListenAddress() {
        Config config = Config.newBuilder().set(BoltConnector.enabled, (Object)true).set(GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("0.0.0.0")).build();
        Assertions.assertEquals((Object)new SocketAddress("0.0.0.0", 7687), (Object)config.get(BoltConnector.listen_address));
    }

    @Test
    void shouldDeriveBoltListenAddressFromDefaultListenAddressAndSpecifiedPort() {
        Config config = Config.newBuilder().set(GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("0.0.0.0")).set(BoltConnector.enabled, (Object)true).set(BoltConnector.listen_address, (Object)new SocketAddress(8000)).build();
        Assertions.assertEquals((Object)new SocketAddress("0.0.0.0", 8000), (Object)config.get(BoltConnector.listen_address));
    }

    @Test
    void testServerDefaultSettings() {
        Config config = Config.newBuilder().setDefaults(GraphDatabaseSettings.SERVER_DEFAULTS).build();
        Assertions.assertEquals((Object)new SocketAddress("localhost", 7474), (Object)config.get(HttpConnector.listen_address));
        Assertions.assertEquals((Object)new SocketAddress("localhost", 7473), (Object)config.get(HttpsConnector.listen_address));
        Assertions.assertEquals((Object)new SocketAddress("localhost", 7687), (Object)config.get(BoltConnector.listen_address));
        Assertions.assertTrue((boolean)((Boolean)config.get(HttpConnector.enabled)));
        Assertions.assertFalse((boolean)((Boolean)config.get(HttpsConnector.enabled)));
        Assertions.assertTrue((boolean)((Boolean)config.get(BoltConnector.enabled)));
    }

    @Test
    void hasDefaultBookmarkAwaitTimeout() {
        Config config = Config.defaults();
        long bookmarkReadyTimeoutMs = ((Duration)config.get(GraphDatabaseSettings.bookmark_ready_timeout)).toMillis();
        Assertions.assertEquals((long)TimeUnit.SECONDS.toMillis(30L), (long)bookmarkReadyTimeoutMs);
    }

    @Test
    void throwsForIllegalBookmarkAwaitTimeout() {
        String[] illegalValues;
        for (String value : illegalValues = new String[]{"0ms", "0s", "10ms", "99ms", "999ms", "42ms"}) {
            Assertions.assertThrows(IllegalArgumentException.class, () -> {
                Config config = Config.defaults((Setting)GraphDatabaseSettings.bookmark_ready_timeout, (Object)((Duration)SettingValueParsers.DURATION.parse(value)));
                config.get(GraphDatabaseSettings.bookmark_ready_timeout);
            }, (String)("Exception expected for value '" + value + "'"));
        }
    }

    @Test
    void shouldDeriveListenAddressFromDefaultListenAddress() {
        Config config = Config.newBuilder().set(GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("0.0.0.0")).setDefaults(GraphDatabaseSettings.SERVER_DEFAULTS).build();
        Assertions.assertEquals((Object)"0.0.0.0", (Object)((SocketAddress)config.get(HttpConnector.listen_address)).getHostname());
        Assertions.assertEquals((Object)"0.0.0.0", (Object)((SocketAddress)config.get(HttpsConnector.listen_address)).getHostname());
        Assertions.assertEquals((Object)"0.0.0.0", (Object)((SocketAddress)config.get(BoltConnector.listen_address)).getHostname());
    }

    @Test
    void shouldDeriveListenAddressFromDefaultListenAddressAndSpecifiedPorts() {
        Config config = Config.newBuilder().set(GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("0.0.0.0")).set(HttpConnector.listen_address, (Object)new SocketAddress(8000)).set(HttpsConnector.listen_address, (Object)new SocketAddress(9000)).set(BoltConnector.listen_address, (Object)new SocketAddress(10000)).build();
        Assertions.assertEquals((Object)new SocketAddress("0.0.0.0", 9000), (Object)config.get(HttpsConnector.listen_address));
        Assertions.assertEquals((Object)new SocketAddress("0.0.0.0", 8000), (Object)config.get(HttpConnector.listen_address));
        Assertions.assertEquals((Object)new SocketAddress("0.0.0.0", 10000), (Object)config.get(BoltConnector.listen_address));
    }

    @Test
    void validateRetentionPolicy() {
        String[] validSet = new String[]{"true", "keep_all", "false", "keep_none", "10 files", "10k files", "10K size", "10m txs", "10M entries", "10g hours", "10G days"};
        String[] invalidSet = new String[]{"invalid", "all", "10", "10k", "10k a"};
        for (String valid : validSet) {
            Assertions.assertEquals((Object)valid, (Object)Config.defaults((Setting)GraphDatabaseSettings.keep_logical_logs, (Object)valid).get(GraphDatabaseSettings.keep_logical_logs));
        }
        for (String invalid : invalidSet) {
            Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.keep_logical_logs, (Object)invalid), (String)("Value \"" + invalid + "\" should be considered invalid"));
        }
    }

    @Test
    void transactionSamplingCanBePercentageValues() {
        IntStream range = IntStream.range(1, 101);
        range.forEach(percentage -> {
            Config config = Config.defaults((Setting)GraphDatabaseSettings.transaction_sampling_percentage, (Object)percentage);
            int configuredSampling = (Integer)config.get(GraphDatabaseSettings.transaction_sampling_percentage);
            Assertions.assertEquals((int)percentage, (int)configuredSampling);
        });
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.transaction_sampling_percentage, (Object)0));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.transaction_sampling_percentage, (Object)101));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.transaction_sampling_percentage, (Object)10101));
    }

    @Test
    void validateTransactionTracingLevelValues() {
        GraphDatabaseSettings.TransactionTracingLevel[] values;
        for (GraphDatabaseSettings.TransactionTracingLevel level : values = GraphDatabaseSettings.TransactionTracingLevel.values()) {
            Assertions.assertEquals((Object)level, (Object)Config.defaults((Setting)GraphDatabaseSettings.transaction_tracing_level, (Object)level).get(GraphDatabaseSettings.transaction_tracing_level));
        }
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.newBuilder().setRaw(Map.of(GraphDatabaseSettings.transaction_tracing_level.name(), "TRACE")).build());
    }

    @Test
    void configValuesContainsConnectors() {
        LogAssertions.assertThat(GraphDatabaseSettings.SERVER_DEFAULTS.keySet().stream().map(Setting::name).collect(Collectors.toList())).contains((Object[])new String[]{"server.http.enabled", "server.https.enabled", "server.bolt.enabled", "dbms.security.auth_enabled"});
    }

    @Test
    void testDefaultAddressOnlyAllowsHostname() {
        Assertions.assertDoesNotThrow(() -> Config.defaults((Setting)GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("foo")));
        Assertions.assertDoesNotThrow(() -> Config.defaults((Setting)GraphDatabaseSettings.default_advertised_address, (Object)new SocketAddress("bar")));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress("foo", 123)));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.default_advertised_address, (Object)new SocketAddress("bar", 456)));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.default_listen_address, (Object)new SocketAddress(123)));
        Assertions.assertThrows(IllegalArgumentException.class, () -> Config.defaults((Setting)GraphDatabaseSettings.default_advertised_address, (Object)new SocketAddress(456)));
    }
}

