/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import inet.ipaddr.IPAddressString;
import java.net.URI;
import java.nio.file.Path;
import java.time.Duration;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.DynamicTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestFactory;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.neo4j.configuration.SettingBuilder;
import org.neo4j.configuration.SettingConstraint;
import org.neo4j.configuration.SettingConstraints;
import org.neo4j.configuration.SettingImpl;
import org.neo4j.configuration.SettingValueParser;
import org.neo4j.configuration.SettingValueParsers;
import org.neo4j.configuration.helpers.DurationRange;
import org.neo4j.configuration.helpers.SocketAddress;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.internal.helpers.HostnamePort;
import org.neo4j.io.ByteUnit;
import org.neo4j.string.SecureString;

class SettingTest {
    private static final SettingValueParser<Integer> POSITIVE_INT = new SettingValueParser<Integer>(){

        public Integer parse(String value) {
            return (Integer)SettingValueParsers.INT.parse(value);
        }

        public String getDescription() {
            return "a positive integer";
        }

        public Class<Integer> getType() {
            return Integer.class;
        }

        public void validate(Integer value) {
            if (value <= 0) {
                throw new IllegalArgumentException("value %d is negative".formatted(value));
            }
        }
    };

    SettingTest() {
    }

    @Test
    void testInteger() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.INT);
        org.junit.jupiter.api.Assertions.assertEquals((int)5, (Integer)((Integer)setting.parse("5")));
        org.junit.jupiter.api.Assertions.assertEquals((int)5, (Integer)((Integer)setting.parse(" 5 ")));
        org.junit.jupiter.api.Assertions.assertEquals((int)-76, (Integer)((Integer)setting.parse("-76")));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
    }

    @Test
    void testLong() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.LONG);
        org.junit.jupiter.api.Assertions.assertEquals((long)112233445566778899L, (Long)((Long)setting.parse("112233445566778899")));
        org.junit.jupiter.api.Assertions.assertEquals((long)112233445566778899L, (Long)((Long)setting.parse(" 112233445566778899 ")));
        org.junit.jupiter.api.Assertions.assertEquals((long)-112233445566778899L, (Long)((Long)setting.parse("-112233445566778899")));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
    }

    @Test
    void testString() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.STRING);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)setting.parse("foo"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"bar", (Object)setting.parse("  bar   "));
    }

    @Test
    void testSecureString() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.SECURE_STRING);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)((SecureString)setting.parse("foo")).getString());
        org.junit.jupiter.api.Assertions.assertNotEquals((Object)"foo", (Object)((SecureString)setting.parse("foo")).toString());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"bar", (Object)((SecureString)setting.parse("  bar   ")).getString());
        org.junit.jupiter.api.Assertions.assertNotEquals((Object)"foo", (Object)setting.valueToString((Object)((SecureString)setting.parse("foo"))));
    }

    @Test
    void testDouble() {
        BiFunction<Double, Double, Boolean> compareDoubles = (d1, d2) -> Math.abs(d1 - d2) < 1.0E-6;
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.DOUBLE);
        org.junit.jupiter.api.Assertions.assertEquals((double)5.0, (Double)((Double)setting.parse("5")));
        org.junit.jupiter.api.Assertions.assertEquals((double)5.0, (Double)((Double)setting.parse("  5 ")));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)compareDoubles.apply(-0.123, (Double)setting.parse("-0.123")));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)compareDoubles.apply(5.0, (Double)setting.parse("5")));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
    }

    @Test
    void testList() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.INT));
        org.junit.jupiter.api.Assertions.assertEquals((int)5, (Integer)((Integer)((List)setting.parse("5")).get(0)));
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)((List)setting.parse("")).size());
        org.junit.jupiter.api.Assertions.assertEquals((int)4, (int)((List)setting.parse("5, 31, -4  ,2")).size());
        org.junit.jupiter.api.Assertions.assertEquals(Arrays.asList(4, 2, 3, 1), (Object)setting.parse("4,2,3,1"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("2,3,foo,7"));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)setting.valueToString((Object)((List)setting.parse("4,2,3,1"))).startsWith("["));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)setting.valueToString((Object)((List)setting.parse("4,2,3,1"))).endsWith("]"));
    }

    @Test
    void testListValidation() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.listOf(POSITIVE_INT));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(List.of(), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(List.of(Integer.valueOf(5)), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(List.of(Integer.valueOf(1), Integer.valueOf(2), Integer.valueOf(3)), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate(List.of(Integer.valueOf(1), Integer.valueOf(-2), Integer.valueOf(3)), Configuration.EMPTY));
    }

    @Test
    void testSet() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.setOf((SettingValueParser)SettingValueParsers.INT));
        Assertions.assertThat((Collection)((Collection)setting.parse("5"))).containsExactly((Object[])new Integer[]{5});
        Assertions.assertThat((Collection)((Collection)setting.parse(""))).isEmpty();
        Assertions.assertThat((Collection)((Collection)setting.parse("5, 31, -4  ,2"))).containsExactlyInAnyOrder((Object[])new Integer[]{5, 31, -4, 2});
        Assertions.assertThat((Collection)((Collection)setting.parse("5, 5, 5, 3, 900, 0"))).containsExactlyInAnyOrder((Object[])new Integer[]{0, 3, 5, 900});
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("2,3,foo,7"));
    }

    @Test
    void testSetValidation() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.setOf(POSITIVE_INT));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(Set.of(), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(Set.of(Integer.valueOf(5)), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(Set.of(Integer.valueOf(1), Integer.valueOf(2), Integer.valueOf(3)), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate(Set.of(Integer.valueOf(1), Integer.valueOf(-2), Integer.valueOf(3)), Configuration.EMPTY));
    }

    @Test
    void testEnum() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.ofEnum(Colors.class));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.BLUE), (Object)setting.parse("BLUE"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.GREEN), (Object)setting.parse("gReEn"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.RED), (Object)setting.parse("red"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.RED), (Object)setting.parse(" red "));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("orange"));
    }

    @Test
    void testPartialEnum() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.ofPartialEnum((Enum[])new Colors[]{Colors.GREEN, Colors.BLUE}));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.BLUE), (Object)setting.parse("BLUE"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)Colors.GREEN), (Object)setting.parse("gReEn"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("red"));
    }

    @Test
    void testStringEnum() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.ofEnum(StringEnum.class));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)StringEnum.DEFAULT), (Object)setting.parse("default"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)StringEnum.V_1), (Object)setting.parse("1.0"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)((Object)StringEnum.V_1_1), (Object)setting.parse("1.1"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("orange"));
    }

    @Test
    void testBool() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.BOOL);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)((Boolean)setting.parse("True")));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)setting.parse("false")));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)setting.parse("false")));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)((Boolean)setting.parse("true")));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)((Boolean)setting.parse(" true ")));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)((Boolean)setting.parse("  false")));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
    }

    @Test
    void testDuration() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.DURATION);
        org.junit.jupiter.api.Assertions.assertEquals((long)60L, (long)((Duration)setting.parse("1m")).toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)60L, (long)((Duration)setting.parse(" 1m ")).toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)1000L, (long)((Duration)setting.parse("1s")).toMillis());
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"1s", (Object)setting.valueToString((Object)((Duration)setting.parse("1s"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"3m", (Object)setting.valueToString((Object)((Duration)setting.parse("3m"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0s", (Object)setting.valueToString((Object)((Duration)setting.parse("0s"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0s", (Object)setting.valueToString((Object)((Duration)setting.parse("1ns"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0s", (Object)setting.valueToString((Object)((Duration)setting.parse("999999ns"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0s", (Object)setting.valueToString((Object)((Duration)setting.parse("999\u03bcs"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"11d19h25m4s50ms", (Object)setting.valueToString((Object)((Duration)setting.parse("11d19h25m4s50ms607\u03bcs80ns"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"2m1ms", (Object)setting.valueToString((Object)((Duration)setting.parse("1m60000ms1000000ns"))));
        String descriptionWithConstraint = SettingImpl.newBuilder((String)"setting", (SettingValueParser)SettingValueParsers.DURATION, (Object)Duration.ofMinutes(1L)).addConstraint(SettingConstraints.min((Comparable)Duration.ofSeconds(10L))).build().description();
        String expected = "setting, a duration (Valid units are: `ns`, `\u03bcs`, `ms`, `s`, `m`, `h` and `d`; default unit is `s`) that is minimum `10s`.";
        org.junit.jupiter.api.Assertions.assertEquals((Object)expected, (Object)descriptionWithConstraint);
    }

    @Test
    void testDurationRange() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.DURATION_RANGE);
        org.junit.jupiter.api.Assertions.assertEquals((long)60L, (long)((DurationRange)setting.parse("1m-2m")).getMin().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)120L, (long)((DurationRange)setting.parse("1m-2m")).getMax().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)60L, (long)((DurationRange)setting.parse(" 1m-2m ")).getMin().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)120L, (long)((DurationRange)setting.parse(" 1m-2m ")).getMax().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)1000L, (long)((DurationRange)setting.parse("1s-2s")).getMin().toMillis());
        org.junit.jupiter.api.Assertions.assertEquals((long)2000L, (long)((DurationRange)setting.parse("1s-2s")).getMax().toMillis());
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("1s"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("1s-"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("-1s"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("-1s--2s"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("2s-1s"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("2000ms-1s"));
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)((DurationRange)setting.parse("1s-1s")).getMin().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)((DurationRange)setting.parse("1s-1s")).getMax().toSeconds());
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)((DurationRange)setting.parse("1s-1s")).getDelta().toNanos());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0ns-0ns", (Object)setting.valueToString((Object)((DurationRange)setting.parse("0s-0s"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"1s-2s", (Object)setting.valueToString((Object)((DurationRange)setting.parse("1s-2s"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"3m-6m", (Object)setting.valueToString((Object)((DurationRange)setting.parse("[3m-6m]"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0ns-1m23s456ms", (Object)setting.valueToString((Object)((DurationRange)setting.parse("0s-1m23s456ms"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"1s-2s500ms", (Object)setting.valueToString((Object)((DurationRange)setting.parse("1000ms-2500ms"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"0ns-0ns", (Object)setting.valueToString((Object)((DurationRange)setting.parse("999\u03bcs-999999ns"))));
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)((DurationRange)setting.parse("999\u03bcs-999999ns")).getDelta().toNanos());
    }

    @Test
    void testHostnamePort() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.HOSTNAME_PORT);
        org.junit.jupiter.api.Assertions.assertEquals((Object)new HostnamePort("localhost", 7474), (Object)setting.parse("localhost:7474"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new HostnamePort("localhost", 1000, 2000), (Object)setting.parse("localhost:1000-2000"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new HostnamePort("localhost"), (Object)setting.parse("localhost"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("localhost:5641:7474"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("localhost:foo"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("7474:localhost"));
    }

    @Test
    void testTimeZone() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.TIMEZONE);
        org.junit.jupiter.api.Assertions.assertEquals((Object)ZoneId.from(ZoneOffset.UTC), (Object)setting.parse("+00:00"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)ZoneId.from(ZoneOffset.UTC), (Object)setting.parse(" +00:00 "));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("foo"));
    }

    @Test
    void testCidrIp() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.CIDR_IP);
        org.junit.jupiter.api.Assertions.assertEquals((Object)new IPAddressString("1.1.1.0/8"), (Object)setting.parse("1.1.1.0/8"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("garbage"));
    }

    @Test
    void testSocket() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.SOCKET_ADDRESS);
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("127.0.0.1", 7474), (Object)setting.parse("127.0.0.1:7474"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("127.0.0.1", 7474), (Object)setting.parse(" 127.0.0.1:7474 "));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("127.0.0.1", -1), (Object)setting.parse("127.0.0.1"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress(null, 7474), (Object)setting.parse(":7474"));
    }

    @Test
    void testSocketSolve() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.SOCKET_ADDRESS);
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("localhost", 7473), (Object)setting.solveDependency((Object)((SocketAddress)setting.parse("localhost:7473")), (Object)((SocketAddress)setting.parse("127.0.0.1:7474"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("127.0.0.1", 7473), (Object)setting.solveDependency((Object)((SocketAddress)setting.parse(":7473")), (Object)((SocketAddress)setting.parse("127.0.0.1:7474"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("127.0.0.1", 7473), (Object)setting.solveDependency((Object)((SocketAddress)setting.parse(":7473")), (Object)((SocketAddress)setting.parse("127.0.0.1"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("localhost", 7474), (Object)setting.solveDependency((Object)((SocketAddress)setting.parse("localhost")), (Object)((SocketAddress)setting.parse(":7474"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("localhost", 7474), (Object)setting.solveDependency((Object)((SocketAddress)setting.parse("localhost")), (Object)((SocketAddress)setting.parse("127.0.0.1:7474"))));
        org.junit.jupiter.api.Assertions.assertEquals((Object)new SocketAddress("localhost", 7474), (Object)setting.solveDependency(null, (Object)((SocketAddress)setting.parse("localhost:7474"))));
    }

    @Test
    void testBytes() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.BYTES);
        org.junit.jupiter.api.Assertions.assertEquals((long)2048L, (Long)((Long)setting.parse("2k")));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("1gig"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("-1M"));
        String descriptionWithConstraint = SettingImpl.newBuilder((String)"setting", (SettingValueParser)SettingValueParsers.BYTES, (Object)ByteUnit.gibiBytes((long)2L)).addConstraint(SettingConstraints.range((Comparable)Long.valueOf(ByteUnit.mebiBytes((long)100L)), (Comparable)Long.valueOf(ByteUnit.gibiBytes((long)10L)))).build().description();
        String expected = "setting, a byte size (valid multipliers are `B`, `KiB`, `KB`, `K`, `kB`, `kb`, `k`, `MiB`, `MB`, `M`, `mB`, `mb`, `m`, `GiB`, `GB`, `G`, `gB`, `gb`, `g`, `TiB`, `TB`, `PiB`, `PB`, `EiB`, `EB`) that is in the range `100.00MiB` to `10.00GiB`.";
        org.junit.jupiter.api.Assertions.assertEquals((Object)expected, (Object)descriptionWithConstraint);
    }

    @Test
    void testURI() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.URI);
        org.junit.jupiter.api.Assertions.assertEquals((Object)URI.create("/path/to/../something/"), (Object)setting.parse("/path/to/../something/"));
    }

    @Test
    void testHttpsURI() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.HTTPS_URI((boolean)true));
        org.junit.jupiter.api.Assertions.assertEquals((Object)URI.create("https://www.example.com/path"), (Object)setting.parse("https://www.example.com/path"));
    }

    @ParameterizedTest
    @ValueSource(strings={"http://www.example.com", "neo4js://database", "/path/to/../something/"})
    void testHttpsURIWithInvalidUris(String uri) {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.HTTPS_URI((boolean)true));
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse(uri));
        org.junit.jupiter.api.Assertions.assertEquals((Object)String.format("'%s' does not have required scheme 'https'", uri), (Object)exception.getMessage());
    }

    @ParameterizedTest
    @ValueSource(strings={"http://localhost/endpoint", "http://127.0.0.1/endpoint", "http://[::1]/endpoint", "http://[0:0:0:0:0:0:0:1]/endpoint"})
    void testHttpURIExemptionForLocalhostURIs(String uri) {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.HTTPS_URI((boolean)true));
        org.junit.jupiter.api.Assertions.assertEquals((Object)URI.create(uri), (Object)setting.parse(uri));
        SettingImpl invalidSetting = SettingTest.setting("setting", SettingValueParsers.HTTPS_URI((boolean)false));
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> invalidSetting.parse(uri));
        org.junit.jupiter.api.Assertions.assertEquals((Object)String.format("'%s' does not have required scheme 'https'", uri), (Object)exception.getMessage());
    }

    @Test
    void testStringMapWithNoConstraintOnKeys() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.MAP_PATTERN);
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k1", "v1", "k2", "v2"), (Object)setting.parse("k1=v1;k2=v2"));
    }

    @Test
    void testStringMapWithValuesContainingEquals() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.MAP_PATTERN);
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k1", "cn=admin,dc=example,dc=com", "k2", "v2"), (Object)setting.parse("k1=cn=admin,dc=example,dc=com;k2=v2"));
    }

    @Test
    void testStringMapWithRequiredKeys() {
        SettingImpl setting = SettingTest.setting("setting", new SettingValueParsers.MapPattern(Set.of("k1", "k2")));
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k1", "v1", "k2", "v2", "k3", "v3"), (Object)setting.parse("k1=v1;k2=v2;k3=v3"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("k1=v1;k3=v3"));
    }

    @Test
    void testStringMapWithRestrictedKeys() {
        SettingImpl setting = SettingTest.setting("setting", new SettingValueParsers.MapPattern(Set.of("k1"), Set.of("k1", "k2")));
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k1", "v1", "k2", "v2"), (Object)setting.parse("k1=v1;k2=v2"));
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k1", "v1"), (Object)setting.parse("k1=v1"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("k2=v2"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("k1=v1;k3=v3"));
        SettingImpl settingWithoutRequired = SettingTest.setting("setting", new SettingValueParsers.MapPattern(null, Set.of("k1", "k2")));
        org.junit.jupiter.api.Assertions.assertEquals(Map.of("k2", "v2"), (Object)settingWithoutRequired.parse("k2=v2"));
    }

    @Test
    void testNormalizedRelativeURI() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.NORMALIZED_RELATIVE_URI);
        org.junit.jupiter.api.Assertions.assertEquals((Object)URI.create("/path/to/something"), (Object)setting.parse("/path/away/from/../../to/something/"));
    }

    @Test
    void testPath() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.PATH);
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/absolute/path", new String[0]), (Object)setting.parse("/absolute/path"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/absolute/path", new String[0]), (Object)setting.parse("/absolute/wrong/../path"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/test/escaped/chars/r/n/dir", new String[0]), (Object)setting.parse("\test\\escaped\\chars\r\n\\\\dir"));
    }

    @Test
    void testSolvePath() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.PATH);
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/path/to/file", new String[0]).toAbsolutePath(), (Object)setting.solveDependency((Object)((Path)setting.parse("to/file")), (Object)((Path)setting.parse("/base/path")).toAbsolutePath()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/to/file", new String[0]).toAbsolutePath(), (Object)setting.solveDependency((Object)((Path)setting.parse("/to/file")), (Object)((Path)setting.parse("/base/path")).toAbsolutePath()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/path/", new String[0]).toAbsolutePath(), (Object)setting.solveDependency((Object)((Path)setting.parse("")), (Object)((Path)setting.parse("/base/path/")).toAbsolutePath()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base/path", new String[0]).toAbsolutePath(), (Object)setting.solveDependency((Object)((Path)setting.parse("path")), (Object)((Path)setting.parse("/base")).toAbsolutePath()));
        org.junit.jupiter.api.Assertions.assertEquals((Object)Path.of("/base", new String[0]).toAbsolutePath(), (Object)setting.solveDependency(null, (Object)((Path)setting.parse("/base")).toAbsolutePath()));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.solveDependency((Object)((Path)setting.parse("path")), (Object)((Path)setting.parse("base"))));
    }

    @Test
    void testJvmAdditional() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.JVM_ADDITIONAL);
        CharSequence[] inputs = new String[]{"value1", "value2 value3", "\"value 4\" \"value 5\"", "\"value  6\"", "value\"quoted\"", " valuewithspace  ", "strwithctrl\u000b\u0002", " values  with   spaces ", "\"one quoted\"   value  ", "  one  \"quoted   value\"", "\"two quoted\"  \"values\""};
        CharSequence[] outputs = new String[]{"value1", "value2", "value3", "value 4", "value 5", "value  6", "value\"quoted\"", "valuewithspace", "strwithctrl", "values", "with", "spaces", "one quoted", "value", "one", "quoted   value", "two quoted", "values"};
        String actualSettings = (String)setting.parse(String.join((CharSequence)System.lineSeparator(), inputs));
        String expectedSettings = String.join((CharSequence)System.lineSeparator(), outputs);
        org.junit.jupiter.api.Assertions.assertEquals((Object)expectedSettings, (Object)actualSettings);
    }

    @Test
    void testJvmAdditionalBadQuoting() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.JVM_ADDITIONAL);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.parse("\"missing_end_quote"));
    }

    @Test
    void testJvmAdditionalWithProperty() {
        SettingImpl setting = SettingTest.setting("setting", SettingValueParsers.JVM_ADDITIONAL);
        Assertions.assertThat((String)((String)setting.parse("-Da=\"string with space\""))).isEqualTo("-Da=\"string with space\"");
    }

    @Test
    void testDefaultSolve() {
        SettingValueParser<String> defaultSolver = new SettingValueParser<String>(){

            public String parse(String value) {
                return value;
            }

            public String getDescription() {
                return "default solver";
            }

            public Class<String> getType() {
                return String.class;
            }
        };
        SettingImpl<String> setting = SettingTest.setting("setting", defaultSolver);
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)setting.solveDependency((Object)"foo", (Object)"bar"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"bar", (Object)setting.solveDependency(null, (Object)"bar"));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"foo", (Object)setting.solveDependency((Object)"foo", null));
        org.junit.jupiter.api.Assertions.assertNull((Object)setting.solveDependency(null, null));
    }

    @Test
    void testMinConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.INT).addConstraint(SettingConstraints.min((Comparable)Integer.valueOf(10))).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)100, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)10, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)9, Configuration.EMPTY));
    }

    @Test
    void testMaxConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.INT).addConstraint(SettingConstraints.max((Comparable)Integer.valueOf(10))).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)-100, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)10, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)11, Configuration.EMPTY));
    }

    @Test
    void testRangeConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.DOUBLE).addConstraint(SettingConstraints.range((Comparable)Double.valueOf(10.0), (Comparable)Double.valueOf(20.0))).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)9.9, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)20.01, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)10.1, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)19.9999, Configuration.EMPTY));
    }

    @Test
    void testLessThanOrEqualConstraint() {
        SettingImpl intLimit = (SettingImpl)SettingTest.settingBuilder("limit.int", SettingValueParsers.INT).build();
        SettingImpl durationLimit = (SettingImpl)SettingTest.settingBuilder("limit.duration", SettingValueParsers.DURATION).build();
        final HashMap<SettingImpl, Comparable<Integer>> settings = new HashMap<SettingImpl, Comparable<Integer>>();
        Configuration simpleConfig = new Configuration(){

            public <T> T get(Setting<T> setting) {
                return (T)settings.get(setting);
            }
        };
        settings.put(intLimit, Integer.valueOf(5));
        settings.put(durationLimit, Duration.ofSeconds(123L));
        SettingImpl mustBeLessSetting = (SettingImpl)SettingTest.settingBuilder("less.than.duration", SettingValueParsers.INT).addConstraint(SettingConstraints.lessThanOrEqual((Setting)intLimit)).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessSetting.validate((Object)-1, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessSetting.validate((Object)0, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessSetting.validate((Object)1, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessSetting.validate((Object)5, simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> mustBeLessSetting.validate((Object)6, simpleConfig));
        SettingImpl mustBeLessThanHalfSetting = (SettingImpl)SettingTest.settingBuilder("less.than.half.int", SettingValueParsers.INT).addConstraint(SettingConstraints.lessThanOrEqual(i -> (long)i, (Setting)intLimit, i -> i / 2L, (String)"divided by 2")).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfSetting.validate((Object)-1, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfSetting.validate((Object)0, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfSetting.validate((Object)2, simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> mustBeLessThanHalfSetting.validate((Object)3, simpleConfig));
        SettingImpl mustBeLessDuration = (SettingImpl)SettingTest.settingBuilder("less.than.duration", SettingValueParsers.DURATION).addConstraint(SettingConstraints.lessThanOrEqual(Duration::toMillis, (Setting)durationLimit)).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessDuration.validate((Object)Duration.ofSeconds(-1L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessDuration.validate((Object)Duration.ofSeconds(0L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessDuration.validate((Object)Duration.ofMinutes(1L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessDuration.validate((Object)Duration.ofSeconds(123L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> mustBeLessDuration.validate((Object)Duration.ofMillis(123001L), simpleConfig));
        SettingImpl mustBeLessThanHalfDuration = (SettingImpl)SettingTest.settingBuilder("less.than.duration", SettingValueParsers.DURATION).addConstraint(SettingConstraints.lessThanOrEqual(Duration::toMillis, (Setting)durationLimit, i -> i / 2L, (String)"divided by 2")).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfDuration.validate((Object)Duration.ofSeconds(-1L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfDuration.validate((Object)Duration.ofSeconds(0L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfDuration.validate((Object)Duration.ofMinutes(1L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> mustBeLessThanHalfDuration.validate((Object)Duration.ofSeconds(61L), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> mustBeLessThanHalfDuration.validate((Object)Duration.ofMillis(61501L), simpleConfig));
    }

    @Test
    void testExceptConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.STRING).addConstraint(SettingConstraints.except((String[])new String[]{"foo"})).build();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)"foo", Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)"bar", Configuration.EMPTY));
    }

    @Test
    void testMatchesConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.STRING).addConstraint(SettingConstraints.matches((String)"^[^.]+\\.[^.]+$")).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)"foo.bar", Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)"foo", Configuration.EMPTY));
    }

    @Test
    void testPowerOf2Constraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.LONG).addConstraint(SettingConstraints.POWER_OF_2).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)8L, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)0x100000000L, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)1023L, Configuration.EMPTY));
    }

    @Test
    void testIsConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.INT).addConstraint(SettingConstraints.is((Object)10)).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate((Object)10, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate((Object)9, Configuration.EMPTY));
    }

    @Test
    void testAnyConstraint() {
        SettingImpl intSetting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.INT).addConstraint(SettingConstraints.any((SettingConstraint)SettingConstraints.min((Comparable)Integer.valueOf(30)), (SettingConstraint[])new SettingConstraint[]{SettingConstraints.is((Object)0), SettingConstraints.is((Object)-10)})).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> intSetting.validate((Object)30, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> intSetting.validate((Object)100, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> intSetting.validate((Object)0, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> intSetting.validate((Object)-10, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> intSetting.validate((Object)29, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> intSetting.validate((Object)1, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> intSetting.validate((Object)-9, Configuration.EMPTY));
        SettingImpl durationSetting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.DURATION).addConstraint(SettingConstraints.any((SettingConstraint)SettingConstraints.min((Comparable)Duration.ofMinutes(30L)), (SettingConstraint[])new SettingConstraint[]{SettingConstraints.is((Object)Duration.ZERO)})).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> durationSetting.validate((Object)Duration.ofMinutes(30L), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> durationSetting.validate((Object)Duration.ofHours(1L), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> durationSetting.validate((Object)Duration.ZERO, Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> durationSetting.validate((Object)Duration.ofMinutes(29L), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> durationSetting.validate((Object)Duration.ofMillis(1L), Configuration.EMPTY));
        String expected = "setting, a duration (Valid units are: `ns`, `\u03bcs`, `ms`, `s`, `m`, `h` and `d`; default unit is `s`) that is minimum `30m` or is `0s`.";
        org.junit.jupiter.api.Assertions.assertEquals((Object)expected, (Object)durationSetting.description());
    }

    @Test
    void testDependencyConstraint() {
        SettingImpl intSetting = (SettingImpl)SettingTest.settingBuilder("int-setting", SettingValueParsers.INT).build();
        SettingImpl enumSetting = (SettingImpl)SettingTest.settingBuilder("enum-setting", SettingValueParsers.ofEnum(Colors.class)).build();
        final HashMap<SettingImpl, Object> settings = new HashMap<SettingImpl, Object>();
        Configuration simpleConfig = new Configuration(){

            public <T> T get(Setting<T> setting) {
                return (T)settings.get(setting);
            }
        };
        SettingImpl dependingIntSetting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.INT).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(3)), (SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(7)), (Setting)intSetting, (SettingConstraint)SettingConstraints.min((Comparable)Integer.valueOf(3)))).build();
        SettingImpl dependingEnumSetting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.STRING)).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.size((int)2), (SettingConstraint)SettingConstraints.size((int)4), (Setting)enumSetting, (SettingConstraint)SettingConstraints.is((Object)((Object)Colors.BLUE)))).build();
        settings.put(intSetting, 5);
        settings.put(enumSetting, (Object)Colors.BLUE);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> dependingIntSetting.validate((Object)3, simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingIntSetting.validate((Object)4, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> dependingEnumSetting.validate(List.of("a", "b"), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingEnumSetting.validate(List.of("a", "b", "c"), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingEnumSetting.validate(List.of("a", "b", "c", "d"), simpleConfig));
        settings.put(intSetting, 2);
        settings.put(enumSetting, (Object)Colors.GREEN);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> dependingIntSetting.validate((Object)4, simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingIntSetting.validate((Object)8, simpleConfig));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> dependingEnumSetting.validate(List.of("a", "b", "c", "d"), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingEnumSetting.validate(List.of("a", "b"), simpleConfig));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> dependingEnumSetting.validate(List.of("a", "b", "c"), simpleConfig));
    }

    @Test
    void testDescriptionWithConstraints() {
        SettingImpl oneConstraintSetting = (SettingImpl)SettingTest.settingBuilder("setting.name", SettingValueParsers.LONG).addConstraint(SettingConstraints.POWER_OF_2).build();
        SettingImpl twoConstraintSetting = (SettingImpl)SettingTest.settingBuilder("setting.name", SettingValueParsers.INT).addConstraint(SettingConstraints.min((Comparable)Integer.valueOf(2))).addConstraint(SettingConstraints.max((Comparable)Integer.valueOf(10))).build();
        SettingImpl enumSetting = (SettingImpl)SettingTest.settingBuilder("setting.name", SettingValueParsers.ofEnum(Colors.class)).build();
        SettingImpl intSetting = (SettingImpl)SettingTest.settingBuilder("setting.name", SettingValueParsers.INT).build();
        SettingImpl dependencySetting1 = (SettingImpl)SettingTest.settingBuilder("setting.depending.name", SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.STRING)).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.size((int)2), (SettingConstraint)SettingConstraints.size((int)4), (Setting)enumSetting, (SettingConstraint)SettingConstraints.is((Object)((Object)Colors.BLUE)))).build();
        SettingImpl dependencySetting2 = (SettingImpl)SettingTest.settingBuilder("setting.depending.name", SettingValueParsers.INT).addConstraint(SettingConstraints.dependency((SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(3)), (SettingConstraint)SettingConstraints.max((Comparable)Integer.valueOf(7)), (Setting)intSetting, (SettingConstraint)SettingConstraints.min((Comparable)Integer.valueOf(3)))).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.name, a long that is power of 2.", (Object)oneConstraintSetting.description());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.name, an integer that is minimum `2` and is maximum `10`.", (Object)twoConstraintSetting.description());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.depending.name, a comma-separated list where each element is a string, which depends on setting.name. If setting.name is `BLUE` then it is of size `2` otherwise it is of size `4`.", (Object)dependencySetting1.description());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.depending.name, an integer that depends on setting.name. If setting.name is minimum `3` then it is maximum `3` otherwise it is maximum `7`.", (Object)dependencySetting2.description());
    }

    @Test
    void testListOfEnums() {
        SettingImpl enumSetting = (SettingImpl)SettingImpl.newBuilder((String)"setting.name", (SettingValueParser)SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.ofEnum(Colors.class)), List.of(Colors.GREEN)).build();
        List parsedSetting = (List)enumSetting.parse("red, blue");
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)parsedSetting.size());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)parsedSetting.containsAll(List.of(Colors.BLUE, Colors.RED)));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)((List)enumSetting.parse("")).isEmpty());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.name, a comma-separated list where each element is one of [BLUE, GREEN, RED].", (Object)enumSetting.description());
        org.junit.jupiter.api.Assertions.assertEquals(List.of(Colors.GREEN), (Object)enumSetting.defaultValue());
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> enumSetting.parse("blue, kaputt"));
    }

    @Test
    void testSetOfEnums() {
        SettingImpl enumSetting = (SettingImpl)SettingImpl.newBuilder((String)"setting.name", (SettingValueParser)SettingValueParsers.setOfEnums(Colors.class), EnumSet.of(Colors.GREEN)).build();
        Set parsedSetting = (Set)enumSetting.parse("red, blue, red");
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)parsedSetting.size());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)parsedSetting.containsAll(List.of(Colors.BLUE, Colors.RED)));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)((Set)enumSetting.parse("")).isEmpty());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"setting.name, a comma-separated set where each element is one of [BLUE, GREEN, RED].", (Object)enumSetting.description());
        org.junit.jupiter.api.Assertions.assertEquals(Set.of(Colors.GREEN), (Object)enumSetting.defaultValue());
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> enumSetting.parse("blue, kaputt"));
    }

    @Test
    void testNoDuplicatesConstraint() {
        SettingImpl setting = (SettingImpl)SettingTest.settingBuilder("setting", SettingValueParsers.listOf((SettingValueParser)SettingValueParsers.STRING)).addConstraint(SettingConstraints.noDuplicates()).build();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(List.of("a", "b"), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> setting.validate(List.of(), Configuration.EMPTY));
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> setting.validate(List.of("a", "b", "b"), Configuration.EMPTY));
        org.junit.jupiter.api.Assertions.assertEquals((Object)"Failed to validate '[a, b, b]' for 'setting': items should not have duplicates: a,b,b", (Object)exception.getMessage());
    }

    @TestFactory
    Collection<DynamicTest> testDescriptionDependency() {
        ArrayList<DynamicTest> tests = new ArrayList<DynamicTest>();
        tests.add(DynamicTest.dynamicTest((String)"Test int dependency description", () -> SettingTest.testDescDependency(SettingValueParsers.INT, "setting.child, an integer. If unset, the value is inherited from setting.parent.")));
        tests.add(DynamicTest.dynamicTest((String)"Test socket dependency description", () -> SettingTest.testDescDependency(SettingValueParsers.SOCKET_ADDRESS, "setting.child, a socket address in the format of `hostname:port`, `hostname`, or `:port`. If missing, it is acquired from setting.parent.")));
        tests.add(DynamicTest.dynamicTest((String)"Test path dependency description", () -> SettingTest.testDescDependency(SettingValueParsers.PATH, "setting.child, a path. If relative, it is resolved from setting.parent.")));
        return tests;
    }

    private static <T> void testDescDependency(SettingValueParser<T> parser, String expectedDescription) {
        Setting parent = SettingTest.settingBuilder("setting.parent", parser).immutable().build();
        Setting child = SettingTest.settingBuilder("setting.child", parser).setDependency(parent).build();
        org.junit.jupiter.api.Assertions.assertEquals((Object)expectedDescription, (Object)child.description());
    }

    private static <T> SettingBuilder<T> settingBuilder(String name, SettingValueParser<T> parser) {
        return SettingImpl.newBuilder((String)name, parser, null);
    }

    private static <T> SettingImpl<T> setting(String name, SettingValueParser<T> parser) {
        return (SettingImpl)SettingImpl.newBuilder((String)name, parser, null).build();
    }

    private static enum Colors {
        BLUE,
        GREEN,
        RED;

    }

    private static enum StringEnum {
        DEFAULT("default"),
        V_1("1.0"),
        V_1_1("1.1");

        private final String name;

        private StringEnum(String name) {
            this.name = name;
        }

        public String toString() {
            return this.name;
        }
    }
}

