/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.cypher.example;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.ObjectWriter;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.neo4j.cypher.javacompat.ExecutionEngine;
import org.neo4j.cypher.javacompat.ExecutionResult;
import org.neo4j.cypher.javacompat.RegularExpressionMatcher;
import org.neo4j.graphdb.DynamicRelationshipType;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.PropertyContainer;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.helpers.collection.IteratorUtil;
import org.neo4j.test.AsciiDocGenerator;
import org.neo4j.test.TestGraphDatabaseFactory;
import org.neo4j.visualization.asciidoc.AsciidocHelper;

public class JavaExecutionEngineDocTest {
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final ObjectWriter WRITER = MAPPER.writerWithDefaultPrettyPrinter();
    private static final File docsTargetDir = new File("target/docs/dev/general");
    private GraphDatabaseService db;
    private ExecutionEngine engine;
    private Node andreasNode;
    private Node johanNode;
    private Node michaelaNode;

    @BeforeClass
    public static void prepare() {
        docsTargetDir.mkdirs();
    }

    @Before
    public void setUp() throws IOException {
        this.db = new TestGraphDatabaseFactory().newImpermanentDatabaseBuilder().newGraphDatabase();
        this.engine = new ExecutionEngine(this.db);
        Transaction tx = this.db.beginTx();
        this.michaelaNode = this.db.createNode();
        this.andreasNode = this.db.createNode();
        this.johanNode = this.db.createNode();
        this.andreasNode.setProperty("name", (Object)"Andreas");
        this.johanNode.setProperty("name", (Object)"Johan");
        this.michaelaNode.setProperty("name", (Object)"Michaela");
        this.index(this.andreasNode);
        this.index(this.johanNode);
        this.index(this.michaelaNode);
        tx.success();
        tx.finish();
    }

    @After
    public void shutdownDb() {
        if (this.db != null) {
            this.db.shutdown();
        }
        this.db = null;
    }

    private void index(Node n) {
        this.db.index().forNodes("people").add((PropertyContainer)n, "name", n.getProperty("name"));
    }

    public static String parametersToAsciidoc(Object params) throws JsonGenerationException, JsonMappingException, IOException {
        StringBuffer sb = new StringBuffer(2048);
        String prettifiedJson = WRITER.writeValueAsString(params);
        sb.append("\n.Parameters\n[source,javascript]\n----\n").append(prettifiedJson).append("\n----\n\n");
        return sb.toString();
    }

    private void dumpToFile(String id2, String query, Object params) throws Exception {
        StringBuffer sb = new StringBuffer(2048);
        String prettifiedJson = WRITER.writeValueAsString(params);
        sb.append("\n.Parameters\n[source,javascript]\n----\n").append(prettifiedJson).append("\n----\n\n.Query\n").append(AsciidocHelper.createAsciiDocSnippet((String)"cypher", (String)this.engine.prettify(query)));
        AsciiDocGenerator.dumpToSeparateFile((File)docsTargetDir, (String)id2, (String)sb.toString());
    }

    @Test
    public void exampleQuery() throws Exception {
        ExecutionEngine engine = new ExecutionEngine(this.db);
        ExecutionResult result = engine.execute("START n=node(0) WHERE 1=1 RETURN n");
        Assert.assertThat((Object)result.columns(), (Matcher)CoreMatchers.hasItem((Object)"n"));
        ResourceIterator n_column = result.columnAs("n");
        Assert.assertThat((Object)IteratorUtil.asIterable((Iterator)n_column), (Matcher)CoreMatchers.hasItem((Object)this.db.getNodeById(0L)));
    }

    @Test
    public void shouldBeAbleToEmitJavaIterables() throws Exception {
        this.makeFriends(this.michaelaNode, this.andreasNode);
        this.makeFriends(this.michaelaNode, this.johanNode);
        ExecutionEngine engine = new ExecutionEngine(this.db);
        ExecutionResult result = engine.execute("START n=node(0) MATCH n-->friend RETURN collect(friend)");
        Iterable friends = (Iterable)result.columnAs("collect(friend)").next();
        Assert.assertThat((Object)friends, (Matcher)CoreMatchers.hasItems((Object[])new Node[]{this.andreasNode, this.johanNode}));
        Assert.assertThat((Object)friends, (Matcher)CoreMatchers.instanceOf(Iterable.class));
    }

    @Test
    public void testColumnAreInTheRightOrder() throws Exception {
        this.createTenNodes();
        String q = "start one=node(1), two=node(2), three=node(3), four=node(4), five=node(5), six=node(6), seven=node(7), eight=node(8), nine=node(9), ten=node(10) return one, two, three, four, five, six, seven, eight, nine, ten";
        ExecutionResult result = this.engine.execute(q);
        Assert.assertThat((Object)result.dumpToString(), (Matcher)RegularExpressionMatcher.matchesPattern((String)"one.*two.*three.*four.*five.*six.*seven.*eight.*nine.*ten"));
    }

    private void createTenNodes() {
        try (Transaction tx = this.db.beginTx();){
            for (int i = 0; i < 10; ++i) {
                this.db.createNode();
            }
            tx.success();
        }
    }

    @Test
    public void exampleWithParameterForNodeId() throws Exception {
        HashMap<String, Integer> params = new HashMap<String, Integer>();
        params.put("id", 0);
        String query = "START n=node({id}) RETURN n.name";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertThat((Object)result.columns(), (Matcher)CoreMatchers.hasItem((Object)"n.name"));
        ResourceIterator n_column = result.columnAs("n.name");
        Assert.assertEquals((Object)"Michaela", n_column.next());
        this.dumpToFile("exampleWithParameterForNodeId", query, params);
    }

    @Test
    public void exampleWithParameterForMultipleNodeIds() throws Exception {
        HashMap<String, List<Integer>> params = new HashMap<String, List<Integer>>();
        params.put("id", Arrays.asList(0, 1, 2));
        String query = "START n=node({id}) RETURN n.name";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertEquals(Arrays.asList("Michaela", "Andreas", "Johan"), this.toList(result, "n.name"));
        this.dumpToFile("exampleWithParameterForMultipleNodeIds", query, params);
    }

    private <T> List<T> toList(ExecutionResult result, String column) {
        ArrayList results = new ArrayList();
        IteratorUtil.addToCollection((Iterator)result.columnAs(column), results);
        return results;
    }

    @Test
    public void exampleWithStringLiteralAsParameter() throws Exception {
        HashMap<String, String> params = new HashMap<String, String>();
        params.put("name", "Johan");
        String query = "MATCH (n) WHERE n.name = {name} RETURN n";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertEquals(Arrays.asList(this.johanNode), this.toList(result, "n"));
        this.dumpToFile("exampleWithStringLiteralAsParameter", query, params);
    }

    @Test
    public void exampleWithParameterForIndexValue() throws Exception {
        try (Transaction ignored = this.db.beginTx();){
            HashMap<String, String> params = new HashMap<String, String>();
            params.put("value", "Michaela");
            String query = "START n=node:people(name = {value}) RETURN n";
            ExecutionResult result = this.engine.execute(query, params);
            Assert.assertEquals(Arrays.asList(this.michaelaNode), this.toList(result, "n"));
            this.dumpToFile("exampleWithParameterForIndexValue", query, params);
        }
    }

    @Test
    public void exampleWithParametersForQuery() throws Exception {
        try (Transaction ignored = this.db.beginTx();){
            HashMap<String, String> params = new HashMap<String, String>();
            params.put("query", "name:Andreas");
            String query = "START n=node:people({query}) RETURN n";
            ExecutionResult result = this.engine.execute(query, params);
            Assert.assertEquals(Arrays.asList(this.andreasNode), this.toList(result, "n"));
            this.dumpToFile("exampleWithParametersForQuery", query, params);
        }
    }

    @Test
    public void exampleWithParameterForNodeObject() throws Exception {
        HashMap<String, Node> params = new HashMap<String, Node>();
        params.put("node", this.andreasNode);
        String query = "START n=node({node}) RETURN n.name";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertThat((Object)result.columns(), (Matcher)CoreMatchers.hasItem((Object)"n.name"));
        ResourceIterator n_column = result.columnAs("n.name");
        Assert.assertEquals((Object)"Andreas", n_column.next());
    }

    @Test
    public void exampleWithParameterForSkipAndLimit() throws Exception {
        HashMap<String, Integer> params = new HashMap<String, Integer>();
        params.put("s", 1);
        params.put("l", 1);
        String query = "MATCH (n) RETURN n.name SKIP {s} LIMIT {l}";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertThat((Object)result.columns(), (Matcher)CoreMatchers.hasItem((Object)"n.name"));
        ResourceIterator n_column = result.columnAs("n.name");
        Assert.assertEquals((Object)"Andreas", n_column.next());
        this.dumpToFile("exampleWithParameterForSkipLimit", query, params);
    }

    @Test
    public void exampleWithParameterRegularExpression() throws Exception {
        HashMap<String, String> params = new HashMap<String, String>();
        params.put("regex", ".*h.*");
        String query = "MATCH (n) WHERE n.name =~ {regex} RETURN n.name";
        ExecutionResult result = this.engine.execute(query, params);
        this.dumpToFile("exampleWithParameterRegularExpression", query, params);
        Assert.assertThat((Object)result.columns(), (Matcher)CoreMatchers.hasItem((Object)"n.name"));
        ResourceIterator n_column = result.columnAs("n.name");
        Assert.assertEquals((Object)"Michaela", n_column.next());
        Assert.assertEquals((Object)"Johan", n_column.next());
    }

    @Test
    public void create_node_from_map() throws Exception {
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("name", "Andres");
        props.put("position", "Developer");
        HashMap<String, HashMap<String, String>> params = new HashMap<String, HashMap<String, String>>();
        params.put("props", props);
        String query = "CREATE ({props})";
        this.engine.execute(query, params);
        this.dumpToFile("create_node_from_map", query, params);
        ExecutionResult result = this.engine.execute("match (n) where n.name = 'Andres' and n.position = 'Developer' return n");
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)1));
    }

    @Test
    public void create_multiple_nodes_from_map() throws Exception {
        HashMap<String, Object> n1 = new HashMap<String, Object>();
        n1.put("name", "Andres");
        n1.put("position", "Developer");
        n1.put("awesome", true);
        HashMap<String, Object> n2 = new HashMap<String, Object>();
        n2.put("name", "Michael");
        n2.put("position", "Developer");
        n2.put("children", 3);
        HashMap<String, List<Map>> params = new HashMap<String, List<Map>>();
        List<Map> maps = Arrays.asList(n1, n2);
        params.put("props", maps);
        String query = "CREATE (n:Person {props}) RETURN n";
        this.engine.execute(query, params);
        this.dumpToFile("create_multiple_nodes_from_map", query, params);
        ExecutionResult result = this.engine.execute("match (n:Person) where n.name in ['Andres', 'Michael'] and n.position = 'Developer' return n");
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)2));
        result = this.engine.execute("match (n:Person) where n.children = 3 return n");
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)1));
        result = this.engine.execute("match (n:Person) where n.awesome = true return n");
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)1));
    }

    @Test
    public void set_properties_on_a_node_from_a_map() throws Exception {
        try (Transaction tx = this.db.beginTx();){
            HashMap<String, String> n1 = new HashMap<String, String>();
            n1.put("name", "Andres");
            n1.put("position", "Developer");
            HashMap<String, HashMap<String, String>> params = new HashMap<String, HashMap<String, String>>();
            params.put("props", n1);
            String query = "MATCH (n) WHERE n.name='Michaela' SET n = {props}";
            this.engine.execute(query, params);
            this.dumpToFile("set_properties_on_a_node_from_a_map", query, params);
            this.engine.execute("match (n) where n.name in ['Andres', 'Michael'] and n.position = 'Developer' return n");
            Assert.assertThat((Object)this.michaelaNode.getProperty("name").toString(), (Matcher)CoreMatchers.is((Object)"Andres"));
        }
    }

    @Test
    public void create_node_using_create_unique_with_java_maps() throws Exception {
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("name", "Andres");
        props.put("position", "Developer");
        HashMap<String, HashMap<String, String>> params = new HashMap<String, HashMap<String, String>>();
        params.put("props", props);
        String query = "START n=node(0) CREATE UNIQUE p = n-[:REL]->({props}) RETURN last(nodes(p)) AS X";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)1));
    }

    @Test
    public void should_be_able_to_handle_two_params_without_named_nodes() throws Exception {
        HashMap<String, String> props1 = new HashMap<String, String>();
        props1.put("name", "Andres");
        props1.put("position", "Developer");
        HashMap<String, String> props2 = new HashMap<String, String>();
        props2.put("name", "Lasse");
        props2.put("awesome", "true");
        HashMap<String, HashMap<String, String>> params = new HashMap<String, HashMap<String, String>>();
        params.put("props1", props1);
        params.put("props2", props2);
        String query = "START n=node(0) CREATE UNIQUE p = n-[:REL]->({props1})-[:LER]->({props2}) RETURN p";
        ExecutionResult result = this.engine.execute(query, params);
        Assert.assertThat((Object)IteratorUtil.count((Iterable)result), (Matcher)CoreMatchers.is((Object)1));
    }

    @Test
    public void prettifier_makes_pretty() throws Exception {
        String given = "match (n)-->() return n";
        String expected = String.format("MATCH (n)-->()%nRETURN n", new Object[0]);
        Assert.assertEquals((Object)expected, (Object)this.engine.prettify(given));
    }

    private void makeFriends(Node a, Node b) {
        try (Transaction tx = this.db.beginTx();){
            a.createRelationshipTo(b, (RelationshipType)DynamicRelationshipType.withName((String)"friend"));
            tx.success();
        }
    }
}

