/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.onlinebackup;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import org.neo4j.kernel.EmbeddedGraphDatabase;
import org.neo4j.kernel.impl.transaction.XaDataSourceManager;
import org.neo4j.kernel.impl.transaction.xaframework.XaDataSource;
import org.neo4j.onlinebackup.Backup;
import org.neo4j.onlinebackup.impl.EmbeddedGraphDatabaseResource;
import org.neo4j.onlinebackup.impl.LocalGraphDatabaseResource;
import org.neo4j.onlinebackup.impl.Neo4jResource;
import org.neo4j.onlinebackup.impl.XaDataSourceResource;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Neo4jBackup
implements Backup {
    private static final String DEFAULT_BACKUP_LOG_LOCATION = "backup.log";
    private static final Level LOG_LEVEL_NORMAL = Level.INFO;
    private static final Level LOG_LEVEL_DEBUG = Level.ALL;
    private static final Level LOG_LEVEL_OFF = Level.OFF;
    private static Logger logger = Logger.getLogger(Neo4jBackup.class.getName());
    private static ConsoleHandler consoleHandler = new ConsoleHandler();
    private static FileHandler fileHandler = null;
    private final EmbeddedGraphDatabase onlineGraphDb;
    private final ResourceFetcher destinationResourceFetcher;
    private final List<String> xaNames;

    public static Backup neo4jDataSource(EmbeddedGraphDatabase source, String destinationDir) {
        return new Neo4jBackup(source, new DestinationDirResourceFetcher(destinationDir), Collections.singletonList("nioneodb"));
    }

    public static Backup neo4jDataSource(EmbeddedGraphDatabase source, EmbeddedGraphDatabase destination) {
        return new Neo4jBackup(source, new GraphDbResourceFetcher(destination), Collections.singletonList("nioneodb"));
    }

    public static Backup allDataSources(EmbeddedGraphDatabase source, String destinationDir) {
        return new Neo4jBackup(source, new DestinationDirResourceFetcher(destinationDir), Neo4jBackup.allDataSources(source));
    }

    public static Backup allDataSources(EmbeddedGraphDatabase source, EmbeddedGraphDatabase destination) {
        return new Neo4jBackup(source, new GraphDbResourceFetcher(destination), Neo4jBackup.allDataSources(source));
    }

    public static Backup customDataSources(EmbeddedGraphDatabase source, String destinationDir, String ... xaDataSourceNames) {
        return new Neo4jBackup(source, new DestinationDirResourceFetcher(destinationDir), new ArrayList<String>(Arrays.asList(xaDataSourceNames)));
    }

    public static Backup customDataSources(EmbeddedGraphDatabase source, EmbeddedGraphDatabase destination, String ... xaDataSourceNames) {
        return new Neo4jBackup(source, new GraphDbResourceFetcher(destination), new ArrayList<String>(new ArrayList<String>(Arrays.asList(xaDataSourceNames))));
    }

    private Neo4jBackup(EmbeddedGraphDatabase source, ResourceFetcher destination, List<String> xaDataSources) {
        if (source == null) {
            throw new IllegalArgumentException("The source graph db instance is null.");
        }
        if (xaDataSources == null) {
            throw new IllegalArgumentException("XA data source name list is null");
        }
        this.onlineGraphDb = source;
        this.destinationResourceFetcher = destination;
        this.xaNames = xaDataSources;
        this.assertLogicalLogsAreKept();
    }

    private static List<String> allDataSources(EmbeddedGraphDatabase db) {
        ArrayList<String> result = new ArrayList<String>();
        for (XaDataSource dataSource : db.getConfig().getTxModule().getXaDataSourceManager().getAllRegisteredDataSources()) {
            result.add(dataSource.getName());
        }
        return result;
    }

    private void assertLogicalLogsAreKept() {
        if (this.xaNames.size() < 1) {
            throw new IllegalArgumentException("No XA data source names in list");
        }
        XaDataSourceManager xaDataSourceManager = this.onlineGraphDb.getConfig().getTxModule().getXaDataSourceManager();
        for (String xaDataSourceName : this.xaNames) {
            XaDataSource xaDataSource = xaDataSourceManager.getXaDataSource(xaDataSourceName);
            if (xaDataSource.isLogicalLogKept()) continue;
            throw new IllegalStateException("Backup cannot be run, as the data source [" + xaDataSourceName + "," + xaDataSource + "] is not configured to keep logical logs.");
        }
    }

    @Override
    public void doBackup() throws IOException {
        logger.info("Initializing backup.");
        EmbeddedGraphDatabaseResource srcResource = new EmbeddedGraphDatabaseResource(this.onlineGraphDb);
        Neo4jResource dstResource = this.destinationResourceFetcher.fetch();
        if (this.xaNames.size() == 1) {
            this.runSimpleBackup(srcResource, dstResource);
        } else {
            this.runMultiBackup(srcResource, dstResource);
        }
        this.destinationResourceFetcher.close(dstResource);
    }

    private void runSimpleBackup(Neo4jResource srcResource, Neo4jResource dstResource) throws IOException {
        Neo4jBackupTask task = new Neo4jBackupTask(srcResource.getDataSource(), dstResource.getDataSource());
        task.prepare();
        task.run();
        logger.info("Completed backup of [" + srcResource.getName() + "] data source.");
    }

    private void runMultiBackup(Neo4jResource srcResource, Neo4jResource dstResource) throws IOException {
        ArrayList<Neo4jBackupTask> tasks = new ArrayList<Neo4jBackupTask>();
        logger.info("Checking and preparing " + this.xaNames + " data sources.");
        for (String xaName : this.xaNames) {
            XaDataSourceResource srcDataSource = srcResource.getDataSource(xaName);
            if (srcDataSource == null) {
                String message = "XaDataSource not found in backup source: [" + xaName + "]";
                logger.severe(message);
                throw new RuntimeException(message);
            }
            XaDataSourceResource dstDataSource = dstResource.getDataSource(xaName);
            if (dstDataSource == null) {
                String message = "XaDataSource not found in backup destination: [" + xaName + "]";
                logger.severe(message);
                throw new RuntimeException(message);
            }
            Neo4jBackupTask task = new Neo4jBackupTask(srcDataSource, dstDataSource);
            task.prepare();
            tasks.add(task);
        }
        if (tasks.size() == 0) {
            String message = "No data sources to backup were found.";
            logger.severe(message);
            throw new RuntimeException(message);
        }
        for (Neo4jBackupTask task : tasks) {
            task.run();
        }
        logger.info("Completed backup of " + tasks + " data sources.");
    }

    @Override
    public void enableFileLogger() throws SecurityException, IOException {
        this.enableFileLogger(DEFAULT_BACKUP_LOG_LOCATION);
    }

    @Override
    public void enableFileLogger(String filename) throws SecurityException, IOException {
        if (filename == null) {
            throw new IllegalArgumentException("Given filename is null.");
        }
        this.disableFileLogger();
        this.setFileHandler(new FileHandler(filename, true));
    }

    @Override
    public void enableFileLogger(FileHandler handler) {
        if (handler == null) {
            throw new IllegalArgumentException("Given FileHandler is null.");
        }
        this.disableFileLogger();
        this.setFileHandler(handler);
    }

    private void setFileHandler(FileHandler handler) {
        if (fileHandler != null) {
            throw new IllegalStateException("File handler already exists.");
        }
        fileHandler = handler;
        fileHandler.setLevel(consoleHandler.getLevel());
        fileHandler.setFormatter(new SimpleFormatter());
        logger.addHandler(fileHandler);
    }

    @Override
    public void disableFileLogger() {
        if (fileHandler != null) {
            fileHandler.flush();
            fileHandler.close();
            logger.removeHandler(fileHandler);
            fileHandler = null;
        }
    }

    @Override
    public void setLogLevelDebug() {
        logger.setLevel(LOG_LEVEL_DEBUG);
        consoleHandler.setLevel(LOG_LEVEL_DEBUG);
        if (fileHandler != null) {
            fileHandler.setLevel(LOG_LEVEL_DEBUG);
        }
    }

    @Override
    public void setLogLevelNormal() {
        logger.setLevel(LOG_LEVEL_NORMAL);
        consoleHandler.setLevel(LOG_LEVEL_NORMAL);
        if (fileHandler != null) {
            fileHandler.setLevel(LOG_LEVEL_NORMAL);
        }
    }

    @Override
    public void setLogLevelOff() {
        logger.setLevel(LOG_LEVEL_OFF);
    }

    static {
        logger.setUseParentHandlers(false);
        logger.setLevel(LOG_LEVEL_NORMAL);
        consoleHandler.setLevel(LOG_LEVEL_NORMAL);
        logger.addHandler(consoleHandler);
    }

    private static class DestinationDirResourceFetcher
    extends ResourceFetcher {
        private final String destDir;

        DestinationDirResourceFetcher(String destDir) {
            if (destDir == null) {
                throw new IllegalArgumentException("Destination dir is null");
            }
            this.destDir = destDir;
        }

        void close(Neo4jResource resource) {
            resource.close();
        }

        Neo4jResource fetch() {
            return LocalGraphDatabaseResource.getInstance(this.destDir);
        }
    }

    private static class GraphDbResourceFetcher
    extends ResourceFetcher {
        private final EmbeddedGraphDatabase db;

        GraphDbResourceFetcher(EmbeddedGraphDatabase db) {
            if (db == null) {
                throw new IllegalArgumentException("Destination graph database is null");
            }
            this.db = db;
        }

        void close(Neo4jResource resource) {
        }

        Neo4jResource fetch() {
            return new EmbeddedGraphDatabaseResource(this.db);
        }
    }

    private static abstract class ResourceFetcher {
        private ResourceFetcher() {
        }

        abstract Neo4jResource fetch();

        abstract void close(Neo4jResource var1);
    }

    private class Neo4jBackupTask {
        private final XaDataSourceResource src;
        private final XaDataSourceResource dst;
        private long srcVersion = -1L;
        private long dstVersion = -1L;
        private final String resourceName;

        private Neo4jBackupTask(XaDataSourceResource src, XaDataSourceResource dst) {
            this.src = src;
            this.dst = dst;
            this.resourceName = src.getName();
        }

        public void prepare() throws IOException {
            logger.fine("Checking and preparing data source: [" + this.resourceName + "]");
            if (this.src.getCreationTime() != this.dst.getCreationTime() && this.src.getIdentifier() != this.dst.getIdentifier()) {
                String message = "Source[" + this.src.getCreationTime() + "," + this.src.getIdentifier() + "] is not same as destination[" + this.dst.getCreationTime() + "," + this.dst.getIdentifier() + "] for resource [" + this.resourceName + "]";
                logger.severe(message);
                throw new IllegalStateException(message);
            }
            this.srcVersion = this.src.getVersion();
            this.dstVersion = this.dst.getVersion();
            if (this.srcVersion < this.dstVersion) {
                String message = "Source srcVersion[" + this.srcVersion + "] < destination srcVersion[" + this.dstVersion + "] for resource [" + this.resourceName + "]";
                logger.severe(message);
                throw new IllegalStateException(message);
            }
            this.src.rotateLog();
            this.srcVersion = this.src.getVersion();
            if (this.srcVersion < this.dstVersion) {
                String message = "Source srcVersion[" + this.srcVersion + "] < destination srcVersion[" + this.dstVersion + "] after rotate for resource [" + this.resourceName + "]";
                logger.severe(message);
                throw new IllegalStateException(message);
            }
            for (long i = this.dstVersion; i < this.srcVersion; ++i) {
                if (this.src.hasLogicalLog(i)) continue;
                String message = "Missing log entry in backup source: [" + i + "] in resource [" + this.resourceName + "]. Can not perform backup.";
                logger.severe(message);
                throw new IllegalStateException(message);
            }
            this.dst.makeBackupSlave();
        }

        public void run() throws IOException {
            if (this.srcVersion == -1L || this.dstVersion == -1L) {
                String message = "Backup can not start: source and/or destination could not be prepared for backup: [" + this.resourceName + "]";
                logger.severe(message);
                throw new IllegalStateException(message);
            }
            logger.fine("Backing up data source: [" + this.resourceName + "]");
            for (long i = this.dstVersion; i < this.srcVersion; ++i) {
                logger.fine("Applying logical log [" + i + "] on [" + this.resourceName + "]");
                this.dst.applyLog(this.src.getLogicalLog(i));
            }
            logger.fine("Source and destination have been synchronized. Backup of data source complete [" + this.dstVersion + "->" + this.srcVersion + "] on [" + this.resourceName + "].");
        }

        public String toString() {
            return this.resourceName;
        }
    }
}

